﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Runtime.InteropServices;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// ループしているサンプルのバイトアライメントを調整するフィルタです。
    /// </summary>
    internal class LoopedSampleAligner : Filter
    {
        private int outputLoopStartFrame = 0;
        private int outputLoopEndFrame = 0;

        private byte[] spoolBuffer = new byte[0];
        private int spooledLength = 0;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public LoopedSampleAligner()
            : base(new InputConnectionPoint())
        {
            this.LoopStartAlignmentFrames = 0;
            this.MinimumLoopFrames = 0;
            this.FramesAfterLoopEnd = 0;
        }

        /// <summary>
        /// ループ開始位置のフレームアライメントを取得または設定します。
        /// </summary>
        public int LoopStartAlignmentFrames { get; set; }

        /// <summary>
        /// ループフレーム数の最小値を取得または設定します。
        /// </summary>
        public int MinimumLoopFrames { get; set; }

        /// <summary>
        /// ループ終了位置より後ろのフレーム数を取得または設定します。
        /// </summary>
        public int FramesAfterLoopEnd { get; set; }

        /// <summary>
        /// 補正後のループ開始位置を取得または設定します。
        /// </summary>
        public int OutputLoopStartFrame
        {
            get { return this.outputLoopStartFrame; }
        }

        /// <summary>
        /// 補正後のループ終了位置を取得または設定します。
        /// </summary>
        public int OutputLoopEndFrame
        {
            get { return this.outputLoopEndFrame; }
        }

        /// <summary>
        /// 入力接続ポイントを取得します。
        /// </summary>
        private InputConnectionPoint Input
        {
            get { return this.Inputs[0]; }
        }

        /// <summary>
        /// 入力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation InputStreamInformation
        {
            get { return this.Inputs[0].StreamInformation; }
        }

        /// <summary>
        /// 出力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation OutputStreamInformation
        {
            get { return this.Outputs[0].StreamInformation; }
        }

        /// <summary>
        /// 入力フォーマットを取得します。
        /// </summary>
        private WaveFormat InputFormat
        {
            get { return this.Inputs[0].StreamInformation.Format; }
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        private WaveFormat OutputFormat
        {
            get { return this.Outputs[0].StreamInformation.Format; }
        }

        /// <summary>
        /// 入力ループ開始位置（オフセット）を取得します。
        /// </summary>
        private int InputLoopStart
        {
            get { return this.InputFormat.LoopStartFrame * this.InputFormat.FrameLength; }
        }

        /// <summary>
        /// 入力ループ終了位置（オフセット）を取得します。
        /// </summary>
        private int InputLoopEnd
        {
            get { return this.InputFormat.LoopEndFrame * this.InputFormat.FrameLength; }
        }

        /// <summary>
        /// 出力接続ポイントを生成します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected override OutputConnectionPoint[] CreateOutputs()
        {
            int outputLength = this.Calculate();

            if (this.InputFormat.HasLoop)
            {
                int expandLength = Math.Min(
                    outputLength - this.InputFormat.LoopEndFrame * this.InputFormat.FrameLength,
                    (this.InputFormat.LoopEndFrame - this.InputFormat.LoopStartFrame) * this.InputFormat.FrameLength);

                if (expandLength > 0)
                {
                    this.spoolBuffer = new byte[expandLength];
                }
            }

            return new OutputConnectionPoint[]
            {
                new OutputConnectionPoint(
                    new WaveStreamInformation(
                        outputLength,
                        outputLength / this.InputFormat.FrameLength,
                        this.GetOutputFormat() )
                    )
            };
        }

        /// <summary>
        /// パケットを処理します。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        /// <param name="packet">パケットを指定します。</param>
        protected override void Process(InputConnectionPoint input, Packet packet)
        {
            Assertion.Argument.NotNull(packet);

            if (!this.InputFormat.HasLoop)
            {
                this.SendToAll(packet);
                return;
            }

            int current = this.Input.ReceivedLength - packet.Samples.Length;

            // ループ終了位置より後ろのパケットなら破棄します。
            if (current >= this.InputLoopEnd)
            {
                return;
            }

            // ループ開始位置より前のパケットなら、そのまま送信します。
            if (this.Input.ReceivedLength <= this.InputLoopStart)
            {
                this.SendToAll(packet);
                return;
            }

            // ループ範囲内のサンプルをスプールし、
            // ループ終了位置までのサンプルを送信します。
            byte[] spoolSamples = this.ExtractSpoolSamples(packet.Samples, current);
            this.Spool(spoolSamples);

            byte[] sendSamples = this.ExtractValidSamples(packet.Samples, current);
            this.SendToAll(Packet.Create(sendSamples));

            current += sendSamples.Length;

            // 最終フレームを出力したなら、ループ補正用サンプルを送信します。
            if (this.InputLoopEnd <= this.Input.ReceivedLength)
            {
                Ensure.Operation.True(this.InputLoopEnd == current);
                this.SendSpooledSamples(this.OutputStreamInformation.Length - current);
            }
        }

        /// <summary>
        /// 送信対象となる有効サンプルを抽出します。
        /// </summary>
        /// <param name="samples">抽出元のサンプルを指定します。</param>
        /// <param name="current">サンプルの現在位置を指定します。</param>
        /// <returns>有効サンプルを返します。</returns>
        private byte[] ExtractValidSamples(byte[] samples, int current)
        {
            Assertion.Argument.NotNull(samples);
            Assertion.Argument.True(current >= 0);

            int end = Math.Min(this.InputLoopEnd - current, samples.Length);

            Ensure.Operation.True(end > 0);

            byte[] targetSamples = new byte[end];
            Array.Copy(samples, 0, targetSamples, 0, end);

            return targetSamples;
        }

        /// <summary>
        /// スプール対象のサンプルを抽出します。
        /// </summary>
        /// <param name="samples">抽出元のサンプルを指定します。</param>
        /// <param name="current">サンプルの現在位置を指定します。</param>
        /// <returns>スプール対象のサンプルを返します。</returns>
        private byte[] ExtractSpoolSamples(byte[] samples, int current)
        {
            Assertion.Argument.NotNull(samples);
            Assertion.Argument.True(current >= 0);
            Assertion.Argument.True(current < this.InputLoopEnd);

            int start = Math.Max(this.InputLoopStart - current, 0);
            int end = Math.Min(this.InputLoopEnd - current, samples.Length);

            Ensure.Operation.True(start < end);

            byte[] targetSamples = new byte[end - start];
            Array.Copy(samples, start, targetSamples, 0, targetSamples.Length);

            return targetSamples;
        }

        /// <summary>
        /// サンプルをスプールします。
        /// </summary>
        /// <param name="samples">サンプルを指定します。</param>
        private void Spool(byte[] samples)
        {
            Assertion.Argument.NotNull(samples);

            int spoolLength = Math.Min(
                this.spoolBuffer.Length - this.spooledLength,
                samples.Length);

            if (spoolLength == 0)
            {
                return;
            }

            Array.Copy(samples, 0, this.spoolBuffer, this.spooledLength, spoolLength);
            this.spooledLength += spoolLength;
        }

        /// <summary>
        /// スプールしておいたサンプルを指定サイズ分だけ送信します。
        /// </summary>
        /// <param name="expandLength">ループを伸長するサイズを指定します。</param>
        private void SendSpooledSamples(int expandLength)
        {
            if (this.spooledLength == 0)
            {
                return;
            }

            int remainLength = expandLength;

            while (remainLength > 0)
            {
                int sendLength = Math.Min(this.spooledLength, remainLength);
                byte[] remainSamples = new byte[sendLength];
                Array.Copy(this.spoolBuffer, 0, remainSamples, 0, sendLength);

                this.SendToAll(Packet.Create(remainSamples));
                remainLength -= sendLength;
            }
        }

        /// <summary>
        /// 出力の長さとループ位置を計算します。
        /// </summary>
        /// <returns>出力の長さを返します。</returns>
        private int Calculate()
        {
            if (!this.InputFormat.HasLoop)
            {
                return this.InputStreamInformation.Length;
            }

            int loopFrames = this.InputFormat.LoopEndFrame - this.InputFormat.LoopStartFrame;

            // ループ開始位置のアライメント調整
            int loopStartFrameOffset = 0;

            if (this.LoopStartAlignmentFrames > 0)
            {
                int remainFrames = this.InputFormat.LoopStartFrame % this.LoopStartAlignmentFrames;

                if (remainFrames > 0)
                {
                    loopStartFrameOffset = this.LoopStartAlignmentFrames - remainFrames;
                }
            }

            this.outputLoopStartFrame = this.InputFormat.LoopStartFrame + loopStartFrameOffset;
            this.outputLoopEndFrame = this.outputLoopStartFrame + loopFrames;

            if (this.MinimumLoopFrames > 0 &&
                this.MinimumLoopFrames > loopFrames)
            {
                int expandCount = this.MinimumLoopFrames / loopFrames;
                this.outputLoopEndFrame += loopFrames * expandCount;
            }

            return (this.outputLoopEndFrame + this.FramesAfterLoopEnd) * this.InputFormat.FrameLength;
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        /// <returns>出力フォーマットを返します。</returns>
        private WaveFormat GetOutputFormat()
        {
            return new WaveFormat(this.InputFormat)
            {
                OriginalLoopStartFrame = this.InputFormat.OriginalLoopStartFrame,
                OriginalLoopEndFrame = this.InputFormat.OriginalLoopEndFrame,
                LoopStartFrame = this.OutputLoopStartFrame,
                LoopEndFrame = this.OutputLoopEndFrame,
            };
        }
    }
}
