﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Runtime.InteropServices;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// リニアPCMをノーマライズするフィルタです。
    /// </summary>
    internal class LinearPcmNormalizer : Filter
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public LinearPcmNormalizer()
            : base(new InputConnectionPoint())
        {
        }

        /// <summary>
        /// 出力フレーム数を取得または設定します。
        /// </summary>
        public int FrameCount { get; set; }

        /// <summary>
        /// 入力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation InputStreamInformation
        {
            get { return this.Inputs[0].StreamInformation; }
        }

        /// <summary>
        /// 入力フォーマットを取得します。
        /// </summary>
        private WaveFormat InputFormat
        {
            get { return this.InputStreamInformation.Format; }
        }

        /// <summary>
        /// 出力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation OutputStreamInformation
        {
            get { return this.Outputs[0].StreamInformation; }
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        private WaveFormat OutputFormat
        {
            get { return this.OutputStreamInformation.Format; }
        }

        /// <summary>
        /// 出力接続ポイントを生成します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected override OutputConnectionPoint[] CreateOutputs()
        {
            WaveFormat outputFormat = this.GetOutputFormat();
            int streamLength = this.InputStreamInformation.Length;

            if (this.FrameCount > 0 &&
                this.FrameCount != this.InputStreamInformation.Length / this.InputFormat.FrameLength)
            {
                streamLength = this.FrameCount * outputFormat.FrameLength;
            }

            return new OutputConnectionPoint[]
            {
                new OutputConnectionPoint(
                    new WaveStreamInformation(
                        this.GetOutputLength(streamLength),
                        this.GetOutputFrameCount(streamLength),
                        outputFormat
                        )
                    )
            };
        }

        /// <summary>
        /// パケットを処理します。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        /// <param name="packet">パケットを指定します。</param>
        protected override void Process(InputConnectionPoint input, Packet packet)
        {
            Assertion.Argument.NotNull(packet);

            int currentFrame = (input.ReceivedLength - packet.Samples.Length) / this.InputFormat.FrameLength;

            // 出力フレーム数を超えている場合は、パケットを破棄します。
            if (this.FrameCount > 0 && currentFrame >= this.FrameCount)
            {
                return;
            }

            Packet outputPacket = packet;
            int outputLength = Math.Min(
                this.GetOutputPartLength(packet.Samples.Length),
                this.GetRestOutputLength(currentFrame)
                );

            // 入力サイズと出力サイズが異なる場合のみバッファを作成します。
            // 同じ場合は、入力バッファ内で処理します。
            if (packet.Samples.Length != outputLength)
            {
                byte[] outputSamples = new byte[outputLength];
                outputPacket = Packet.Create(outputSamples);
            }

            GCHandle inputHandle = GCHandle.Alloc(packet.Samples, GCHandleType.Pinned);
            GCHandle outputHandle = GCHandle.Alloc(outputPacket.Samples, GCHandleType.Pinned);

            try
            {
                Ensure.Operation.True(
                    Normalize(
                        outputHandle.AddrOfPinnedObject(),
                        inputHandle.AddrOfPinnedObject(),
                        (outputLength / this.OutputFormat.FrameLength) * input.StreamInformation.Format.FrameLength,
                        this.InputFormat.SampleLength,
                        this.InputFormat.IsLittleEndian,
                        this.InputFormat.IsSigned
                        ),
                    "Failed to normalize samples."
                    );

                this.SendToAll(outputPacket);
            }
            finally
            {
                inputHandle.Free();
                outputHandle.Free();
            }
        }

        protected override void ProcessEndOfAllStreams()
        {
            int currentFrame = this.Inputs[0].ReceivedLength / this.InputFormat.FrameLength;
            int restFrameCount = this.GetRestOutputFrameCount(currentFrame);

            if (restFrameCount > 0)
            {
                // 足りないフレーム数だけ無音サンプルを補完します。
                this.SendToAll(Packet.Create(
                    this.CreateOutputSilentSamples(restFrameCount)
                    ));
            }

            base.ProcessEndOfAllStreams();
        }

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern int GetBufferSizeForNormalize(int inputByte, int bytePerSample);

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern bool Normalize(
            IntPtr output, IntPtr input, int inputByte,
            int bytePerSample, bool isLittleEndian, bool isSigned);

        /// <summary>
        /// 出力の長さを取得します。
        /// </summary>
        /// <param name="inputLength">入力の長さを指定します。</param>
        /// <returns>出力の長さを返します。</returns>
        private int GetOutputLength(int inputLength)
        {
            int length = GetBufferSizeForNormalize(
                inputLength,
                this.InputFormat.SampleLength);

            if (this.InputFormat.HasLoop)
            {
                length = Math.Max(length, this.InputFormat.LoopEndFrame * this.InputFormat.FrameLength);
            }

            return length;
        }

        /// <summary>
        /// 出力フレーム数を取得します。
        /// </summary>
        /// <param name="inputLength">入力の長さを指定します。</param>
        /// <returns>出力フレーム数を返します。</returns>
        private int GetOutputFrameCount(int inputLength)
        {
            return this.GetOutputLength(inputLength) / this.InputFormat.FrameLength;
        }

        /// <summary>
        /// 出力の長さを取得します。
        /// </summary>
        /// <param name="inputLength">入力の長さを指定します。</param>
        /// <returns>出力の長さを返します。</returns>
        private int GetOutputPartLength(int inputLength)
        {
            return GetBufferSizeForNormalize(
                inputLength,
                this.InputFormat.SampleLength);
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        /// <returns>出力フォーマットを返します。</returns>
        private WaveFormat GetOutputFormat()
        {
            return new WaveFormat(this.InputFormat)
            {
                BitsPerSample = 16,
                IsLittleEndian = true,
                IsSigned = true,
            };
        }

        /// <summary>
        /// 残り出力フレーム数を取得します。
        /// </summary>
        /// <param name="currentFrame">現在のフレーム位置を指定します。</param>
        /// <returns>残り出力フレーム数を返します。</returns>
        private int GetRestOutputFrameCount(int currentFrame)
        {
            if (this.FrameCount == 0)
            {
                return this.OutputStreamInformation.Length / this.OutputFormat.FrameLength - currentFrame;
            }

            return this.FrameCount - currentFrame;
        }

        /// <summary>
        /// 残り出力ストリーム長を取得します。
        /// </summary>
        /// <param name="currentFrame">現在のフレーム位置を指定します。</param>
        /// <returns>残り出力ストリーム長を返します。</returns>
        private int GetRestOutputLength(int currentFrame)
        {
            return this.GetRestOutputFrameCount(currentFrame) * this.OutputFormat.FrameLength;
        }

        /// <summary>
        /// 出力用の無音サンプルデータを作成します。
        /// </summary>
        /// <param name="frameCount">フレーム数を指定します。</param>
        /// <returns>作成した無音サンプルデータの配列を返します。</returns>
        private byte[] CreateOutputSilentSamples(int frameCount)
        {
            return new byte[frameCount * this.OutputFormat.FrameLength];
        }
    }
}
