﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Runtime.InteropServices;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// リニアPCMにエンコードするフィルタです。
    /// </summary>
    internal class LinearPcmEncoder : Filter
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public LinearPcmEncoder()
            : base(new InputConnectionPoint())
        {
            this.BitsPerSample = 16;
            this.IsLittleEndian = true;
        }

        /// <summary>
        /// サンプル毎のビット数を取得または設定します。
        /// </summary>
        public int BitsPerSample { get; set; }

        /// <summary>
        /// リトルエンディアンで出力するかどうかを取得または設定します。
        /// </summary>
        public bool IsLittleEndian { get; set; }

        /// <summary>
        /// 入力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation InputStreamInformation
        {
            get { return this.Inputs[0].StreamInformation; }
        }

        /// <summary>
        /// 入力フォーマットを取得します。
        /// </summary>
        private WaveFormat InputFormat
        {
            get { return this.InputStreamInformation.Format; }
        }

        /// <summary>
        /// 出力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation OutputStreamInformation
        {
            get { return this.Outputs[0].StreamInformation; }
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        private WaveFormat OutputFormat
        {
            get { return this.OutputStreamInformation.Format; }
        }

        /// <summary>
        /// 出力接続ポイントを生成します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected override OutputConnectionPoint[] CreateOutputs()
        {
            return new OutputConnectionPoint[]
            {
                new OutputConnectionPoint(
                    new WaveStreamInformation(
                        this.GetOutputLength(this.InputStreamInformation.Length),
                        this.InputStreamInformation.FrameCount,
                        this.GetOutputFormat() )
                    )
            };
        }

        /// <summary>
        /// パケットを処理します。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        /// <param name="packet">パケットを指定します。</param>
        protected override void Process(InputConnectionPoint input, Packet packet)
        {
            Assertion.Argument.NotNull(packet);

            Packet outputPacket = packet;
            int outputLength = this.GetOutputLength(packet.Samples.Length);

            if (packet.Samples.Length != outputLength)
            {
                byte[] outputSamples = new byte[outputLength];
                outputPacket = Packet.Create(outputSamples);
            }

            GCHandle inputHandle = GCHandle.Alloc(packet.Samples, GCHandleType.Pinned);
            GCHandle outputHandle = GCHandle.Alloc(outputPacket.Samples, GCHandleType.Pinned);

            try
            {
                int frameCount = packet.Samples.Length / this.InputFormat.FrameLength;

                switch (this.OutputFormat.BitsPerSample)
                {
                    case 8:
                        EncodePcm8(
                            outputHandle.AddrOfPinnedObject(),
                            inputHandle.AddrOfPinnedObject(),
                            frameCount);
                        break;

                    case 16:
                        if (this.IsLittleEndian)
                        {
                            EncodePcm16LE(
                                outputHandle.AddrOfPinnedObject(),
                                inputHandle.AddrOfPinnedObject(),
                                frameCount);
                        }
                        else
                        {
                            EncodePcm16BE(
                                outputHandle.AddrOfPinnedObject(),
                                inputHandle.AddrOfPinnedObject(),
                                frameCount);
                        }
                        break;

                    default:
                        throw new Exception("invalid format.");
                }

                this.SendToAll(outputPacket);
            }
            finally
            {
                inputHandle.Free();
                outputHandle.Free();
            }
        }

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern bool EncodePcm8(
            IntPtr output, IntPtr input, int frameCount);

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern bool EncodePcm16BE(
            IntPtr output, IntPtr input, int frameCount);

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern bool EncodePcm16LE(
            IntPtr output, IntPtr input, int frameCount);

        /// <summary>
        /// 出力の長さを取得します。
        /// </summary>
        /// <param name="inputLength">入力の長さを指定します。</param>
        /// <returns>出力の長さを返します。</returns>
        private int GetOutputLength(int inputLength)
        {
            return inputLength / this.InputFormat.SampleLength * this.BitsPerSample / 8;
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        /// <returns>出力フォーマットを返します。</returns>
        private WaveFormat GetOutputFormat()
        {
            return new WaveFormat(this.InputFormat)
            {
                BitsPerSample = this.BitsPerSample,
                IsLittleEndian = this.BitsPerSample > 8 && this.IsLittleEndian,
            };
        }
    }
}
