﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Collections.Generic;
    using System.Runtime.InteropServices;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// チャンネル毎に分割するフィルタです。
    /// </summary>
    internal class ChannelDivider : Filter
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ChannelDivider()
            : base(new InputConnectionPoint())
        {
        }

        /// <summary>
        /// 入力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation InputStreamInformation
        {
            get { return this.Inputs[0].StreamInformation; }
        }

        /// <summary>
        /// 入力フォーマットを取得します。
        /// </summary>
        private WaveFormat InputFormat
        {
            get { return this.Inputs[0].StreamInformation.Format; }
        }

        /// <summary>
        /// 出力接続ポイントを生成します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected override OutputConnectionPoint[] CreateOutputs()
        {
            List<OutputConnectionPoint> outputs = new List<OutputConnectionPoint>();

            WaveStreamInformation outputStreamInformation =
                new WaveStreamInformation(
                    this.GetOutputLength(this.InputStreamInformation.Length),
                    this.InputStreamInformation.FrameCount,
                    this.GetOutputFormat());

            // チャンネル数だけ出力接続ポイントを生成する
            for (int i = 0; i < this.InputFormat.ChannelCount; i++)
            {
                outputs.Add(new OutputConnectionPoint(outputStreamInformation));
            }

            return outputs.ToArray();
        }

        /// <summary>
        /// パケットを処理します。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        /// <param name="packet">パケットを指定します。</param>
        protected override void Process(InputConnectionPoint input, Packet packet)
        {
            Assertion.Argument.NotNull(packet);

            if (this.InputFormat.ChannelCount == 1)
            {
                this.SendToAll(packet);
                return;
            }

            int outputLength = this.GetOutputLength(packet.Samples.Length);

            GCHandle inputHandle = GCHandle.Alloc(packet.Samples, GCHandleType.Pinned);

            try
            {
                for (int i = 0; i < this.InputFormat.ChannelCount; i++)
                {
                    byte[] outputSamples = new byte[outputLength];
                    Packet outputPacket = Packet.Create(outputSamples);

                    GCHandle outputHandle = GCHandle.Alloc(outputPacket.Samples, GCHandleType.Pinned);

                    try
                    {
                        DevideChannel(
                            outputHandle.AddrOfPinnedObject(),
                            inputHandle.AddrOfPinnedObject(),
                            packet.Samples.Length / this.InputFormat.SampleLength,
                            i,
                            this.InputFormat.ChannelCount);

                        this.Outputs[i].Send(outputPacket);
                    }
                    finally
                    {
                        outputHandle.Free();
                    }
                }
            }
            finally
            {
                inputHandle.Free();
            }
        }

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern int GetBufferSizeForDevideChannel(int sampleCount, int channelMax);

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern bool DevideChannel(
            IntPtr output, IntPtr input, int inputSampleCount,
            int channelIndex, int channelCount);

        /// <summary>
        /// 出力の長さを取得します。
        /// </summary>
        /// <param name="inputLength">入力の長さを指定します。</param>
        /// <returns>出力の長さを返します。</returns>
        private int GetOutputLength(int inputLength)
        {
            if (this.InputFormat.ChannelCount == 1)
            {
                return inputLength;
            }

            return GetBufferSizeForDevideChannel(
                inputLength / this.InputFormat.SampleLength,
                this.InputFormat.ChannelCount);
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        /// <returns>出力フォーマットを返します。</returns>
        private WaveFormat GetOutputFormat()
        {
            return new WaveFormat(this.InputFormat)
            {
                ChannelCount = 1,
            };
        }
    }
}
