﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System.Collections.Generic;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// リニアPCMストリームにエンコードします。
    /// </summary>
    internal class NintendoWareLinearPcmStreamEncoder : NintendoWareStreamEncoder
    {
        private StreamChannelMixer streamChannelMixer = null;
        private SampleGrabber sampleGrabber = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public NintendoWareLinearPcmStreamEncoder()
        {
            this.BitsPerSample = 16;
        }

        /// <summary>
        /// 出力のビット深度を取得または設定します。
        /// </summary>
        public int BitsPerSample { get; set; }

        /// <summary>
        /// リトルエンディアンで出力するかどうかを取得または設定します。
        /// </summary>
        public bool IsLittleEndian { get; set; }

        /// <summary>
        /// フィルタを初期化します。
        /// </summary>
        /// <param name="waveStream">波形ストリームの配列を指定します。</param>
        /// <param name="normalizeFrameCount">正規化の出力フレーム数を指定します。</param>
        /// <returns>開始フィルタの配列を返します。</returns>
        protected override SourceStreamReader[] InitializeFilters(
            WaveStream[] waveStreams,
            int normalizeFrameCount)
        {
            Assertion.Argument.NotNull(waveStreams);
            Assertion.Argument.True(waveStreams.Length > 0);

            this.sampleGrabber = null;

            List<SourceStreamReader> readers = new List<SourceStreamReader>();
            List<OutputConnectionPoint> encoderOutputs = new List<OutputConnectionPoint>();

            // 1. SourceStreamReader
            foreach (WaveStream waveStream in waveStreams)
            {
                readers.Add(
                    new SourceStreamReader()
                    {
                        Source = waveStream.Payload,
                        SourceFormat = waveStream.Format,
                    });
            }

            foreach (SourceStreamReader reader in readers)
            {
                // 2. LinearPcmNormalizer
                LinearPcmNormalizer normalizer = new LinearPcmNormalizer()
                {
                    FrameCount = normalizeFrameCount,
                };
                reader.Outputs[0].ConnectTo(normalizer.Inputs[0]);

                // 3. ChannelDivider
                ChannelDivider channelDivider = new ChannelDivider();
                normalizer.Outputs[0].ConnectTo(channelDivider.Inputs[0]);

                for (int i = 0; i < channelDivider.Outputs.Length; i++)
                {
                    // 4. LoopedSampleAligner
                    LoopedSampleAligner loopedSampleAligner = new LoopedSampleAligner()
                    {
                        LoopStartAlignmentFrames = this.GetLoopStartAlignmentFrames(),
                        MinimumLoopFrames = this.GetMinimumLoopFrames(),
                    };
                    channelDivider.Outputs[i].ConnectTo(loopedSampleAligner.Inputs[0]);

                    // 5. LinearPcmEncoder
                    LinearPcmEncoder encoder = new LinearPcmEncoder()
                    {
                        BitsPerSample = this.BitsPerSample,
                        IsLittleEndian = this.IsLittleEndian,
                    };
                    loopedSampleAligner.Outputs[0].ConnectTo(encoder.Inputs[0]);

                    encoderOutputs.Add(encoder.Outputs[0]);
                }
            }

            // 6. StreamChannelMixer
            StreamChannelMixer channelMixer = new StreamChannelMixer(encoderOutputs.Count);

            for (int i = 0; i < encoderOutputs.Count; i++)
            {
                encoderOutputs[i].ConnectTo(channelMixer.Inputs[i]);
            }

            this.streamChannelMixer = channelMixer;

            // 7. SampleGrabber
            SampleGrabber sampleGrabber = new SampleGrabber();
            channelMixer.Outputs[0].ConnectTo(sampleGrabber.Inputs[0]);

            this.sampleGrabber = sampleGrabber;

            return readers.ToArray();
        }

        /// <summary>
        /// エンコード処理の出力を作成します。
        /// </summary>
        /// <returns>エンコード処理の出力を返します。</returns>
        protected override CodecOutput CreateOutput()
        {
            return new CodecOutput()
            {
                Data = this.sampleGrabber.Samples ?? new byte[0],
                Format = this.sampleGrabber.Inputs[0].StreamInformation.Format,
                FrameCount = this.sampleGrabber.Inputs[0].StreamInformation.FrameCount,
                StreamData = this.streamChannelMixer.WaveStreamData,
            };
        }

        /// <summary>
        /// ループ開始位置のフレームアライメントを取得します。
        /// </summary>
        /// <returns>ループ開始位置のフレームアライメントを返します。</returns>
        private int GetLoopStartAlignmentFrames()
        {
            // 8KB 分
            return 8 * 1024 / (this.BitsPerSample / 8);
        }

        /// <summary>
        /// ループフレーム数の最小値を取得します。
        /// </summary>
        /// <returns>ループフレーム数の最小値を返します。</returns>
        private int GetMinimumLoopFrames()
        {
            // ユーザーが使うであろう一番大きいサンプリングレート 48KHz
            // 1オーディオフレーム 3ms
            // ピッチの最大倍率 8 倍
            // => 48000 * 0.003 * 8 = 1152 サンプル
            return (int)(48000 * 0.003 * 8);
        }
    }
}
