﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using NintendoWare.ToolDevelopmentKit;
    using System;
    using System.Collections.Generic;
    using System.Linq;

    /// <summary>
    /// DSPADPCMストリームにエンコードします。
    /// </summary>
    internal class NintendoWareDspAdpcmStreamEncoder : NintendoWareStreamEncoder
    {
        /// <summary>
        /// DSP ADPCM ループのつなぎを滑らかにするためにエンコーダに余分に渡すサンプル数です。
        /// </summary>
        /// <remarks>
        /// DSP ADPCM 1ブロックあたりのサンプル数(14)で割り切れる数値にします。
        /// </remarks>
        private const int ExtraSamplesForEncoding = 140;

        private List<AdpcmEncoder> adpcmEncoders = new List<AdpcmEncoder>();
        private StreamChannelMixer streamChannelMixer = null;
        private SampleGrabber sampleGrabber = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public NintendoWareDspAdpcmStreamEncoder()
        {
            this.OutputEncoding = Encoding.DspAdpcm;
        }

        /// <summary>
        /// 出力エンコーディングを指定します。
        /// </summary>
        /// <remarks>
        /// "DSPADPCM", "IMAADPCM" を指定できます。
        /// </remarks>
        public string OutputEncoding { get; set; }

        /// <summary>
        /// フィルタを初期化します。
        /// </summary>
        /// <param name="waveStream">波形ストリームの配列を指定します。</param>
        /// <param name="normalizeFrameCount">正規化の出力フレーム数を指定します。</param>
        /// <returns>開始フィルタの配列を返します。</returns>
        protected override SourceStreamReader[] InitializeFilters(
            WaveStream[] waveStreams,
            int normalizeFrameCount)
        {
            Assertion.Argument.NotNull(waveStreams);

            this.sampleGrabber = null;

            List<SourceStreamReader> readers = new List<SourceStreamReader>();
            List<OutputConnectionPoint> encoderOutputs = new List<OutputConnectionPoint>();

            // 1. SourceStreamReader
            foreach (WaveStream waveStream in waveStreams)
            {
                readers.Add(
                    new SourceStreamReader()
                    {
                        Source = waveStream.Payload,
                        SourceFormat = waveStream.Format,
                    });
            }

            foreach (SourceStreamReader reader in readers)
            {
                // 2. LinearPcmNormalizer
                LinearPcmNormalizer normalizer = new LinearPcmNormalizer()
                {
                    FrameCount = normalizeFrameCount,
                };
                reader.Outputs[0].ConnectTo(normalizer.Inputs[0]);

                // 3. ChannelDivider
                ChannelDivider channelDivider = new ChannelDivider();
                normalizer.Outputs[0].ConnectTo(channelDivider.Inputs[0]);

                for (int i = 0; i < channelDivider.Outputs.Length; i++)
                {
                    // 4. LoopedSampleAligner
                    LoopedSampleAligner loopedSampleAligner = new LoopedSampleAligner()
                    {
                        LoopStartAlignmentFrames = this.GetLoopStartAlignmentFrames(),
                        MinimumLoopFrames = this.GetMinimumLoopFrames(),
                        FramesAfterLoopEnd = ExtraSamplesForEncoding,
                    };
                    channelDivider.Outputs[i].ConnectTo(loopedSampleAligner.Inputs[0]);

                    // 5. SampleSpooler
                    // AdpcmEncoder は部分エンコードに対応していないので、
                    // パケットを１つにまとめる必要があります。
                    SampleSpooler sampleSpooler = new SampleSpooler();
                    loopedSampleAligner.Outputs[0].ConnectTo(sampleSpooler.Inputs[0]);

                    // 6. AdpcmEncoder
                    AdpcmEncoder encoder = new AdpcmEncoder()
                    {
                        OutputEncoding = this.OutputEncoding,
                        BlockLength = 8 * 1024, // 8KB
                        ExtraSamplesForSmoothLoop = ExtraSamplesForEncoding,
                    };
                    sampleSpooler.Outputs[0].ConnectTo(encoder.Inputs[0]);

                    encoderOutputs.Add(encoder.Outputs[0]);

                    this.adpcmEncoders.Add(encoder);
                }
            }

            // 7. StreamChannelMixer
            StreamChannelMixer channelMixer = new StreamChannelMixer(encoderOutputs.Count);

            for (int i = 0; i < encoderOutputs.Count; i++)
            {
                encoderOutputs[i].ConnectTo(channelMixer.Inputs[i]);
            }

            this.streamChannelMixer = channelMixer;

            // 8. SampleGrabber
            SampleGrabber sampleGrabber = new SampleGrabber();
            channelMixer.Outputs[0].ConnectTo(sampleGrabber.Inputs[0]);

            this.sampleGrabber = sampleGrabber;

            return readers.ToArray();
        }

        /// <summary>
        /// エンコード処理の出力を作成します。
        /// </summary>
        /// <returns>エンコード処理の出力を返します。</returns>
        protected override CodecOutput CreateOutput()
        {
            return new CodecOutput()
            {
                Data = this.sampleGrabber.Samples ?? new byte[0],
                Format = this.sampleGrabber.Inputs[0].StreamInformation.Format,
                FrameCount = this.sampleGrabber.Inputs[0].StreamInformation.FrameCount,
                StreamData = this.streamChannelMixer.WaveStreamData,
                CodecData = new DspAdpcmStreamData()
                {
                    DspAdpcmData = this.GetDspAdpcmInformations(),
                    DspAdpcmSeekDatas = this.streamChannelMixer.AdpcmSeekInformations,
                    DspAdpcmContexts = this.GetDspAdpcmContexts(),
                },
            };
        }

        /// <summary>
        /// DSPADPCM情報を取得します。
        /// </summary>
        /// <returns>出力ストリーム毎のDSPADPCM情報を返します。</returns>
        private DspAdpcmData[] GetDspAdpcmInformations()
        {
            List<DspAdpcmData> adpcmInformations = new List<DspAdpcmData>();

            foreach (AdpcmEncoder adpcmEncoder in this.adpcmEncoders)
            {
                if (adpcmEncoder.AdpcmInformation == null)
                {
                    adpcmInformations.Add(new DspAdpcmData() { Coef = new Int16[DspAdpcmData.CoefLength] });
                }
                else
                {
                    adpcmInformations.Add((DspAdpcmData)adpcmEncoder.AdpcmInformation);
                }
            }

            return adpcmInformations.ToArray();
        }

        /// <summary>
        /// リージョンDSPADPCMコンテキスト情報を取得します。
        /// </summary>
        /// <returns>出力ストリーム毎のリージョンDSPADPCM情報を返します。</returns>
        private DspAdpcmContext[][] GetDspAdpcmContexts()
        {
            List<DspAdpcmContext[]> adpcmContexts = new List<DspAdpcmContext[]>();

            foreach (AdpcmEncoder adpcmEncoder in this.adpcmEncoders)
            {
                adpcmContexts.Add(adpcmEncoder.RegionAdpcmContexts.Cast<DspAdpcmContext>().ToArray());
            }

            return adpcmContexts.ToArray();
        }

        /// <summary>
        /// ループ開始位置のフレームアライメントを取得します。
        /// </summary>
        /// <returns>ループ開始位置のフレームアライメントを返します。</returns>
        private int GetLoopStartAlignmentFrames()
        {
            // 8KB アライメント
            // DSP ADPCM 1 ブロックあたり 8 バイトで、デコードすると 14 サンプルになる
            return 8 * 1024 * 14 / 8;
        }

        /// <summary>
        /// ループフレーム数の最小値を取得します。
        /// </summary>
        /// <returns>ループフレーム数の最小値を返します。</returns>
        private int GetMinimumLoopFrames()
        {
            // ユーザーが使うであろう一番大きいサンプリングレート 48KHz
            // 1オーディオフレーム 3ms
            // ピッチの最大倍率 8 倍
            // => 48000 * 0.003 * 8 = 1152 サンプル
            return (int)(48000 * 0.003 * 8);
        }
    }
}
