﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Commands;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    /// <summary>
    /// Keys の拡張機能を提供します。
    /// </summary>
    public static class KeysEx
    {
        /// <summary>
        /// Keys[] から テキストを取得します。
        /// </summary>
        /// <param name="keys">Keys[]。</param>
        /// <returns>テキスト。</returns>
        public static string ToShortcutKeyText(this Keys[] keys)
        {
            string keysText = string.Empty;

            foreach (Keys key in keys)
            {

                if (0 < keysText.Length)
                {
                    keysText += ", ";
                }

                keysText += key.ToShortcutKeyText();

            }

            return keysText;
        }

        /// <summary>
        /// Keys から テキストを取得します。
        /// </summary>
        /// <param name="keys">Keys。</param>
        /// <returns>テキスト。</returns>
        public static string ToShortcutKeyText(this Keys key)
        {
            return new KeysConverter().ConvertToInvariantString(key);
        }

        /// <summary>
        /// Keys[] から KeyStroke[] へ変換します。
        /// </summary>
        /// <param name="keys">変換元の Keys[]。</param>
        /// <returns>変換後の KeyStroke[]。</returns>
        public static KeyStroke[] ToKeyStroke(this Keys[] keys)
        {
            List<KeyStroke> keyStrokes = new List<KeyStroke>();

            foreach (Keys key in keys)
            {
                keyStrokes.Add(key.ToKeyStroke());
            }

            return keyStrokes.ToArray();
        }

        /// <summary>
        /// Keys から KeyStroke へ変換します。
        /// </summary>
        /// <param name="key">変換元の Keys。</param>
        /// <returns>変換後の KeyStroke。</returns>
        public static KeyStroke ToKeyStroke(this Keys key)
        {
            return new KeyStroke((int)(key & Keys.KeyCode), (int)(key & Keys.Modifiers));
        }

        /// <summary>
        /// ショートカットキーテキストを解析して Keys[] を作成します。
        /// </summary>
        /// <param name="shortcutKeyText">変換元のショートカットキーテキスト。</param>
        /// <returns>変換後の Keys[]。</returns>
        public static Keys[] ParseKeys(string shortcutKeyText)
        {
            if (null == shortcutKeyText) { throw new ArgumentNullException("shortcutKeyText"); }
            if (0 == shortcutKeyText.Length) { return new Keys[0]; }


            string[] keyStrokeTexts = shortcutKeyText.Split(',');
            if (0 == keyStrokeTexts.Length) { return new Keys[0]; }


            List<Keys> keys = new List<Keys>();

            foreach (string keyStrokeText in keyStrokeTexts)
            {

                Keys key = KeysEx.ParseKey(keyStrokeText.Trim());
                if (Keys.None == key) { return new Keys[0]; }

                keys.Add(key);

            }

            return keys.ToArray();
        }

        /// <summary>
        /// キーストロークテキストを解析して Keys を作成します。
        /// </summary>
        /// <param name="shortcutKeyText">変換元のキーストロークテキスト。</param>
        /// <returns>変換後の Keys。</returns>
        public static Keys ParseKey(string keyStrokeText)
        {
            if (null == keyStrokeText) { throw new ArgumentNullException("keyStrokeText"); }
            return (Keys)new KeysConverter().ConvertFromString(keyStrokeText);
        }
    }
}
