﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using System.Windows.Forms.Layout;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    #region ** Generic

    #region ** コンテナユーティリティ

    public class SortedListUtility<_KeyType, _ValueType>
    {
        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>コレクションを反復処理する列挙子</returns>
        public static IEnumerator<_ValueType> FromSortedList(SortedList<_KeyType, _ValueType> list)
        {
            Debug.Assert(null != list);

            foreach (KeyValuePair<_KeyType, _ValueType> item in list)
            {
                yield return item.Value;
            }
        }
    }

    #endregion

    #region ** 同期ユーティリティ

    /// <summary>
    /// 値の同期用インターフェイス
    /// </summary>
    public interface IValueSynchronize
    {
        /// <summary>
        /// 同期元の値を取得します。
        /// </summary>
        decimal Value { get; }

        /// <summary>
        /// ２オブジェクト間にて値を同期します。
        /// </summary>
        /// <param name="sender">同期元オブジェクト</param>
        /// <param name="e">値変更イベントパラメータ</param>
        void OnSynchronizedControlValueChanged(object sender, ValueChangedEventArgs e);
    }

    /// <summary>
    /// 値の同期を可能するためのインターフェイス
    /// </summary>
    public interface IValueSynchronizable
    {
        /// <summary>
        /// Owner の値を取得または設定します。
        /// </summary>
        decimal Value { get; set; }
    }

    /// <summary>
    /// 値の同期コンポーネントクラス
    /// </summary>
    public class ValueSynchronizable : IValueSynchronize
    {
        private IValueSynchronizable _owner;
        private bool _locked = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="owner"></param>
        public ValueSynchronizable(IValueSynchronizable owner)
        {
            _owner = owner;
        }

        #region ** プロパティ

        private bool Locked
        {
            get { return _locked; }
            set { _locked = value; }
        }

        #endregion

        #region ** イベント

        public event ValueChangedEventHandler ValueChanged;
        public event ValueChangedEventHandler Synchronized;

        #endregion

        #region ** メソッド

        /// <summary>
        /// ValueChanged イベントを発行します。
        /// </summary>
        public void DispatchValueChangedEvent()
        {
            if (!Locked && null != Synchronized)
            {
                Synchronized(this, new ValueChangedEventArgs(true, Locked));
            }

            if (null != ValueChanged)
            {
                ValueChanged(_owner, new ValueChangedEventArgs(true, false));
            }
        }

        #endregion

        #region ** オペレータ

        public static ValueSynchronizable operator +(ValueSynchronizable target1, ValueSynchronizable target2)
        {
            target1.Synchronized += target2.OnSynchronizedControlValueChanged;
            target2.Synchronized += target1.OnSynchronizedControlValueChanged;
            return target1;
        }

        public static ValueSynchronizable operator -(ValueSynchronizable target1, ValueSynchronizable target2)
        {
            target1.Synchronized -= target2.OnSynchronizedControlValueChanged;
            target2.Synchronized -= target1.OnSynchronizedControlValueChanged;
            return target1;
        }

        #endregion

        #region ** IValueSynchronize の実装

        /// <summary>
        /// Owner の値を取得または設定します。
        /// </summary>
        public decimal Value
        {
            get
            {
                Debug.Assert(null != _owner);
                return _owner.Value;
            }
            set
            {
                Debug.Assert(null != _owner);
                _owner.Value = value;
            }
        }

        public void OnSynchronizedControlValueChanged(object sender, ValueChangedEventArgs e)
        {
            Debug.Assert(null != _owner);

            IValueSynchronize sync = sender as IValueSynchronize;
            Debug.Assert(null != sync);

            Locked = true;
            Value = sync.Value;
            Locked = false;
        }

        #endregion
    }

    #endregion

    #endregion
}
