﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Diagnostics;
    using System.Drawing;
    using System.Drawing.Drawing2D;
    using System.Drawing.Imaging;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TreeCtrl : UserControl, ITreeCtrlContainer
    {
        private ITreeItemsSource _ItemsSource = null;
        private TreeItemInfo[] _Infos = null;

        private TreeImageDictionary _Images = null;
        private InplaceEditorDictionary _InplaceEditors = null;
        private TreePartDrawerDictionary _PartDrawers = null;
        private TreeSubItemDictionary _SubItems = null;

        private TreeItemSelectedDictionary _ItemSelecteds = null;
        private List<TreeCtrlCore> _CoreControls = new List<TreeCtrlCore>();

        private ContextMenuStrip _ContextMenuStrip = null;

        public event EventHandler SelectChanged;
        public event EventHandler EditBegan;
        public event EventHandler EditEnded;

        public event FileDroppedEventHandler QueryFileDropped;
        public event FileDroppedEventHandler FileDropped;
        public delegate void FileDroppedEventHandler(object sender, FileDroppedEventArgs e);

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeCtrl()
        {
            IndentLength = 16;

            Drawer = new TreeDrawer();
            Surveyer = new TreeSurveyer();

            ItemDrawer = new TreeItemDrawer();

            _Images = new TreeImageDictionary();

            _InplaceEditors = new InplaceEditorDictionary();
            _InplaceEditors.Add("Caption", new TextInplaceEditor());

            _PartDrawers = new TreePartDrawerDictionary();
            _PartDrawers.Add(new TreePartIconDrawer());
            _PartDrawers.Add(new TreePartCaptionDrawer());

            _SubItems = new TreeSubItemDictionary();
            _SubItems.Add(new TreeSubItem());

            _ItemSelecteds = new TreeItemSelectedDictionary();

            FollowSelectedItemToCaret = true;
            MultiSelect = true;
            FullRowSelect = true;
            ClickedOutsideUnselectItem = false;
            AllowDragItem = true;

            //
            TreeCtrlCore coreCtrl = null;

            coreCtrl = AddCoreCtrl(this);
            coreCtrl.Focus();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override ContextMenuStrip ContextMenuStrip
        {
            get { return null; }
            set { }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ContextMenuStrip AttachedContextMenuStrip
        {
            get { return _ContextMenuStrip; }
            set { _ContextMenuStrip = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ITreeItemsSource ItemsSource
        {
            get { return _ItemsSource; }
            set
            {
                if (_ItemsSource != null)
                {
                    _ItemsSource.Updated -= OnItemsSourceUpdated;
                }

                _ItemsSource = value;

                if (_ItemsSource != null)
                {
                    _ItemsSource.Updated += OnItemsSourceUpdated;
                }

                UpdateInfos();
            }
        }

        ///--------------------------------
        /// <summary>
        /// 読み取り専用なのか調べる
        /// </summary>
        public bool ReadOnly { get; set; }

        ///--------------------------------
        /// <summary>
        /// アイテムのドラッグの有効/無効
        /// </summary>
        public bool AllowDragItem { get; set; }

        ///--------------------------------
        /// <summary>
        /// 範囲外をクリックした時に選択を解除するか
        /// </summary>
        public bool ClickedOutsideUnselectItem { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool MultiSelect { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool FullRowSelect { get; set; }

        ///--------------------------------
        /// <summary>
        /// 行のダブルクリックでもキャプションのダブルクリックとするのか？
        /// </summary>
        public bool FullRowDoubleClickedCaption { get; set; }

        ///--------------------------------
        /// <summary>
        /// カレットに選択がついていくのか？
        /// </summary>
        public bool FollowSelectedItemToCaret { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int IndentLength { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeImageDictionary Images
        {
            get { return _Images; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ITreeSurveyer Surveyer { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ITreeDrawer Drawer { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ITreeItemDrawer ItemDrawer { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public InplaceEditorDictionary InplaceEditors
        {
            get { return _InplaceEditors; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreePartDrawerDictionary PartDrawers
        {
            get { return _PartDrawers; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeSubItemDictionary SubItems
        {
            get { return _SubItems; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeItemSelectedDictionary GetItemSelecteds()
        {
            return _ItemSelecteds;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeItemSelectedDictionary GetItemSelecteds(TreeCtrlCore ctrl)
        {
            return _ItemSelecteds;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ITreeItemInfo[] ItemInfos
        {
            get { return _Infos; }
        }

        /// <summary>
        /// 編集中なのか調べる
        /// </summary>
        public bool IsEditing
        {
            get
            {
                foreach (TreeCtrlCore coreCtrl in _CoreControls)
                {
                    if (coreCtrl.IsEditing == true)
                    {
                        return true;
                    }
                }
                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool BeginEditBySelectedItem()
        {
            TreeCtrlCore ctrl = null;

            if ((ctrl = FocusedCoreCtrl) == null)
            {
                return false;
            }

            return ctrl.BeginEditBySelectedItem();
        }

        public bool BeginEdit(ITreeItem item)
        {
            var ctrl = FocusedCoreCtrl;
            if (ctrl == null)
            {
                return false;
            }

            return ctrl.BeginEdit(item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public new void Invalidate()
        {
            DemandInvalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void DemandInvalidate()
        {
            foreach (TreeCtrlCore coreCtrl in _CoreControls)
            {
                coreCtrl.Invalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SplitHorizontal()
        {
            Split(Orientation.Horizontal);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SplitVertical()
        {
            Split(Orientation.Vertical);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected TreeCtrlCore FocusedCoreCtrl
        {
            get
            {
                foreach (TreeCtrlCore ctrl in _CoreControls)
                {
                    if (ctrl.Focused != false)
                    {
                        return ctrl;
                    }
                }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void OnSelectChanged()
        {
            if (SelectChanged != null)
            {
                SelectChanged(this, new EventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        /// 編集が開始された時に呼ばれる
        /// </summary>
        public void OnEditBegan()
        {
            if (EditBegan != null)
            {
                EditBegan(this, new EventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        /// 編集が終了した時に呼ばれる
        /// </summary>
        public void OnEditEnded()
        {
            if (EditEnded != null)
            {
                EditEnded(this, new EventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        /// 編集を終了させる。
        /// </summary>
        public void EndEdit()
        {
            ActiveCoreCtrl.EndEdit();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual TreeDoubleClickedAction OnDoubleClicked(ITreeItem item, string name)
        {
            return TreeDoubleClickedAction.None;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool CanDropFiles(string[] filePaths, ITreeItem item)
        {
            FileDroppedEventArgs args = null;

            if (FileDropped != null)
            {
                args = new FileDroppedEventArgs(filePaths, item);
                QueryFileDropped(this, args);

                if (args.Cancel == false)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DropFiles(string[] filePaths, ITreeItem item)
        {
            FileDroppedEventArgs args = null;

            if (FileDropped != null)
            {
                args = new FileDroppedEventArgs(filePaths, item);
                FileDropped(this, args);

                if (args.Cancel == false)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public void SetCaret(ITreeItem item)
        {
            TreeCtrlCore coreCtrl = ActiveCoreCtrl;

            if (coreCtrl != null)
            {
                coreCtrl.MoveCaret(item);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateInfos()
        {
            //Debug.WriteLine( "-------- Update item info");

            if (ItemsSource == null)
            {
                _Infos = null;
                return;
            }

            _Infos = InfoCreator.Create(ItemsSource.Items.ToArray());

            foreach (TreeCtrlCore coreCtrl in _CoreControls)
            {
                coreCtrl.UpdateScrollBar();
                coreCtrl.Invalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentTreeItem[] SelectedItems
        {
            get
            {
                TreeItemSelectedDictionary selectedItems = GetItemSelecteds(null);
                return selectedItems.GetAll().Cast<ComponentTreeItem>().ToArray();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool ITreeCtrlContainer.CanMoveSelectedItemToChild(ITreeItem targetItem, ITreeItem[] items)
        {
            return CanMoveSelectedItemToChild(targetItem, items);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool ITreeCtrlContainer.CanMoveSelectedItemToPreviousSibling(
                                                                     ITreeItem targetItem,
                                                                     ITreeItem[] items
                                                                     )
        {
            return CanMoveSelectedItemToPreviousSibling(targetItem, items);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool ITreeCtrlContainer.CanMoveSelectedItemToNextSibling(
                                                                 ITreeItem targetItem,
                                                                 ITreeItem[] items
                                                                 )
        {
            return CanMoveSelectedItemToNextSibling(targetItem, items);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool ITreeCtrlContainer.CanExpanded(ITreeItem targetItem)
        {
            return CanExpanded(targetItem);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void ITreeCtrlContainer.MovedItems(ITreeItem[] items)
        {
            MovedItemsInternal(items);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void MovedItemsInternal(ITreeItem[] items)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual bool CanMoveSelectedItemToChild(ITreeItem targetItem, ITreeItem[] items)
        {
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual bool CanMoveSelectedItemToPreviousSibling(
                                                                    ITreeItem targetItem,
                                                                    ITreeItem[] items
                                                                    )
        {
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual bool CanMoveSelectedItemToNextSibling(
                                                                ITreeItem targetItem,
                                                                ITreeItem[] items
                                                                )
        {
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual bool CanExpanded(ITreeItem targetItem)
        {
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        protected void UpdateSelectedItems()
        {
            if (_Infos == null)
            {
                _ItemSelecteds.Clear();
                return;
            }

            ITreeItem[] items = _ItemSelecteds.GetAll();
            foreach (ITreeItem item in items)
            {
                if (ContainsInfo(item) == false)
                {
                    _ItemSelecteds.Remove(item);
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool ContainsInfo(ITreeItem item)
        {
            if (_Infos != null)
            {
                foreach (TreeItemInfo info in _Infos)
                {
                    if (info.Item == item)
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void Split(Orientation orientation)
        {
            SplitContainer container = null;
            Control parent = null;
            TreeCtrlCore coreCtrl = null;
            TreeCtrlCore newCoreCtrl = null;

            if ((coreCtrl = FocusedCoreCtrl) == null)
            {
                return;
            }

            parent = coreCtrl.Parent;

            container = new SplitContainer();
            container.Orientation = orientation;
            container.Parent = parent;
            container.Dock = DockStyle.Fill;
            container.Show();

            coreCtrl.Parent = container.Panel1;
            newCoreCtrl = AddCoreCtrl(container.Panel2);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected TreeCtrlCore ActiveCoreCtrl
        {
            get
            {
                TreeCtrlCore coreCtrl = FocusedCoreCtrl;
                return coreCtrl != null ? coreCtrl : _CoreControls[0];
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private TreeCtrlCore AddCoreCtrl(Control parent)
        {
            TreeCtrlCore coreCtrl = null;

            coreCtrl = new TreeCtrlCore();
            coreCtrl.Parent = parent;
            coreCtrl.OwnerCtrl = this;
            coreCtrl.Dock = DockStyle.Fill;
            coreCtrl.Show();

            _CoreControls.Add(coreCtrl);

            return coreCtrl;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnItemsSourceUpdated(object sender, EventArgs e)
        {
            Debug.WriteLine("-------- ItemsSource updated");
            UpdateInfos();
            UpdateSelectedItems();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private class InfoCreator
        {
            private static List<TreeItemInfo> _List = new List<TreeItemInfo>();
            private static int _Index = 0;
            private static int _Y = 0;

            ///
            public static TreeItemInfo[] Create(ITreeItem[] items)
            {
                _List.Clear();
                _Index = 0;
                _Y = 0;

                foreach (ITreeItem item in items)
                {
                    Create(item);
                }
                return _List.ToArray();
            }

            ///
            private static void Create(ITreeItem item)
            {
                if (item.Visible == false)
                {
                    return;
                }

                TreeItemInfo info = new TreeItemInfo();

                info.Item = item;
                info.Index = _Index;
                info.Y = _Y;

                _List.Add(info);

                _Index++;
                _Y += item.Length;

                if (item.Expander != false && item.Expanded != false)
                {
                    foreach (ITreeItem childItem in item.Children)
                    {
                        Create(childItem);
                    }
                }
            }
        }
    }

    ///------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface ITreeCtrlContainer
    {
        bool CanMoveSelectedItemToChild(ITreeItem targetItem, ITreeItem[] items);
        bool CanMoveSelectedItemToPreviousSibling(ITreeItem targetItem, ITreeItem[] items);
        bool CanMoveSelectedItemToNextSibling(ITreeItem targetItem, ITreeItem[] items);

        bool CanExpanded(ITreeItem targetItem);

        void MovedItems(ITreeItem[] items);
    }

    ///------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TreeCollisionResult
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeCollisionResult() : this(null, "None")
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeCollisionResult(ITreeItemInfo info, string name)
        {
            _Info = info;
            _Name = name;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeCollisionResult(ITreeItemInfo info, string name, bool selected)
        {
            _Info = info;
            _Name = name;
            _Selected = selected;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeCollisionResult(ITreeItemInfo info, string name, TreeItemInPosition inPosition)
        {
            _Info = info;
            _Name = name;
            _InPosition = inPosition;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ITreeItemInfo _Info = null;
        private string _Name = String.Empty;
        private TreeItemInPosition _InPosition = TreeItemInPosition.None;
        private bool _Selected = false;

        public ITreeItemInfo Info { get { return _Info; } }
        public string Name { get { return _Name; } }
        public TreeItemInPosition InPosition { get { return _InPosition; } }
        public bool Selected { get { return _Selected; } }

        public ITreeItem Item
        {
            get { return _Info != null ? _Info.Item : null; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// カレットが移動した時のイベント引数
    /// </summary>
    public class TreeCaretMovedEventArgs
    {
        private ITreeItem item = null;
        private bool moved = false;
        private bool followSelectedItemToCaret = true;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TreeCaretMovedEventArgs(ITreeItem item, bool moved, bool followSelectedItemToCaret)
        {
            this.item = item;
            this.moved = moved;
            this.followSelectedItemToCaret = followSelectedItemToCaret;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TreeCaretMovedEventArgs(ITreeItem item, bool moved)
        {
            this.item = item;
            this.moved = moved;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TreeCaretMovedEventArgs(ITreeItem item) : this(item, true)
        {
        }

        /// <summary>
        ///
        /// </summary>
        public ITreeItem Item
        {
            get
            {
                return this.item;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool Moved
        {
            get
            {
                return this.moved;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool FollowSelectedItemToCaret
        {
            get
            {
                return this.followSelectedItemToCaret;
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TreeImageDictionary : Dictionary<string, Image>
    {
        public void Add(Bitmap bitmap)
        {
            throw new NotImplementedException();
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TreePartDrawerDictionary : Dictionary<string, ITreePartDrawer>
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Add(ITreePartDrawer partDrawer)
        {
            Add(partDrawer.Name, partDrawer);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TreeSubItemDictionary : Dictionary<string, ITreeSubItem>
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Add(ITreeSubItem subItem)
        {
            Add(subItem.Name, subItem);
        }
    }

    /// <summary>
    /// アイテムの選択状態を記録するクラスです。
    /// </summary>
    public class TreeItemSelectedDictionary : HashSet<ITreeItem>
    {
        /// <summary>
        ///
        /// </summary>
        public int SelectedItemCount
        {
            get
            {
                return this.Count();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public ITreeItem[] GetAll()
        {
            return this.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsSelected(ITreeItem item)
        {
            return Contains(item);
        }

        /// <summary>
        ///
        /// </summary>
        public void Selected(ITreeItem item, bool b)
        {
            if (b != false)
            {
                if (Contains(item) == false)
                {
                    Add(item);
                }
            }
            else
            {
                if (Contains(item) != false)
                {
                    Remove(item);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void ToggleSelected(ITreeItem item)
        {
            if (Contains(item) == false)
            {
                Add(item);
            }
            else
            {
                Remove(item);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public enum TreeItemInPosition
    {
        None,
        Upper,
        Center,
        Lower,
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public enum TreeDoubleClickedAction
    {
        None,
        ToggleExpand,
        Edit,
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface ITreeItemCollection : INotifyCollectionChanged, IList<ITreeItem>
    {
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface ITreeItemInfo
    {
        ITreeItem Item { get; set; }
        int Y { get; }
    }

    ///-----------------------------------------------------------------------------
    ///
    public class TreeItemInfo : ITreeItemInfo
    {
        public ITreeItem Item { get; set; }
        public int Index { get; set; }
        public int Y { get; set; }
    }
}
