﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using NintendoWare.SoundFoundation.Windows.Forms.Windowless;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public class NTabControlCore : NWTabControl
    {
        #region ** パラメータ

        private bool _minimizable = false;

        #endregion

        public NTabControlCore(NWControlHost host) : base(host) { }

        public NTabControlCore(NWControlHost host, Core model, TabLayoutData layoutData)
            : base(host, string.Empty, model, layoutData) { }

        #region ** プロパティ

        public new NTabPanel TabPanel
        {
            get { return base.TabPanel as NTabPanel; }
        }

        public bool Minimizable
        {
            get { return _minimizable; }
            set
            {
                if (value == _minimizable) { return; }

                _minimizable = value;
                PerformLayout();
            }
        }

        protected override INWControlTemplate DefaultTemplate
        {
            get { return new NTabControlCoreTemplateTop(this); }
        }

        #endregion

        #region ** コントロール

        /// <summary>
        /// ドロップダウンボタン
        /// </summary>
        public new class DropDownButton : NWTabControl.DropDownButton
        {
            public DropDownButton(NWControlHost host) : base(host) { }

            #region ** プロパティ

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new ImageButtonTemplate(this); }
            }

            #endregion
        }

        /// <summary>
        /// 最小化ボタン
        /// </summary>
        public class MinimizeButton : NWNCButton
        {
            public const string ControlName = "MinimizeButton";

            public MinimizeButton(NWControlHost host) : base(host, ControlName) { }

            #region ** プロパティ

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new ImageButtonTemplate(this); }
            }

            public override bool Visible
            {
                get
                {
                    if (NTabControlStyle.MinimizedStyleBegin <= (Owner as NTabControl).TabControlStyle &&
                        (Owner as NTabControl).TabControlStyle <= NTabControlStyle.MinimizedStyleEnd)
                    {
                        return false;
                    }

                    return base.Visible;
                }
            }

            #endregion
        }

        /// <summary>
        /// 元に戻すボタン
        /// </summary>
        public class RestoreButton : NWNCButton
        {
            public const string ControlName = "RestoreButton";

            public RestoreButton(NWControlHost host) : base(host, ControlName) { }

            #region ** プロパティ

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new ImageButtonTemplate(this); }
            }

            public override bool Visible
            {
                get
                {
                    if (NTabControlStyle.MinimizedStyleBegin <= (Owner as NTabControl).TabControlStyle &&
                        (Owner as NTabControl).TabControlStyle <= NTabControlStyle.MinimizedStyleEnd)
                    {
                        return base.Visible;
                    }

                    return false;
                }
            }

            #endregion
        }

        public class NTabPanel : NWTabPanel
        {
            public NTabPanel(NWControlHost host) : base(host) { }

            #region ** プロパティ

            public MinimizeButton MinimizeButton
            {
                get { return NcControls[NTabControlCore.MinimizeButton.ControlName] as MinimizeButton; }
            }

            public RestoreButton RestoreButton
            {
                get { return NcControls[NTabControlCore.RestoreButton.ControlName] as RestoreButton; }
            }

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new NTabPanelTemplateTop(this); }
            }

            #endregion

            #region ** イベントハンドラ

            protected override void OnInitialized(EventArgs e)
            {
                base.OnInitialized(e);
                UpdateButtonImages();
            }

            protected override void OnUpdateResource(NWControl control, string key)
            {
                base.OnUpdateResource(control, key);
                UpdateButtonImages();
            }

            #endregion

            #region ** メソッド

            private void UpdateButtonImages()
            {
                // ボタンイメージを設定する
                DropDownButton.Image = GetResource(NWTabControl.DropDownButton.ControlName) as Image;
                MinimizeButton.Image = GetResource(MinimizeButton.ControlName) as Image;
                RestoreButton.Image = GetResource(RestoreButton.ControlName) as Image;
            }

            #endregion
        }

        #endregion

        #region ** テンプレート

        public class NTabControlHostTemplate : NWTabControlHostTemplate
        {
            public NTabControlHostTemplate(NWControlHost host) : base(host) { }

            public static new INWComponentTemplate CreateInstance(NWControlHost host)
            {
                return new NTabControlHostTemplate(host);
            }

            #region ** メソッド

            protected override NWControlRoot CreateRootControl(NWControlHost host)
            {
                return new NTabControlCore(host);
            }

            #endregion
        }

        public class ImageButtonTemplate : NWControlTemplate
        {
            public ImageButtonTemplate(NWControl control)
                : base(control, new NWImageButtonDrawer())
            {
                Debug.Assert(control is NWButton || control is NWNCButton, "unexpected error.");
            }
        }

        #region ** Horizontal

        public class NTabPanelTemplateHorizontal : NWTabPanelTemplateHorizontal
        {
            #region ** 固定値

            private const int MinimizeButtonSize = 14;

            #endregion

            protected NTabPanelTemplateHorizontal(NWControl control, INWControlDrawer drawer)
                : base(control, drawer) { }

            #region ** プロパティ

            protected new NTabPanel Control
            {
                get { return base.Control as NTabPanel; }
            }

            #endregion

            #region ** メソッド

            #region ** 初期化

            public override void InitializeComponent()
            {
                base.InitializeComponent();

                // DropDownButton を置き換える
                Control.NcControls.Remove(Control.NcControls[NWTabControl.DropDownButton.ControlName]);

                Control.NcControls.Add(new DropDownButton(Host));
                Control.NcControls.Add(new MinimizeButton(Host));
                Control.NcControls.Add(new RestoreButton(Host));
            }

            #endregion

            #region ** レイアウト

            public override void PerformNcLayout(Rectangle newBounds, ref Rectangle clientRect)
            {
                if ((Control.Root as NTabControlCore).Minimizable)
                {
                    clientRect.Width -= ControlSpacing + MinimizeButtonSize + ControlSpacing;
                }

                int scrollButtonWidth = 0;

                if (1 < Control.Controls.Count)
                {

                    clientRect.Width -= ControlSpacing + DropDownButtonSize + ControlSpacing;

                    NWTab lastTab = Control.Controls[Control.Controls.Count - 1] as NWTab;
                    Debug.Assert(null != lastTab, "unexpected error.");

                    if (clientRect.Width < lastTab.Right && clientRect.Width > 0)
                    {

                        scrollButtonWidth = Math.Min(ScrollButtonWidth, clientRect.Width / 2);

                        clientRect.X += scrollButtonWidth + ControlSpacing;
                        clientRect.Width -= (scrollButtonWidth + ControlSpacing) * 2;

                    }

                }

                PerformLayoutScrollButtons(clientRect, scrollButtonWidth);
                PerformLayoutMinimizeButton(newBounds);
                PerformLayoutDropDownButton(newBounds);
            }

            private void PerformLayoutScrollButtons(Rectangle newClientRect, int scrollButtonWidth)
            {
                // 表示する必要がない場合は Empty
                if (0 == newClientRect.Left || 0 >= scrollButtonWidth)
                {
                    Control.ScrollUpButton.Bounds = Rectangle.Empty;
                    Control.ScrollDownButton.Bounds = Rectangle.Empty;
                    return;
                }

                // ScrollUp ボタンを再配置する
                Rectangle scrollUpButtonRect = Rectangle.Empty;
                scrollUpButtonRect.X = 0;
                scrollUpButtonRect.Y = Padding.Top;
                scrollUpButtonRect.Width = scrollButtonWidth;
                scrollUpButtonRect.Height = newClientRect.Height - Padding.Vertical;

                Control.ScrollUpButton.Bounds = scrollUpButtonRect;

                // ScrollDown ボタンを再配置する
                Rectangle scrollDownButtonRect = Rectangle.Empty;
                scrollDownButtonRect.X = Math.Max(newClientRect.Right + ControlSpacing,
                                                        scrollUpButtonRect.Right);
                scrollDownButtonRect.Y = Padding.Top;
                scrollDownButtonRect.Width = scrollButtonWidth;
                scrollDownButtonRect.Height = newClientRect.Height - Padding.Vertical;

                Control.ScrollDownButton.Bounds = scrollDownButtonRect;
            }

            private void PerformLayoutMinimizeButton(Rectangle newBounds)
            {
                if (!(Control.Root as NTabControlCore).Minimizable)
                {
                    Control.MinimizeButton.Bounds = Rectangle.Empty;
                    Control.RestoreButton.Bounds = Rectangle.Empty;
                    return;
                }

                Rectangle bounds = Rectangle.Empty;
                bounds.X = newBounds.Width - MinimizeButtonSize - ControlSpacing;
                bounds.Y = (newBounds.Height - MinimizeButtonSize) / 2;
                bounds.Width = MinimizeButtonSize;
                bounds.Height = MinimizeButtonSize;

                Control.MinimizeButton.Bounds = bounds;
                Control.RestoreButton.Bounds = bounds;
            }

            private void PerformLayoutDropDownButton(Rectangle newBounds)
            {
                // 表示する必要がない場合は Empty
                if (!RootControl.ShowDropDownButton || 1 >= Control.Controls.Count)
                {
                    Control.DropDownButton.Bounds = Rectangle.Empty;
                    return;
                }

                Rectangle bounds = Rectangle.Empty;
                bounds.X = newBounds.Width - DropDownButtonSize - ControlSpacing;
                bounds.Y = (newBounds.Height - DropDownButtonSize) / 2;
                bounds.Width = DropDownButtonSize;
                bounds.Height = DropDownButtonSize;

                if ((Control.Root as NTabControlCore).Minimizable)
                {
                    bounds.X -= MinimizeButtonSize + ControlSpacing;
                }

                Control.DropDownButton.Bounds = bounds;
            }

            #endregion

            #endregion
        }

        #region ** Top

        public class NTabControlCoreTemplateTop : NWTabControlTemplateTop
        {
            public NTabControlCoreTemplateTop(NWControl control) : base(control) { }

            public static new INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabControlCoreTemplateTop(control);
            }

            #region ** メソッド

            public override void InitializeComponent()
            {
                Control.Controls.Add(new NTabPanel(Host));
                Control.Controls.Add(new NWTabPagePanel(Host));
            }

            #endregion
        }

        public class NTabPanelTemplateTop : NTabPanelTemplateHorizontal
        {
            public NTabPanelTemplateTop(NWControl control) : base(control, new NWTabPanelDrawerTop()) { }

            protected NTabPanelTemplateTop(NWControl control, INWControlDrawer drawer) : base(control, drawer) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabPanelTemplateTop(control);
            }

            #region ** メソッド

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                Rectangle workRect = Control.AbsoluteBounds;
                workRect.Y = workRect.Bottom - Padding.Bottom;
                workRect.Height = Padding.Bottom;

                NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                drawInfo.BorderRect = workRect;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        #endregion

        #region ** Bottom

        public class NTabControlCoreTemplateBottom : NWTabControlTemplateBottom
        {
            public NTabControlCoreTemplateBottom(NWControl control) : base(control) { }

            public static new INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabControlCoreTemplateBottom(control);
            }

            #region ** メソッド

            public override void InitializeComponent()
            {
                Control.Controls.Add(new NTabPanel(Host));
                Control.Controls.Add(new NWTabPagePanel(Host));
            }

            #endregion
        }

        public class NTabPanelTemplateBottom : NTabPanelTemplateHorizontal
        {
            public NTabPanelTemplateBottom(NWControl control) : base(control, new NWTabPanelDrawerBottom()) { }

            protected NTabPanelTemplateBottom(NWControl control, INWControlDrawer drawer) : base(control, drawer) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabPanelTemplateBottom(control);
            }

            #region ** メソッド

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                Rectangle workRect = Control.AbsoluteBounds;
                workRect.Height = Padding.Top;

                NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                drawInfo.BorderRect = workRect;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        #endregion

        #region ** Minimized

        public class NTabControlCoreTemplateHorizontalMinimized : NWTabControlTemplateHorizontalMinimized
        {
            public NTabControlCoreTemplateHorizontalMinimized(NWControl control) : base(control) { }

            public static new INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabControlCoreTemplateHorizontalMinimized(control);
            }

            #region ** メソッド

            public override void InitializeComponent()
            {
                Control.Controls.Add(new NTabPanel(Host));
                Control.Controls.Add(new NWTabPagePanel(Host));
            }

            #endregion
        }

        public class NTabPanelTemplateHorizontalMinimized : NTabPanelTemplateHorizontal
        {
            public NTabPanelTemplateHorizontalMinimized(NWControl control) : base(control, new NWControlDrawer()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabPanelTemplateHorizontalMinimized(control);
            }
        }

        #endregion

        #endregion

        #region ** Vertical

        public class NTabPanelTemplateVertical : NWTabPanelTemplateVertical
        {
            #region ** 固定値

            private const int MinimizeButtonSize = 14;

            #endregion

            protected NTabPanelTemplateVertical(NWControl control, INWControlDrawer drawer)
                : base(control, drawer) { }

            #region ** プロパティ

            protected new NTabPanel Control
            {
                get { return base.Control as NTabPanel; }
            }

            #endregion

            #region ** メソッド

            #region ** 初期化

            public override void InitializeComponent()
            {
                base.InitializeComponent();

                // DropDownButton を置き換える
                Control.NcControls.Remove(Control.NcControls[NWTabControl.DropDownButton.ControlName]);

                Control.NcControls.Add(new DropDownButton(Host));
                Control.NcControls.Add(new MinimizeButton(Host));
                Control.NcControls.Add(new RestoreButton(Host));
            }

            #endregion

            #region ** レイアウト

            public override void PerformNcLayout(Rectangle newBounds, ref Rectangle clientRect)
            {
                if ((Control.Root as NTabControlCore).Minimizable)
                {
                    int height = ControlSpacing + MinimizeButtonSize + ControlSpacing;
                    clientRect.Y += height;
                    clientRect.Height -= height;
                }

                int scrollButtonHeight = 0;

                if (1 < Control.Controls.Count)
                {

                    clientRect.Height -= ControlSpacing + DropDownButtonSize + ControlSpacing;

                    NWTab lastTab = Control.Controls[Control.Controls.Count - 1] as NWTab;
                    Debug.Assert(null != lastTab, "unexpected error.");

                    if (clientRect.Height < lastTab.Bottom && clientRect.Height > 0)
                    {

                        scrollButtonHeight = Math.Min(ScrollButtonHeight, clientRect.Height / 2);

                        clientRect.Y += scrollButtonHeight + ControlSpacing;
                        clientRect.Height -= (scrollButtonHeight + ControlSpacing) * 2;

                    }

                }

                PerformLayoutScrollButtons(clientRect, scrollButtonHeight);
                PerformLayoutMinimizeButton(newBounds);
                PerformLayoutDropDownButton(newBounds);
            }

            private void PerformLayoutScrollButtons(Rectangle newClientRect, int scrollButtonHeight)
            {
                // 表示する必要がない場合は Empty
                if (1 >= Control.Controls.Count || 0 >= scrollButtonHeight)
                {
                    Control.ScrollUpButton.Bounds = Rectangle.Empty;
                    Control.ScrollDownButton.Bounds = Rectangle.Empty;
                    return;
                }

                NWTab lastTab = Control.Controls[Control.Controls.Count - 1] as NWTab;
                Debug.Assert(null != lastTab, "unexpected error.");

                if (newClientRect.Height >= lastTab.Bottom)
                {
                    Control.ScrollUpButton.Bounds = Rectangle.Empty;
                    Control.ScrollDownButton.Bounds = Rectangle.Empty;
                    return;
                }


                // ScrollUp ボタンを再配置する
                Rectangle scrollUpButtonRect = Rectangle.Empty;
                scrollUpButtonRect.X = Padding.Left;
                scrollUpButtonRect.Y = newClientRect.Top - scrollButtonHeight - ControlSpacing;
                scrollUpButtonRect.Width = newClientRect.Width - Padding.Horizontal;
                scrollUpButtonRect.Height = scrollButtonHeight;

                Control.ScrollUpButton.Bounds = scrollUpButtonRect;

                // ScrollDown ボタンを再配置する
                Rectangle scrollDownButtonRect = Rectangle.Empty;
                scrollDownButtonRect.X = Padding.Left;
                scrollDownButtonRect.Y = Math.Max(newClientRect.Bottom + ControlSpacing,
                                                        scrollUpButtonRect.Bottom);
                scrollDownButtonRect.Width = newClientRect.Width - Padding.Horizontal;
                scrollDownButtonRect.Height = scrollButtonHeight;

                Control.ScrollDownButton.Bounds = scrollDownButtonRect;
            }

            private void PerformLayoutMinimizeButton(Rectangle newBounds)
            {
                if (!(Control.Root as NTabControlCore).Minimizable)
                {
                    Control.MinimizeButton.Bounds = Rectangle.Empty;
                    Control.RestoreButton.Bounds = Rectangle.Empty;
                    return;
                }

                Rectangle bounds = Rectangle.Empty;
                bounds.X = (newBounds.Width - MinimizeButtonSize) / 2;
                bounds.Y = ControlSpacing;
                bounds.Width = MinimizeButtonSize;
                bounds.Height = MinimizeButtonSize;

                Control.MinimizeButton.Bounds = bounds;
                Control.RestoreButton.Bounds = bounds;
            }

            private void PerformLayoutDropDownButton(Rectangle newBounds)
            {
                // 表示する必要がない場合は Empty
                if (!RootControl.ShowDropDownButton || 1 >= Control.Controls.Count)
                {
                    Control.DropDownButton.Bounds = Rectangle.Empty;
                    return;
                }

                Rectangle bounds = Rectangle.Empty;
                bounds.X = (newBounds.Width - DropDownButtonSize) / 2;
                bounds.Y = newBounds.Height - DropDownButtonSize - ControlSpacing;
                bounds.Width = DropDownButtonSize;
                bounds.Height = DropDownButtonSize;

                Control.DropDownButton.Bounds = bounds;
            }

            #endregion

            #endregion
        }

        #region ** Left

        public class NTabControlCoreTemplateLeft : NWTabControlTemplateLeft
        {
            public NTabControlCoreTemplateLeft(NWControl control) : base(control) { }

            public static new INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabControlCoreTemplateLeft(control);
            }

            #region ** メソッド

            public override void InitializeComponent()
            {
                Control.Controls.Add(new NTabPanel(Host));
                Control.Controls.Add(new NWTabPagePanel(Host));
            }

            #endregion
        }

        public class NTabPanelTemplateLeft : NTabPanelTemplateVertical
        {
            public NTabPanelTemplateLeft(NWControl control) : base(control, new NWTabPanelDrawerLeft()) { }

            protected NTabPanelTemplateLeft(NWControl control, INWControlDrawer drawer) : base(control, drawer) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabPanelTemplateLeft(control);
            }

            #region ** メソッド

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                Rectangle workRect = Control.AbsoluteBounds;
                workRect.X = workRect.Right - Padding.Right;
                workRect.Width = Padding.Right;

                NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                drawInfo.BorderRect = workRect;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        #endregion

        #region ** Right

        public class NTabControlCoreTemplateRight : NWTabControlTemplateRight
        {
            public NTabControlCoreTemplateRight(NWControl control) : base(control) { }

            public static new INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabControlCoreTemplateRight(control);
            }

            #region ** メソッド

            public override void InitializeComponent()
            {
                Control.Controls.Add(new NTabPanel(Host));
                Control.Controls.Add(new NWTabPagePanel(Host));
            }

            #endregion
        }

        public class NTabPanelTemplateRight : NTabPanelTemplateVertical
        {
            public NTabPanelTemplateRight(NWControl control) : base(control, new NWTabPanelDrawerRight()) { }

            protected NTabPanelTemplateRight(NWControl control, INWControlDrawer drawer) : base(control, drawer) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabPanelTemplateRight(control);
            }

            #region ** メソッド

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                Rectangle workRect = Control.AbsoluteBounds;
                workRect.Width = Padding.Left;

                NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                drawInfo.BorderRect = workRect;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        #endregion

        #region ** Minimized

        public class NTabControlCoreTemplateVerticalMinimized : NWTabControlTemplateVerticalMinimized
        {
            public NTabControlCoreTemplateVerticalMinimized(NWControl control) : base(control) { }

            public static new INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabControlCoreTemplateVerticalMinimized(control);
            }

            #region ** メソッド

            public override void InitializeComponent()
            {
                Control.Controls.Add(new NTabPanel(Host));
                Control.Controls.Add(new NWTabPagePanel(Host));
            }

            #endregion
        }

        public class NTabPanelTemplateVerticalMinimized : NTabPanelTemplateVertical
        {
            public NTabPanelTemplateVerticalMinimized(NWControl control) : base(control, new NWControlDrawer()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NTabPanelTemplateVerticalMinimized(control);
            }
        }

        #endregion

        #endregion

        #endregion

        #region ** Drawer

        public class NWImageButtonDrawer : NWControlDrawer
        {
            #region ** イベントハンドラ

            /// <summary>
            /// コントロールを描画します。
            /// </summary>
            /// <param name="g">Graphics オブジェクト</param>
            /// <param name="drawInfo">コントロールの描画情報</param>
            protected override void OnDraw(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!(drawInfo.Control is NWButton || drawInfo.Control is NWNCButton))
                {
                    Debug.Fail("target is not Button.");
                    return;
                }

                Rectangle drawRect = drawInfo.Control.AbsoluteBounds;
                drawRect.Width -= 1;
                drawRect.Height -= 1;

                if (drawInfo.Control.Selected)
                {
                    g.FillRectangle(new SolidBrush(ControlPaint.LightLight(SystemColors.Highlight)), drawRect);
                    g.DrawRectangle(SystemPens.Highlight, drawRect);
                }
                else if (drawInfo.Control.Hot)
                {
                    g.FillRectangle(new SolidBrush(ControlPaint.LightLight(ControlPaint.Light(SystemColors.Highlight))), drawRect);
                    g.DrawRectangle(SystemPens.Highlight, drawRect);
                }

                DrawImage(g, drawInfo.Control);
            }

            #endregion

            #region ** メソッド

            private void DrawImage(Graphics g, NWControl control)
            {
                Debug.Assert(null != g, "unexpected error.");
                Debug.Assert(null != control, "unexpected error.");

                Image image = null;
                if (control is NWButton) { image = (control as NWButton).Image; }
                if (control is NWNCButton) { image = (control as NWNCButton).Image; }
                if (null == image) { return; }

                Rectangle rect = control.AbsoluteBounds;
                rect.X += (rect.Width - image.Size.Width) / 2;
                rect.Y += (rect.Height - image.Size.Height) / 2;
                rect.Width = image.Size.Width;
                rect.Height = image.Size.Height;

                g.DrawImage(image, rect);
            }

            #endregion
        }

        #endregion
    }
}
