﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Resources;

    public class SampleMapCtrl : OperatableControl
    {
        // TODO: (SIGLO-64975) コマンドバインディングに変更する
        /// <summary>
        /// メニューアイテム「参照ファイルを開く」で呼び出されます。
        /// </summary>
        public interface IContextMenuHandler
        {
            void ExecuteItemFolder(SampleMapCtrl control);
        }

        private ISampleMapItemsSource _ItemsSource = null;
        private KeyRegionInfo[] _Infos = null;

        private ExchangeOperator _ExchangeOperator = null;
        private KeyRegionLengthOperator _KeyRegionLengthOperator = null;
        private VelocityRegionLengthOperator _VelocityRegionLengthOperator = null;
        private RegionDragOperator _RegionDragOperator = null;

        private bool _Stretch = true;
        private int _KeyRegionVisibleLength = 16;

        private int _VelocityStage = 128;
        private int[] _VelocityPositions = null;

        private SelectedRegionInfo _SelectedRegionInfo = null;

        private System.ComponentModel.IContainer components;
        private ContextMenuStrip _ContextMenuStrip = null;
        private ToolStripMenuItem _MenuItem_ChangeFilePath = null;
        private ToolStripMenuItem _MenuItem_Remove = null;
        private ToolStripMenuItem _MenuItem_DivideKeyRegion = null;
        private ToolStripMenuItem _MenuItem_DivideVelocityRegion = null;
        private ToolStripSeparator toolStripSeparator1;
        private ToolStripMenuItem _MenuItem_OpenItemFolder;
        private ToolStripMenuItem _MenuItem_ExecuteItemFolder;

        private ToolStripMenuItem[] _InsertContextMenuItems = new ToolStripMenuItem[0];
        private ToolStripSeparator _ToolStripSeparator = null;

        private SampleMapCrtlToolTip toolTip = null;
        private string _VelocityRegionSplitterType = string.Empty;

        public event EventHandler SelectChanged;
        public event EventHandler ItemsSourceChanged;

        public delegate void DroppedEventHandler(object sender, DroppedEventArgs e);
        public event DroppedEventHandler Dropped;

        public delegate void ModifyDrawDescriptorHandler(SampleMapDrawDescriptor desc, KeyRegionInfo keyRegionInfo, VelocityRegionInfo velocityRegionInfo);
        public ModifyDrawDescriptorHandler ModifyDrawDescriptor;

        /// <summary>
        ///
        /// </summary>
        public class DroppedEventArgs : EventArgs
        {
            public string FilePath { get; set; }
            public bool Cancel { get; set; }
            public DroppedEventArgs(string filePath)
            {
                FilePath = filePath;
                Cancel = false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public SampleMapCtrl()
        {
            InitializeComponent();

            _ExchangeOperator = new ExchangeOperator(this);
            _KeyRegionLengthOperator = new KeyRegionLengthOperator(this);
            _VelocityRegionLengthOperator = new VelocityRegionLengthOperator(this);
            _RegionDragOperator = new RegionDragOperator(this);
            _SelectedRegionInfo = new SelectedRegionInfo();

            toolTip = new SampleMapCrtlToolTip(this);
            toolTip.Parent = this;

            _DrawDropRegion = null;

            CurrentOperator = _ExchangeOperator;

            KeyRegionDrawer = new KeyRegionDrawer();
            VelocityRegionDrawer = new VelocityRegionDrawer();

#if false
            GetVelocityRegionText = delegate( VelocityRegionInfo velocityRegionInfo)
                {
                    return Path.GetFileName( velocityRegionInfo.FilePath);
                };
#else
#endif

            HorizontalScrollBar.Visible = false;
            VerticalScrollBar.Visible = false;

            UpdateVelocityPositions();
        }

        public int VelocityStage
        {
            get
            {
                return _VelocityStage;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IKeyRegionDrawer KeyRegionDrawer { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IVelocityRegionDrawer VelocityRegionDrawer { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int[] Widths { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ISampleMapItemsSource ItemsSource
        {
            get { return _ItemsSource; }
            set
            {
                if (_ItemsSource != null)
                {
                    _ItemsSource.Updated -= OnItemsSourceUpdated;
                }

                _ItemsSource = value;

                if (_ItemsSource != null)
                {
                    _ItemsSource.Updated += OnItemsSourceUpdated;
                }

                UpdateInfos();
                OnItemsSourceChanged(EventArgs.Empty);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IKeyRegion[] SelectedKeyRegions
        {
            get
            {
                return _SelectedRegionInfo.SelectedKeyRegions;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IVelocityRegion[] SelectedVelocityRegions
        {
            get
            {
                return _SelectedRegionInfo.SelectedVelocityRegions;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Component[] SelectedComponents
        {
            get
            {
                IKeyRegion[] keyRegions = SelectedKeyRegions;
                IVelocityRegion[] velocityRegions = SelectedVelocityRegions;
                List<Component> components = new List<Component>();

                foreach (IKeyRegion kRegion in keyRegions)
                {
                    components.Add(kRegion.Target);
                }
                foreach (IVelocityRegion vRegion in velocityRegions)
                {
                    foreach (IKeyRegion kRegion in keyRegions)
                    {
                        if (vRegion.Parent == kRegion) goto Next;
                    }
                    components.Add(vRegion.Target);
                    Next:
                    continue;
                }

                return components.ToArray();
            }
        }

        public delegate string GetFilePathHandler(string filePath);
        public GetFilePathHandler GetFilePath { get; set; }

        public IContextMenuHandler ContextMenuHandler { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateInfos()
        {
            if (ItemsSource == null)
            {
                _Infos = null;
                Invalidate();
                return;
            }

            _Infos = CreateInfos();
            UpdateSelected(_Infos);
            Invalidate();
        }

        public ToolStripMenuItem[] InsertContextMenuItems
        {
            get
            {
                return _InsertContextMenuItems;
            }
            set
            {
                _InsertContextMenuItems = value;
                if (_InsertContextMenuItems == null)
                {
                    _InsertContextMenuItems = new ToolStripMenuItem[0];
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// 読み取り専用なのか調べる
        /// </summary>
        public bool ReadOnly { get; set; }

        ///--------------------------------
        /// <summary>
        /// フォーカスや有効無効に関係なくグレー表示する
        /// </summary>
        public bool GrayOut { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected KeyRegionInfo[] Infos
        {
            get { return _Infos; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private IRegionInfo _DrawDropRegion;

        private bool DrawVSplitEnable { get; set; }
        private int DrawVSplitPosition { get; set; }
        private KeyRegionInfo[] _DrawVSplitKeyRegions;

        private bool DrawHSplitEnable { get; set; }
        private int DrawHSplitPosition { get; set; }
        private int DrawHSplitBegin { get; set; }
        private int DrawHSplitEnd { get; set; }
        private VelocityRegionInfo[] _DrawHSplitVelocityRegions;

        ///--------------------------------
        /// <summary>
        /// オペレータの状態の変更
        /// </summary>
        protected void ChangeOperatorToExchange()
        {
            ChangeOperatorToExchange(new Point(-1, -1));
        }

        protected void ChangeOperatorToExchange(Point point)
        {
            Debug.WriteLine("Operator changed to exchange");
            CurrentOperator = _ExchangeOperator;
            CurrentOperator.Initialize(point);
        }

        protected void ChangeOperatorToKeyRegionLength(Point point)
        {
            Debug.WriteLine("Operator changed to key region length");
            CurrentOperator = _KeyRegionLengthOperator;
            CurrentOperator.Initialize(point);
        }

        protected void ChangeOperatorToVelocityRegionLength(Point point)
        {
            Debug.WriteLine("Operator changed to velocity region length");
            CurrentOperator = _VelocityRegionLengthOperator;
            CurrentOperator.Initialize(point);
        }

        protected void ChangeOperatorToRegionDragOperator(Point point)
        {
            Debug.WriteLine("Operator changed to region drag operator");
            CurrentOperator = _RegionDragOperator;
            CurrentOperator.Initialize(point);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected SampleMapCollisionResult Collision(int x, int y)
        {
            SampleMapCollisionResult result = new SampleMapCollisionResult();

            if (Infos == null)
            {
                return result;
            }

            if (Contains(new Point(x, y)) == true)
            {
                foreach (KeyRegionInfo k_info in Infos)
                {
                    int splitterWidth = 6;
                    if (k_info.Bounds.Width <= splitterWidth * 2)
                    {
                        splitterWidth = 0;
                    }
                    else if (k_info.Bounds.Width <= splitterWidth * 3)
                    {
                        splitterWidth = k_info.Bounds.Width - (splitterWidth * 2);
                    }

                    int LSplitterBegin = k_info.Bounds.X;
                    int LSplitterEnd = k_info.Bounds.X + splitterWidth;
                    int RSplitterBegin = k_info.Bounds.X + k_info.Bounds.Width - splitterWidth;
                    int RSplitterEnd = k_info.Bounds.X + k_info.Bounds.Width;

                    if (LSplitterBegin <= x && x <= LSplitterEnd)
                    {
                        result.Type = "KeyRegionLeftSplitter";
                        result.RegionInfo = k_info;
                        goto Return;
                    }
                    else if (RSplitterBegin <= x && x <= RSplitterEnd)
                    {
                        result.Type = "KeyRegionRightSplitter";
                        result.RegionInfo = k_info;
                        goto Return;
                    }
                    else if (k_info.Bounds.Contains(new Point(x, y)) == true)
                    {
                        if (k_info.Region != null)
                        {
                            result.Type = "KeyRegion";
                        }
                        else
                        {
                            result.Type = "None";
                        }
                        result.RegionInfo = k_info;
                        goto Return;
                    }
                    else
                    {
                        if (k_info.VelocityRegionInfos != null)
                        {
                            foreach (VelocityRegionInfo v_info in k_info.VelocityRegionInfos)
                            {
                                if (v_info.Bounds.Contains(new Point(x, y)) == true)
                                {
                                    int splitterHeight = 6;
                                    if (v_info.Bounds.Height <= splitterHeight * 2)
                                    {
                                        splitterHeight = 0;
                                    }
                                    else if (v_info.Bounds.Height <= splitterHeight * 3)
                                    {
                                        splitterHeight =
                                            v_info.Bounds.Height - (splitterHeight * 2);
                                    }

                                    int TSplitterBegin = v_info.Bounds.Y;
                                    int TSplitterEnd = v_info.Bounds.Y + splitterHeight;
                                    int BSplitterBegin = (v_info.Bounds.Y +
                                                           v_info.Bounds.Height - splitterHeight);
                                    int BSplitterEnd = v_info.Bounds.Y + v_info.Bounds.Height;

                                    result.RegionInfo = v_info;
                                    if (v_info.Maximum != _VelocityStage - 1 &&
                                        TSplitterBegin <= y && y <= TSplitterEnd)
                                    {
                                        result.Type = "VelocityRegionTopSplitter";
                                        result.VelocitySplitterValue = v_info.Maximum + 1;
                                    }
                                    else if (v_info.Minimum != 0 &&
                                             BSplitterBegin <= y && y <= BSplitterEnd)
                                    {
                                        result.Type = "VelocityRegionBottomSplitter";
                                        result.VelocitySplitterValue = v_info.Minimum;
                                    }
                                    else if (v_info.Region != null)
                                    {
                                        result.Type = "VelocityRegion";
                                    }
                                    else
                                    {
                                        result.Type = "None";
                                        result.RegionInfo = k_info;
                                    }
                                    goto Return;
                                }
                            }
                        }
                    }
                }
            }

            Return:
            return result;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void UpdateVelocityPositions()
        {
            int stage = _VelocityStage + 1;
            int[] positions = new int[stage];
            double height = 0.0;
            double one = 0.0;
            int modDot = 0;
            int interval = 0;
            int nextY = 0;

            height = ClientRectangle.Height - _KeyRegionVisibleLength;
            one = height / (double)_VelocityStage;

            modDot = (int)(height - (one * (double)_VelocityStage));
            if (modDot > 0)
            {
                interval = _VelocityStage / modDot;
            }

            for (int index = 0; index < stage; index++)
            {
                positions[index] = nextY;

                nextY = (int)((double)(index + 1) * one);
                if (interval > 0 && (index % interval) == 0)
                {
                    nextY++;
                }
            }

            _VelocityPositions = positions;
        }

        ///--------------------------------
        /// <summary>
        /// クライアント座標位置からキーリージョンを取得
        /// </summary>
        protected int GetKeyByPosition(int position)
        {
            Debug.Assert(Widths != null, "Width is not set");
            int pos = 0;
            int key;

            for (key = 0; key < Widths.Length; key++)
            {
                pos += Widths[key];
                if (position < pos)
                {
                    break;
                }
            }

            return key;
        }

        ///--------------------------------
        /// <summary>
        /// キーリージョンのクライアント座標での位置の取得
        /// </summary>
        protected int GetPositionByKey(int targetKey)
        {
            Debug.Assert(Widths != null, "Width is not set");
            int position = 0;

            for (int key = 0; key < targetKey; key++)
            {
                position += Widths[key];
            }

            return position;
        }

        ///--------------------------------
        /// <summary>
        /// キーリージョンのクライアント座標での幅の取得
        /// </summary>
        protected int GetLengthByKey(int beginKey, int endKey)
        {
            Debug.Assert(Widths != null, "Width is not set");
            int length = 0;

            for (int key = beginKey; key <= endKey; key++)
            {
                length += Widths[key];
            }
            return length;
        }

        ///--------------------------------
        /// <summary>
        /// クライアント座標からベロシティリージョンの取得
        /// </summary>
        protected int GetVelocityByPosition(int position)
        {
            int vel;

            for (vel = 0; vel < _VelocityStage; vel++)
            {
                if (ClientRectangle.Height - _VelocityPositions[vel] <= position)
                {
                    break;
                }
            }

            return vel;
        }

        ///--------------------------------
        /// <summary>
        /// ベロシティリージョンのクライアント座標での位置の取得
        /// </summary>
        protected int GetPositionByVelocity(int targetVel)
        {
            return ClientRectangle.Height - _VelocityPositions[targetVel];
        }

        ///--------------------------------
        /// <summary>
        /// ベロシティリージョンのクライアント座標での幅の取得
        /// </summary>
        protected int GetLengthByVelocity(int beginVel, int endVel)
        {
            int beginY = _VelocityPositions[beginVel];
            int endY = _VelocityPositions[endVel + 1];

            return endY - beginY;
        }

        /// <summary>
        /// キーリージョンの描画
        /// </summary>
        protected void DrawKeyRegion(SampleMapDrawDescriptor drawDesc, KeyRegionInfo keyRegionInfo)
        {
            if (_Stretch == false)
            {
                //今は実装の必要無し
                return;
            }

            //空白なのか？
            drawDesc.Blank = keyRegionInfo.Region == null ? true : false;
            drawDesc.Bounds = keyRegionInfo.Bounds;
            KeyRegionDrawer.Draw(drawDesc);

            //
            bool modify = drawDesc.Modify;
            bool drop = drawDesc.Drop;

            foreach (VelocityRegionInfo velRegionInfo in keyRegionInfo.VelocityRegionInfos)
            {
                drawDesc.Bounds = velRegionInfo.Bounds;
                drawDesc.Text = Path.GetFileName(velRegionInfo.FilePath);
                drawDesc.Modify = modify;
                drawDesc.Selected = IsSelected(velRegionInfo);

                if (drop == true)
                {
                    drawDesc.Drop = true;
                }
                else if (_DrawDropRegion == velRegionInfo)
                {
                    drawDesc.Drop = true;
                }
                else
                {
                    drawDesc.Drop = false;
                }

                if (DrawHSplitEnable == true)
                {
                    if (_DrawHSplitVelocityRegions[0] == velRegionInfo ||
                        _DrawHSplitVelocityRegions[1] == velRegionInfo)
                    {
                        drawDesc.Modify = true;
                        drawDesc.HSplitPosition = DrawHSplitPosition;
                    }
                }

                if (this.ModifyDrawDescriptor != null)
                {
                    this.ModifyDrawDescriptor(drawDesc, keyRegionInfo, velRegionInfo);
                }

                VelocityRegionDrawer.Draw(drawDesc);
            }
        }

        /// <summary>
        /// 描画
        /// </summary>
        protected override void Draw(Graphics gc)
        {
            if (Infos == null)
            {
                return;
            }

            //
            SampleMapDrawDescriptor drawDesc = null;
            drawDesc = new SampleMapDrawDescriptor(gc, Font);
            drawDesc.Focused = Focused || GrayOut == false;
            drawDesc.Height = Height;

            foreach (KeyRegionInfo keyRegionInfo in Infos)
            {
                drawDesc.Enabled = true;
                drawDesc.Selected = IsSelected(keyRegionInfo);
                drawDesc.Modify = false;
                drawDesc.Bounds = new Rectangle();
                drawDesc.Drop = false;

                if (DrawVSplitEnable == true)
                {
                    if (_DrawVSplitKeyRegions[0] == keyRegionInfo ||
                        _DrawVSplitKeyRegions[1] == keyRegionInfo)
                    {
                        drawDesc.Modify = true;
                        drawDesc.VSplitPosition = DrawVSplitPosition;
                    }
                }

                if (_DrawDropRegion == keyRegionInfo)
                {
                    drawDesc.Drop = true;
                }

                DrawKeyRegion(drawDesc, keyRegionInfo);
            }

            if (DrawVSplitEnable == true)
            {
                DrawVSplit(gc);
            }

            if (DrawHSplitEnable == true)
            {
                DrawHSplit(gc);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnTrackContextMenu(MouseEventArgs e)
        {
            _MenuItem_ChangeFilePath.Enabled = false;
            _MenuItem_DivideKeyRegion.Enabled = false;
            _MenuItem_DivideVelocityRegion.Enabled = false;
            _MenuItem_Remove.Enabled = false;

            if (_ContextMenuStrip != null)
            {
                if (ReadOnly == false)
                {
                    if (_SelectedRegionInfo.Count > 0)
                    {
                        _MenuItem_ChangeFilePath.Enabled = true;
                        _MenuItem_DivideKeyRegion.Enabled = true;
                        _MenuItem_DivideVelocityRegion.Enabled = true;
                        _MenuItem_Remove.Enabled = true;
                    }
                    if (_SelectedRegionInfo.ContainsKeyRegionInfo == true)
                    {
                        _MenuItem_ChangeFilePath.Enabled = false;
                        IKeyRegion[] keyRegions = SelectedKeyRegions;
                        foreach (IKeyRegion keyRegion in keyRegions)
                        {
                            if (keyRegion.Children.Count != 1)
                            {
                                _MenuItem_DivideVelocityRegion.Enabled = false;
                            }
                        }
                    }
                    if (_SelectedRegionInfo.ContainsVelocityRegionInfo == true)
                    {
                        IVelocityRegion[] velRegions = SelectedVelocityRegions;
                        foreach (IVelocityRegion velRegion in velRegions)
                        {
                            if (velRegion.Parent.Children.Count != 1)
                            {
                                _MenuItem_DivideKeyRegion.Enabled = false;
                            }
                        }
                    }

                    bool enabled = false;
                    if (_SelectedRegionInfo.ContainsKeyRegionInfo == true)
                    {
                        foreach (IKeyRegion keyRegion in
                                  _SelectedRegionInfo.SelectedKeyRegions)
                        {
                            if (keyRegion.Minimum != keyRegion.Maximum)
                            {
                                enabled = true;
                            }
                        }
                        if (enabled == false)
                        {
                            _MenuItem_DivideKeyRegion.Enabled = false;
                        }
                    }

                    enabled = false;
                    if (_SelectedRegionInfo.ContainsVelocityRegionInfo == true)
                    {
                        foreach (IVelocityRegion velRegion in
                                  _SelectedRegionInfo.SelectedVelocityRegions)
                        {
                            if (velRegion.Minimum != velRegion.Maximum)
                            {
                                enabled = true;
                            }
                        }
                        if (enabled == false)
                        {
                            _MenuItem_DivideVelocityRegion.Enabled = false;
                        }
                    }
                }
                _ContextMenuStrip.Show(this, new Point(e.X, e.Y));
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            UpdateVelocityPositions();
            UpdateInfos();
        }

        private string OnGetFilePath(string filePath)
        {
            if (GetFilePath != null)
            {
                return GetFilePath(filePath);
            }
            else
            {
                FileDialog dialog = new OpenFileDialog();
                if (filePath != string.Empty)
                {
                    dialog.FileName = Path.GetFileName(filePath);
                    dialog.InitialDirectory = Path.GetDirectoryName(filePath);
                }
                dialog.Title = MessageResource.DialogTitle_SampleMap_ReferenceWaveFile_Text;
                dialog.Filter =
                    "Wave Files(*.wav;*.aif;*.aiff)|*.wav;*.aif;*.aiff|All files (*.*)|*.*";
                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    return dialog.FileName;
                }
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void DrawVSplit(Graphics gc)
        {
            gc.DrawLine(SystemPens.Highlight,
                         DrawVSplitPosition, 0,
                         DrawVSplitPosition, Height);
            gc.DrawLine(SystemPens.Highlight,
                         DrawVSplitPosition - 1, 0,
                         DrawVSplitPosition - 1, Height);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void DrawHSplit(Graphics gc)
        {
            gc.DrawLine(SystemPens.Highlight,
                         DrawHSplitBegin, DrawHSplitPosition,
                         DrawHSplitEnd, DrawHSplitPosition);
            gc.DrawLine(SystemPens.Highlight,
                         DrawHSplitBegin, DrawHSplitPosition - 1,
                         DrawHSplitEnd, DrawHSplitPosition - 1);
        }
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnItemsSourceUpdated(object sender, EventArgs e)
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new MethodInvoker(delegate ()
                {
                    this.UpdateInfos();
                    OnSelectChanged(EventArgs.Empty);
                }));
            }
            else
            {
                this.UpdateInfos();
                OnSelectChanged(EventArgs.Empty);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private KeyRegionInfo CreateKeyRegionInfo(int min, int max)
        {
            KeyRegionInfo keyInfo = new KeyRegionInfo();
            keyInfo.Region = null;
            keyInfo.Minimum = min;
            keyInfo.Maximum = max;
            Rectangle bounds = new Rectangle(GetPositionByKey(min),
                                              0,
                                              GetLengthByKey(min, max),
                                              _KeyRegionVisibleLength);
            keyInfo.Bounds = bounds;

            VelocityRegionInfo[] velInfos = new VelocityRegionInfo[1];
            VelocityRegionInfo velInfo = velInfos[0] = new VelocityRegionInfo();
            velInfo.Region = null;
            velInfo.FilePath = string.Empty;
            velInfo.Minimum = 0;
            velInfo.Maximum = 127;
            velInfo.KeyRegionInfo = keyInfo;
            {
                bounds = new Rectangle();
                int height = GetLengthByVelocity(velInfo.Minimum, velInfo.Maximum);
                bounds.X = keyInfo.Bounds.X;
                bounds.Y = GetPositionByVelocity(velInfo.Minimum) - height;
                bounds.Width = keyInfo.Bounds.Width;
                bounds.Height = height;
                velInfo.Bounds = bounds;
            }

            keyInfo.VelocityRegionInfos = velInfos;

            return keyInfo;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private KeyRegionInfo CreateKeyRegionInfo(IKeyRegion keyRegion)
        {
            List<VelocityRegionInfo> list = new List<VelocityRegionInfo>();
            KeyRegionInfo keyInfo = null;
            VelocityRegionInfo velInfo = null;

            keyInfo = CreateKeyRegionInfo(keyRegion.Minimum, keyRegion.Maximum);
            keyInfo.Region = keyRegion;

            VelocityRegionInfo prev = null;
            foreach (IVelocityRegion velRegion in keyRegion.Children)
            {
                velInfo = new VelocityRegionInfo();
                velInfo.Region = velRegion;
                velInfo.FilePath = velRegion.FilePath;
                velInfo.Minimum = velRegion.Minimum;
                velInfo.Maximum = velRegion.Maximum;
                velInfo.KeyRegionInfo = keyInfo;
                {
                    Rectangle bounds = new Rectangle();
                    int height = GetLengthByVelocity(velInfo.Minimum, velInfo.Maximum);
                    bounds.X = keyInfo.Bounds.X;
                    bounds.Y = GetPositionByVelocity(velInfo.Minimum) - height;
                    bounds.Width = keyInfo.Bounds.Width;
                    bounds.Height = height;
                    velInfo.Bounds = bounds;

                }
                velInfo.Prev = prev;
                if (prev != null)
                {
                    prev.Next = velInfo;
                }
                prev = velInfo;
                list.Add(velInfo);
            }

            keyInfo.VelocityRegionInfos = list.ToArray();

            return keyInfo;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private KeyRegionInfo[] CreateInfos()
        {
            List<KeyRegionInfo> list = new List<KeyRegionInfo>();
            KeyRegionInfo info = null;
            int position = 0;
            KeyRegionInfo prevInfo = null;

            if (ItemsSource.Items == null) goto Return;
            foreach (IKeyRegion keyRegion in ItemsSource.Items)
            {
                //空白リージョン情報の作成;
                if (position < keyRegion.Minimum)
                {
                    info = CreateKeyRegionInfo(position, keyRegion.Minimum - 1);
                    info.Prev = prevInfo;
                    if (prevInfo != null)
                    {
                        prevInfo.Next = info;
                    }
                    list.Add(info);
                    prevInfo = info;
                    position = keyRegion.Minimum;
                }

                //キーリージョン情報の作成;
                info = CreateKeyRegionInfo(keyRegion);
                info.Prev = prevInfo;
                if (prevInfo != null)
                {
                    prevInfo.Next = info;
                }
                list.Add(info);
                prevInfo = info;
                position = keyRegion.Maximum + 1;
            }
            //空白リージョン情報の作成;
            if (ItemsSource.Items.Count > 0 && position < Widths.Length)
            {
                info = CreateKeyRegionInfo(position, Widths.Length - 1);
                info.Prev = prevInfo;
                if (prevInfo != null)
                {
                    prevInfo.Next = info;
                }
                list.Add(info);
            }
            Return:
            return list.ToArray();
        }


        ///@@@
        ///------------------------------------------------------------------------
        /// <summary>
        /// 基本のオペレータ
        /// </summary>
        public class SampleMapCtrlOperator : OperatableControlOperator
        {

            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public SampleMapCtrlOperator(SampleMapCtrl ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected SampleMapCtrl OwnerCtrl
            {
                get { return (SampleMapCtrl)base.OperatableControl; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool PressedControlKey
            {
                get { return (Control.ModifierKeys & Keys.Control) != 0; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool PressedShiftKey
            {
                get { return (Control.ModifierKeys & Keys.Shift) != 0; }
            }
        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// 待機状態のオペレータ
        /// </summary>
        private class ExchangeOperator : SampleMapCtrlOperator
        {
            private Point _FirstMouseDownPoint = new Point(-1, -1);
            private SampleMapCollisionResult _FirstCollisionResult = null;

            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public ExchangeOperator(SampleMapCtrl ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _FirstMouseDownPoint = beginPoint;
                _FirstCollisionResult = OwnerCtrl.Collision(beginPoint.X, beginPoint.Y);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                if (OwnerCtrl.ItemsSource.Items != null &&
                     0 < OwnerCtrl.ItemsSource.Items.Count)
                {
                    switch (e.Button)
                    {
                        case MouseButtons.Left:
                            OwnerCtrl.SelectRegion(e.X, e.Y);
                            _FirstMouseDownPoint = new Point(e.X, e.Y);
                            _FirstCollisionResult = OwnerCtrl.Collision(e.X, e.Y);
                            break;

                        case MouseButtons.Middle:
                            break;

                        case MouseButtons.Right:
                            OwnerCtrl.SelectRegion(e.X, e.Y);
                            break;
                    }
                }

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                if (OwnerCtrl.ItemsSource.Items != null &&
                     0 < OwnerCtrl.ItemsSource.Items.Count && OwnerCtrl.ReadOnly == false)
                {
                    Size dragSize = SystemInformation.DragSize;

                    if (_FirstCollisionResult != null &&
                        (Math.Abs(e.X - _FirstMouseDownPoint.X) > dragSize.Width ||
                          Math.Abs(e.Y - _FirstMouseDownPoint.Y) > dragSize.Height))
                    {
                        switch (_FirstCollisionResult.Type)
                        {
                            case "KeyRegionLeftSplitter":
                            case "KeyRegionRightSplitter":
                                OwnerCtrl.ChangeOperatorToKeyRegionLength(_FirstMouseDownPoint);
                                break;

                            case "VelocityRegionTopSplitter":
                            case "VelocityRegionBottomSplitter":
                                OwnerCtrl.ChangeOperatorToVelocityRegionLength(_FirstMouseDownPoint);
                                break;

                            case "KeyRegion":
                                if (OwnerCtrl._SelectedRegionInfo.SelectedKeyRegions.Length == 1 &&
                                     OwnerCtrl._SelectedRegionInfo.
                                     ContainsVelocityRegionInfo == false &&
                                     OwnerCtrl._SelectedRegionInfo.SelectedKeyRegions[0] ==
                                     _FirstCollisionResult.RegionInfo.Region)
                                {
                                    OwnerCtrl.ChangeOperatorToRegionDragOperator
                                        (_FirstMouseDownPoint);
                                }
                                break;

                            case "VelocityRegion":
                                if (OwnerCtrl._SelectedRegionInfo.
                                     SelectedVelocityRegions.Length == 1 &&
                                     OwnerCtrl._SelectedRegionInfo.ContainsKeyRegionInfo == false &&
                                     OwnerCtrl._SelectedRegionInfo.SelectedVelocityRegions[0] ==
                                     _FirstCollisionResult.RegionInfo.Region)
                                {
                                    OwnerCtrl.ChangeOperatorToRegionDragOperator
                                        (_FirstMouseDownPoint);
                                }
                                break;
                        }
                    }
                    SampleMapCollisionResult result = OwnerCtrl.Collision(e.X, e.Y);
                    OwnerCtrl.Cursor = OwnerCtrl.GetCursor(result.Type);
                    switch (result.Type)
                    {
                        case "VelocityRegionTopSplitter":
                        case "VelocityRegionBottomSplitter":
                            OwnerCtrl._VelocityRegionSplitterType = result.Type;
                            OwnerCtrl.toolTip.Value = result.VelocitySplitterValue;
                            OwnerCtrl.toolTip.SetLocation(e.X, e.Y);
                            OwnerCtrl.toolTip.Show();
                            OwnerCtrl.Refresh();
                            break;

                        default:
                            if (OwnerCtrl.toolTip.Visible == true)
                            {
                                OwnerCtrl.toolTip.Hide();
                                OwnerCtrl.Refresh();
                            }
                            break;
                    }

                    OwnerCtrl.Invalidate();
                }

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                OwnerCtrl.Cursor = Cursors.Default;
                _FirstMouseDownPoint = new Point(-1, -1);
                _FirstCollisionResult = null;

                switch (e.Button)
                {
                    case MouseButtons.Left:
                        break;

                    case MouseButtons.Middle:
                        break;

                    case MouseButtons.Right:
                        OwnerCtrl.OnTrackContextMenu(e);
                        break;
                }

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragOver(DragEventArgs e)
            {
                if (OwnerCtrl.ReadOnly == true)
                {
                    OwnerCtrl._DrawDropRegion = null;
                    e.Effect = DragDropEffects.None;
                    return;
                }

                e.Effect = DragDropEffects.Copy;
                Point p = OwnerCtrl.PointToClient(new Point(e.X, e.Y));
                SampleMapCollisionResult result = OwnerCtrl.Collision(p.X, p.Y);
                IRegionInfo regionInfo = result.RegionInfo;

                OwnerCtrl._DrawDropRegion = null;
                if (regionInfo.Region == null)
                {
                    OwnerCtrl._DrawDropRegion = regionInfo;
                }
                else if (regionInfo is VelocityRegionInfo)
                {
                    OwnerCtrl._DrawDropRegion = regionInfo;
                }
                else if (regionInfo is KeyRegionInfo &&
                          regionInfo.Region.Target.Children.Count == 1)
                {
                    OwnerCtrl._DrawDropRegion = regionInfo;
                }
                OwnerCtrl.Invalidate();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragLeave(System.EventArgs e)
            {
                OwnerCtrl._DrawDropRegion = null;
                OwnerCtrl.Invalidate();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragDrop(DragEventArgs e)
            {
                if (OwnerCtrl.ReadOnly == true)
                {
                    OwnerCtrl._DrawDropRegion = null;
                    e.Effect = DragDropEffects.None;
                    return;
                }

                string[] files = e.Data.GetData(DataFormats.FileDrop) as string[];

                if (files != null && files.Length == 1)
                {
                    DroppedEventArgs args = new DroppedEventArgs(files[0]);
                    if (OwnerCtrl.OnDropped(args) == false) goto Return;

                    Point p = OwnerCtrl.PointToClient(new Point(e.X, e.Y));
                    SampleMapCollisionResult result = OwnerCtrl.Collision(p.X, p.Y);
                    IRegionInfo regionInfo = result.RegionInfo;
                    if (regionInfo is VelocityRegionInfo)
                    {
                        ((IVelocityRegion)((VelocityRegionInfo)(regionInfo)).Region).FilePath =
                            files[0];
                    }
                    else if (regionInfo is KeyRegionInfo)
                    {
                        KeyRegionInfo keyInfo = (KeyRegionInfo)regionInfo;
                        if (regionInfo.Region == null)
                        {
                            int index = 0;
                            KeyRegionInfo ki = keyInfo;
                            while (ki != null)
                            {
                                if (ki.Region != null) ++index;
                                ki = (KeyRegionInfo)(ki.Prev);
                            }
                            OwnerCtrl.CreateKeyRegion(index,
                                                       keyInfo.Minimum,
                                                       keyInfo.Maximum,
                                                       files[0]);
                        }
                        else if (regionInfo.Region.Target.Children.Count == 1)
                        {
                            ((IVelocityRegion)
                             ((KeyRegionInfo)(regionInfo)).
                             VelocityRegionInfos[0].Region).FilePath = files[0];
                        }
                    }
                }

                Return:
                OwnerCtrl._DrawDropRegion = null;
                OwnerCtrl.UpdateInfos();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDoubleClick(MouseEventArgs e)
            {
                if (OwnerCtrl.ItemsSource.Items != null && OwnerCtrl.ReadOnly == false)
                {
                    switch (e.Button)
                    {
                        case MouseButtons.Left:
                            SampleMapCollisionResult result = OwnerCtrl.Collision(e.X, e.Y);
                            if (result.Type == "None")
                            {
                                OwnerCtrl.CreateKeyRegion(result.RegionInfo);
                                OwnerCtrl.UpdateInfos();
                                OwnerCtrl.Invalidate();
                            }
                            else if (result.Type == "VelocityRegion")
                            {
                                OwnerCtrl.ChangeVelocityRegionFilePath(result.RegionInfo);
                                OwnerCtrl.UpdateInfos();
                                OwnerCtrl.Invalidate();
                            }
                            break;
                    }
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseWheel(MouseEventArgs e)
            {
                int relativeMove = 0;

                relativeMove = e.Delta * SystemInformation.MouseWheelScrollLines / 120;
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnKeyDown(KeyEventArgs e)
            {
                switch (e.KeyData)
                {
                    case Keys.Up:
                    case Keys.Down:
                    case Keys.Left:
                    case Keys.Right:
                    case Keys.Return:
                        break;
                }
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// キーリージョンの長さ変更のオペレータ
        /// </summary>
        private class KeyRegionLengthOperator : SampleMapCtrlOperator
        {
            private Point _FirstMouseDownPoint = new Point(-1, -1);
            private SampleMapCollisionResult _FirstCollisionResult = null;

            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public KeyRegionLengthOperator(SampleMapCtrl ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _FirstMouseDownPoint = beginPoint;
                _FirstCollisionResult = OwnerCtrl.Collision(beginPoint.X, beginPoint.Y);
                OwnerCtrl.DrawVSplitEnable = true;
                OwnerCtrl.DrawVSplitPosition =
                    OwnerCtrl.VSplitPosition(_FirstMouseDownPoint.X,
                                              _FirstCollisionResult.Type,
                                              _FirstCollisionResult.RegionInfo);
                OwnerCtrl._DrawVSplitKeyRegions =
                    OwnerCtrl.VSplitKeyRegions(_FirstCollisionResult.Type,
                                                _FirstCollisionResult.RegionInfo);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                OwnerCtrl.DrawVSplitPosition =
                    OwnerCtrl.VSplitPosition(e.X,
                                              _FirstCollisionResult.Type,
                                              _FirstCollisionResult.RegionInfo);
                OwnerCtrl.Invalidate();

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                int position = OwnerCtrl.VSplitPosition(e.X,
                                                         _FirstCollisionResult.Type,
                                                         _FirstCollisionResult.RegionInfo);
                OwnerCtrl.ChangeKeyRegionValue
                    (OwnerCtrl.VSplitKeyRegions(_FirstCollisionResult.Type,
                                                  _FirstCollisionResult.RegionInfo),
                      OwnerCtrl.GetKeyByPosition(position));

                OwnerCtrl.Cursor = Cursors.Default;
                _FirstMouseDownPoint = new Point(-1, -1);
                _FirstCollisionResult = null;
                OwnerCtrl.DrawVSplitEnable = false;
                OwnerCtrl.ChangeOperatorToExchange();
                OwnerCtrl.Invalidate();

                return true;
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// ベロシティーリージョンの長さ変更のオペレータ
        /// </summary>
        private class VelocityRegionLengthOperator : SampleMapCtrlOperator
        {
            private Point _FirstMouseDownPoint = new Point(-1, -1);
            private SampleMapCollisionResult _FirstCollisionResult = null;

            // --------------------------------
            // <summary>
            // コンストラクタ
            // </summary>
            public VelocityRegionLengthOperator(SampleMapCtrl ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _FirstMouseDownPoint = beginPoint;
                _FirstCollisionResult = OwnerCtrl.Collision(beginPoint.X, beginPoint.Y);
                OwnerCtrl.DrawHSplitEnable = true;
                OwnerCtrl.DrawHSplitPosition =
                    OwnerCtrl.HSplitPosition(_FirstMouseDownPoint.Y,
                                              _FirstCollisionResult.Type,
                                              _FirstCollisionResult.RegionInfo);
                OwnerCtrl.DrawHSplitBegin = _FirstCollisionResult.RegionInfo.Bounds.X;
                OwnerCtrl.DrawHSplitEnd = (_FirstCollisionResult.RegionInfo.Bounds.X +
                                            _FirstCollisionResult.RegionInfo.Bounds.Width);
                OwnerCtrl._DrawHSplitVelocityRegions =
                    OwnerCtrl.HSplitVelocityRegions(_FirstCollisionResult.Type,
                                                     _FirstCollisionResult.RegionInfo);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                int pos = OwnerCtrl.HSplitPosition(e.Y,
                                                    _FirstCollisionResult.Type,
                                                    _FirstCollisionResult.RegionInfo);
                OwnerCtrl.DrawHSplitPosition = pos;
                int vel = OwnerCtrl.GetChangeVelocityRegionValue
                    (OwnerCtrl.HSplitVelocityRegions(_FirstCollisionResult.Type,
                                                       _FirstCollisionResult.RegionInfo),
                      OwnerCtrl.GetVelocityByPosition(pos));
                OwnerCtrl.toolTip.Value = vel;
                OwnerCtrl.toolTip.SetLocation(e.X, e.Y);
                OwnerCtrl.toolTip.Show();
                OwnerCtrl.Refresh();

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                int position = OwnerCtrl.HSplitPosition(e.Y,
                                                         _FirstCollisionResult.Type,
                                                         _FirstCollisionResult.RegionInfo);
                OwnerCtrl.ChangeVelocityRegionValue
                    (OwnerCtrl.HSplitVelocityRegions(_FirstCollisionResult.Type,
                                                       _FirstCollisionResult.RegionInfo),
                      OwnerCtrl.GetVelocityByPosition(position));

                OwnerCtrl.Cursor = Cursors.Default;
                _FirstMouseDownPoint = new Point(-1, -1);
                _FirstCollisionResult = null;
                OwnerCtrl.DrawHSplitEnable = false;
                OwnerCtrl.ChangeOperatorToExchange();
                OwnerCtrl.toolTip.Hide();
                OwnerCtrl.Refresh();

                return true;
            }
        }

        private bool Contains(Point pos)
        {
            Rectangle r = new Rectangle(0, 0, Width, Height);

            return r.Contains(pos);
        }

        private void ChangeKeyRegionValue(KeyRegionInfo[] infos, int key)
        {
            try
            {
                _ItemsSource.BeginTransaction();

                if (infos[0] != null && infos[0].Region != null)
                {
                    infos[0].Region.MaximumIntact = key - 1;
                }
                if (infos[1] != null && infos[1].Region != null)
                {
                    infos[1].Region.MinimumIntact = key;
                }
                UpdateInfos();
                _ItemsSource.EndTransaction();
            }
            catch
            {
                _ItemsSource.CancelTransaction();
            }
        }

        private void ChangeVelocityRegionValue(VelocityRegionInfo[] infos, int vel)
        {
            try
            {
                int vel0 = -1;
                int vel1 = 0;

                _ItemsSource.BeginTransaction();
                if (infos[0] != null && infos[0].Region != null)
                {
                    if (infos[0].Region.Minimum == vel)
                    {
                        vel0 = 0;
                        vel1 = 1;
                    }
                }

                if (infos[0] != null && infos[0].Region != null)
                {
                    infos[0].Region.MaximumIntact = vel + vel0;
                }
                if (infos[1] != null && infos[1].Region != null)
                {
                    infos[1].Region.MinimumIntact = vel + vel1;
                }
                UpdateInfos();
                _ItemsSource.EndTransaction();
            }
            catch
            {
                _ItemsSource.CancelTransaction();
            }
        }

        private int GetChangeVelocityRegionValue(VelocityRegionInfo[] infos, int vel)
        {
            int returnValue = vel;

            if (infos[0] != null && infos[0].Region != null)
            {
                if (vel <= infos[0].Region.Minimum)
                {
                    returnValue = vel + 1;
                }
            }

            return returnValue;
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// リージョンドラッグアンドドロップオペレータ
        /// </summary>
        private class RegionDragOperator : SampleMapCtrlOperator
        {
            private Point _FirstMouseDownPoint = new Point(-1, -1);
            private SampleMapCollisionResult _FirstCollisionResult = null;

            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public RegionDragOperator(SampleMapCtrl ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _FirstMouseDownPoint = beginPoint;
                _FirstCollisionResult = OwnerCtrl.Collision(beginPoint.X, beginPoint.Y);
                OwnerCtrl.DoDragDrop(_FirstCollisionResult, (DragDropEffects.None |
                                                               DragDropEffects.Copy |
                                                               DragDropEffects.Link));
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                if (OwnerCtrl.ItemsSource.Items != null &&
                     0 < OwnerCtrl.ItemsSource.Items.Count)
                {
                    switch (e.Button)
                    {
                        case MouseButtons.Left:
                            OwnerCtrl.SelectRegion(e.X, e.Y);
                            _FirstMouseDownPoint = new Point(e.X, e.Y);
                            _FirstCollisionResult = OwnerCtrl.Collision(e.X, e.Y);
                            break;

                        case MouseButtons.Middle:
                            break;

                        case MouseButtons.Right:
                            OwnerCtrl.SelectRegion(e.X, e.Y);
                            break;
                    }
                }

                OwnerCtrl.ChangeOperatorToExchange(new Point(e.X, e.Y));

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                OwnerCtrl.ChangeOperatorToExchange();

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragOver(DragEventArgs e)
            {
                e.Effect = DragDropEffects.Link;
                OwnerCtrl._DrawDropRegion = null;
                Point p = OwnerCtrl.PointToClient(new Point(e.X, e.Y));
                SampleMapCollisionResult result = OwnerCtrl.Collision(p.X, p.Y);
                if ((ModifierKeys & Keys.Control) == Keys.Control)
                {
                    e.Effect = DragDropEffects.Copy;
                }
                if (result.RegionInfo == _FirstCollisionResult.RegionInfo)
                {
                    e.Effect = DragDropEffects.None;
                }
                else
                {
                    IRegionInfo regionInfo = result.RegionInfo;
                    if (regionInfo.Region == null)
                    {
                        OwnerCtrl._DrawDropRegion = regionInfo;
                    }
                    else if (regionInfo is VelocityRegionInfo)
                    {
                        VelocityRegionInfo velInfo = (VelocityRegionInfo)regionInfo;
                        if (_FirstCollisionResult.Type == "VelocityRegion")
                        {
                            OwnerCtrl._DrawDropRegion = regionInfo;
                        }
                        else if (_FirstCollisionResult.Type == "KeyRegion")
                        {
                            if (velInfo.KeyRegionInfo != _FirstCollisionResult.RegionInfo)
                            {
                                OwnerCtrl._DrawDropRegion =
                                    ((VelocityRegionInfo)regionInfo).KeyRegionInfo;
                            }
                            else
                            {
                                e.Effect = DragDropEffects.None;
                            }
                        }
                    }
                    else if (regionInfo is KeyRegionInfo)
                    {
                        if (_FirstCollisionResult.Type == "VelocityRegion")
                        {
                            VelocityRegionInfo velInfo =
                                (VelocityRegionInfo)(_FirstCollisionResult.RegionInfo);
                            if (velInfo.KeyRegionInfo == regionInfo)
                            {
                                e.Effect = DragDropEffects.None;
                            }
                            else if (regionInfo.Region.Target.Children.Count == 1)
                            {
                                OwnerCtrl._DrawDropRegion = regionInfo;
                            }
                            else
                            {
                                e.Effect = DragDropEffects.None;
                            }
                        }
                        else
                        {
                            OwnerCtrl._DrawDropRegion = regionInfo;
                        }
                    }
                }
                if (OwnerCtrl.toolTip.Visible == true)
                {
                    OwnerCtrl.toolTip.Hide();
                    OwnerCtrl.Refresh();
                }
                OwnerCtrl.Invalidate();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragLeave(System.EventArgs e)
            {
                Debug.WriteLine("OnDragLeave");
                if (
                    _FirstCollisionResult.RegionInfo == OwnerCtrl._DrawDropRegion)
                {
                    _FirstMouseDownPoint = new Point(-1, -1);
                    _FirstCollisionResult = null;
                    OwnerCtrl.ChangeOperatorToExchange();
                }

                OwnerCtrl._DrawDropRegion = null;
                OwnerCtrl.Invalidate();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragDrop(DragEventArgs e)
            {
                Debug.WriteLine("OnDragDrop");
                if (OwnerCtrl._DrawDropRegion != null)
                {
                    if ((ModifierKeys & Keys.Control) == Keys.Control)
                    {
                        if (_FirstCollisionResult.RegionInfo is KeyRegionInfo)
                        {
                            if (
                                OwnerCtrl._DrawDropRegion.Region != null)
                            {
                                OwnerCtrl.ItemsSource.CopyKeyRegion
                                    ((IKeyRegion)(_FirstCollisionResult.RegionInfo.Region),
                                      (IKeyRegion)(OwnerCtrl._DrawDropRegion.Region));
                            }
                            else
                            {
                                OwnerCtrl.ItemsSource.CopyKeyRegion
                                    ((IKeyRegion)(_FirstCollisionResult.RegionInfo.Region),
                                      OwnerCtrl._DrawDropRegion.Minimum,
                                      OwnerCtrl._DrawDropRegion.Maximum);
                            }
                        }
                        else if (_FirstCollisionResult.RegionInfo is VelocityRegionInfo)
                        {
                            if (OwnerCtrl._DrawDropRegion.Region != null)
                            {
                                IVelocityRegion iVelRegion = null;
                                if (OwnerCtrl._DrawDropRegion.Region is IVelocityRegion)
                                {
                                    iVelRegion =
                                        (IVelocityRegion)(OwnerCtrl._DrawDropRegion.Region);
                                }
                                else
                                {
                                    iVelRegion =
                                        (IVelocityRegion)
                                        (((IKeyRegion)(OwnerCtrl._DrawDropRegion.Region)).
                                         Children[0]);
                                }
                                OwnerCtrl.ItemsSource.CopyVelocityRegion
                                    ((IVelocityRegion)(_FirstCollisionResult.RegionInfo.Region),
                                      iVelRegion);

                            }
                            else
                            {
                                OwnerCtrl.ItemsSource.CopyVelocityRegion
                                    ((IVelocityRegion)(_FirstCollisionResult.RegionInfo.Region),
                                      OwnerCtrl._DrawDropRegion.Minimum,
                                      OwnerCtrl._DrawDropRegion.Maximum);
                            }
                        }
                    }
                    else
                    {
                        if (_FirstCollisionResult.RegionInfo is KeyRegionInfo)
                        {
                            if (OwnerCtrl._DrawDropRegion.Region != null)
                            {
                                OwnerCtrl.ItemsSource.SwapKeyRegion
                                    ((IKeyRegion)(_FirstCollisionResult.RegionInfo.Region),
                                      (IKeyRegion)(OwnerCtrl._DrawDropRegion.Region));
                            }
                            else
                            {
                                OwnerCtrl.ItemsSource.ChangeKeyRegionValue
                                    ((IKeyRegion)(_FirstCollisionResult.RegionInfo.Region),
                                      OwnerCtrl._DrawDropRegion.Minimum,
                                      OwnerCtrl._DrawDropRegion.Maximum);
                            }
                        }
                        else if (_FirstCollisionResult.RegionInfo is VelocityRegionInfo)
                        {
                            if (OwnerCtrl._DrawDropRegion.Region != null)
                            {
                                IVelocityRegion iVelRegion = null;
                                if (OwnerCtrl._DrawDropRegion.Region is IVelocityRegion)
                                {
                                    iVelRegion =
                                        (IVelocityRegion)(OwnerCtrl._DrawDropRegion.Region);
                                }
                                else
                                {
                                    iVelRegion =
                                        (IVelocityRegion)
                                        (((IKeyRegion)(OwnerCtrl._DrawDropRegion.Region)).
                                         Children[0]);
                                }
                                OwnerCtrl.ItemsSource.SwapVelocityRegion
                                    ((IVelocityRegion)(_FirstCollisionResult.RegionInfo.Region),
                                      iVelRegion);
                            }
                            else
                            {
                                OwnerCtrl.ItemsSource.RemoveAndCreateVelocityRegion
                                    ((IVelocityRegion)(_FirstCollisionResult.RegionInfo.Region),
                                      OwnerCtrl._DrawDropRegion.Minimum,
                                      OwnerCtrl._DrawDropRegion.Maximum);
                            }
                        }
                    }
                }
                OwnerCtrl._DrawDropRegion = null;
                OwnerCtrl.ChangeOperatorToExchange();
                OwnerCtrl.Invalidate();
            }
        }

        //
        //


        ///------------------------------------------------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SelectRegion(int x, int y)
        {
            if (ItemsSource.Items != null &&
                 0 < ItemsSource.Items.Count)
            {
                SampleMapCollisionResult result = Collision(x, y);
                if (result.RegionInfo != null)
                {
                    if ((ModifierKeys & Keys.Control) == Keys.Control)
                    {
                        ToggleSelect(result.RegionInfo);
                    }
                    else
                    {
                        ClearSelect();
                        Selected(result.RegionInfo);
                    }
                }
                else
                {
                    ClearSelect();
                }
                OnSelectChanged(EventArgs.Empty);
            }
        }

        private bool IsSelected(IRegionInfo info)
        {
            return _SelectedRegionInfo.IsSelected(info);
        }

        private void Selected(IRegionInfo info)
        {
            Debug.Assert(info != null, "selected region is null");

            if (info.Region != null)
            {
                _SelectedRegionInfo.Select(info);
                if (info is KeyRegionInfo)
                {
                    KeyRegionInfo kInfo = (KeyRegionInfo)info;
                    foreach (VelocityRegionInfo vInfo in kInfo.VelocityRegionInfos)
                    {
                        _SelectedRegionInfo.Select(vInfo);
                    }
                }
            }
        }

        private void ToggleSelect(IRegionInfo info)
        {
            Debug.Assert(info != null, "selected region is null");

            if (info.Region != null)
            {
                if (info is VelocityRegionInfo)
                {
                    VelocityRegionInfo vInfo = (VelocityRegionInfo)info;
                    if (IsSelected(vInfo.KeyRegionInfo) == true)
                    {
                        return;
                    }
                }
                if (IsSelected(info) == true)
                {
                    _SelectedRegionInfo.UnSelect(info);
                    if (info is KeyRegionInfo)
                    {
                        KeyRegionInfo kInfo = (KeyRegionInfo)info;
                        foreach (VelocityRegionInfo vInfo in kInfo.VelocityRegionInfos)
                        {
                            _SelectedRegionInfo.UnSelect(vInfo);
                        }
                    }
                }
                else
                {
                    _SelectedRegionInfo.Select(info);
                    if (info is KeyRegionInfo)
                    {
                        KeyRegionInfo kInfo = (KeyRegionInfo)info;
                        foreach (VelocityRegionInfo vInfo in kInfo.VelocityRegionInfos)
                        {
                            _SelectedRegionInfo.Select(vInfo);
                        }
                    }
                }
            }
        }

        private void ClearSelect()
        {
            _SelectedRegionInfo.Clear();
        }

        private void UpdateSelected(KeyRegionInfo[] infos)
        {
            SelectedRegionInfo newSelect = new SelectedRegionInfo();

            if (_SelectedRegionInfo != null)
            {

                foreach (KeyRegionInfo kInfo in infos)
                {
                    if (_SelectedRegionInfo.Contains(kInfo.Region) == true)
                    {
                        newSelect.Select(kInfo);
                        foreach (VelocityRegionInfo vInfo in kInfo.VelocityRegionInfos)
                        {
                            newSelect.Select(vInfo);
                        }
                    }
                    else if (kInfo.VelocityRegionInfos != null)
                    {
                        foreach (VelocityRegionInfo vInfo in kInfo.VelocityRegionInfos)
                        {
                            if (_SelectedRegionInfo.Contains(vInfo.Region) == true)
                            {
                                newSelect.Select(vInfo);
                            }
                        }
                    }
                }
            }

            _SelectedRegionInfo = newSelect;
        }

        //

        private Cursor GetCursor(string type)
        {
            Cursor cursor = Cursors.Default;
            switch (type)
            {
                case "KeyRegionLeftSplitter":
                case "KeyRegionRightSplitter":
                    cursor = Cursors.VSplit;
                    break;

                case "VelocityRegionTopSplitter":
                case "VelocityRegionBottomSplitter":
                    cursor = Cursors.HSplit;
                    break;
            }

            return cursor;
        }

        private KeyRegionInfo[] VSplitKeyRegions(string type, IRegionInfo info)
        {
            IRegionInfo[] infos = new KeyRegionInfo[2];

            switch (type)
            {
                case "KeyRegionLeftSplitter":
                    infos[0] = info.Prev;
                    infos[1] = info;
                    break;

                case "KeyRegionRightSplitter":
                    infos[0] = info;
                    infos[1] = info.Next;
                    break;
            }

            return (KeyRegionInfo[])infos;
        }

        private int VSplitPosition(int position, string type, IRegionInfo info)
        {
            int min = 0;
            int max = 127;
            IRegionInfo Prev = info.Prev;
            IRegionInfo Next = info.Next;

            switch (type)
            {
                case "KeyRegionLeftSplitter":
                    if (Prev != null)
                    {
                        if (Prev.Region == null)
                        {
                            min = Prev.Minimum;
                        }
                        else
                        {
                            min = Prev.Minimum + 1;
                        }
                    }
                    if (info.Region == null)
                    {
                        max = info.Maximum + 1;
                    }
                    else
                    {
                        max = info.Maximum;
                    }
                    break;

                case "KeyRegionRightSplitter":
                    if (Next != null)
                    {
                        if (Next.Region == null)
                        {
                            max = Next.Maximum + 1;
                        }
                        else
                        {
                            max = Next.Maximum;
                        }
                    }
                    if (info.Region == null)
                    {
                        min = info.Minimum;
                    }
                    else
                    {
                        min = info.Minimum + 1;
                    }
                    break;
            }

            int key = GetKeyByPosition(position);
            if (key < min)
            {
                key = min;
            }
            if (max < key)
            {
                key = max;
            }
            position = GetPositionByKey(key);

            return position;
        }


        private VelocityRegionInfo[] HSplitVelocityRegions(string type, IRegionInfo info)
        {
            IRegionInfo[] infos = new VelocityRegionInfo[2];

            switch (type)
            {
                case "VelocityRegionTopSplitter":
                    infos[0] = info;
                    infos[1] = info.Next;
                    break;

                case "VelocityRegionBottomSplitter":
                    infos[0] = info.Prev;
                    infos[1] = info;
                    break;
            }

            return (VelocityRegionInfo[])infos;
        }

        private int HSplitPosition(int position, string type, IRegionInfo info)
        {
            int min = 0;
            int max = 0;

            switch (type)
            {
                case "VelocityRegionBottomSplitter":
                    if (info.Prev == null)
                    {
                        min = 0;
                    }
                    else
                    {
                        min = info.Prev.Minimum;
                    }
                    max = info.Maximum;
                    break;

                case "VelocityRegionTopSplitter":
                    if (info.Next == null)
                    {
                        max = VelocityStage - 1;
                    }
                    else
                    {
                        max = info.Next.Maximum;
                    }
                    min = info.Minimum;
                    break;
            }

            int vel = GetVelocityByPosition(position);
            if (vel <= min)
            {
                vel = min;
            }
            else if (max <= vel)
            {
                vel = max;
            }

            position = GetPositionByVelocity(vel);

            return position;
        }

        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(SampleMapCtrl));
            this._ContextMenuStrip = new System.Windows.Forms.ContextMenuStrip(this.components);
            this._MenuItem_ChangeFilePath = new System.Windows.Forms.ToolStripMenuItem();
            this._MenuItem_DivideKeyRegion = new System.Windows.Forms.ToolStripMenuItem();
            this._MenuItem_DivideVelocityRegion = new System.Windows.Forms.ToolStripMenuItem();
            this._MenuItem_Remove = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator1 = new System.Windows.Forms.ToolStripSeparator();
            this._MenuItem_OpenItemFolder = new System.Windows.Forms.ToolStripMenuItem();
            this._MenuItem_ExecuteItemFolder = new System.Windows.Forms.ToolStripMenuItem();
            this._ToolStripSeparator = new System.Windows.Forms.ToolStripSeparator();
            this._ContextMenuStrip.SuspendLayout();
            this.SuspendLayout();
            //
            // _ContextMenuStrip
            //
            this._ContextMenuStrip.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
                this._MenuItem_ChangeFilePath,
                    this._MenuItem_DivideKeyRegion,
                    this._MenuItem_DivideVelocityRegion,
                    this._MenuItem_Remove,
                    this.toolStripSeparator1,
                    this._MenuItem_OpenItemFolder,
                    this._MenuItem_ExecuteItemFolder});
            this._ContextMenuStrip.Name = "SampleMapContextMenuStrip";
            resources.ApplyResources(this._ContextMenuStrip, "_ContextMenuStrip");
            this._ContextMenuStrip.Opening += new System.ComponentModel.CancelEventHandler(this.OnOpeningContextMenu);
            //
            // _MenuItem_ChangeFilePath
            //
            this._MenuItem_ChangeFilePath.Name = "_MenuItem_ChangeFilePath";
            resources.ApplyResources(this._MenuItem_ChangeFilePath, "_MenuItem_ChangeFilePath");
            this._MenuItem_ChangeFilePath.Click += new System.EventHandler(this.OnChangeFilePath);
            //
            // _MenuItem_DivideKeyRegion
            //
            this._MenuItem_DivideKeyRegion.Name = "_MenuItem_DivideKeyRegion";
            resources.ApplyResources(this._MenuItem_DivideKeyRegion, "_MenuItem_DivideKeyRegion");
            this._MenuItem_DivideKeyRegion.Click += new System.EventHandler(this.OnDivideKeyRegion);
            //
            // _MenuItem_DivideVelocityRegion
            //
            this._MenuItem_DivideVelocityRegion.Name = "_MenuItem_DivideVelocityRegion";
            resources.ApplyResources(this._MenuItem_DivideVelocityRegion, "_MenuItem_DivideVelocityRegion");
            this._MenuItem_DivideVelocityRegion.Click += new System.EventHandler(this.OnDivideVelocityRegion);
            //
            // _MenuItem_Remove
            //
            resources.ApplyResources(this._MenuItem_Remove, "_MenuItem_Remove");
            this._MenuItem_Remove.Name = "_MenuItem_Remove";
            this._MenuItem_Remove.Click += new System.EventHandler(this.OnRemove);
            //
            // toolStripSeparator1
            //
            this.toolStripSeparator1.Name = "toolStripSeparator1";
            resources.ApplyResources(this.toolStripSeparator1, "toolStripSeparator1");
            //
            // _MenuItem_OpenItemFolder
            //
            resources.ApplyResources(this._MenuItem_OpenItemFolder, "_MenuItem_OpenItemFolder");
            this._MenuItem_OpenItemFolder.Name = "_MenuItem_OpenItemFolder";
            this._MenuItem_OpenItemFolder.Click += new System.EventHandler(this.OnOpenItemFolder);
            //
            // _MenuItem_ExecuteItemFolder
            //
            this._MenuItem_ExecuteItemFolder.Name = "_MenuItem_ExecuteItemFolder";
            resources.ApplyResources(this._MenuItem_ExecuteItemFolder, "_MenuItem_ExecuteItemFolder");
            this._MenuItem_ExecuteItemFolder.Click += new System.EventHandler(this.OnExecuteItemFolder);
            //
            // _ToolStripSeparator
            //
            this._ToolStripSeparator.Name = "_ToolStripSeparator";
            resources.ApplyResources(this._ToolStripSeparator, "_ToolStripSeparator");
            //
            // SampleMapCtrl
            //
            this.Name = "SampleMapCtrl";
            this.Leave += new System.EventHandler(this.OnLeave);
            this.KeyUp += new System.Windows.Forms.KeyEventHandler(this.OnKeyUp);
            this._ContextMenuStrip.ResumeLayout(false);
            this.ResumeLayout(false);

        }

        private void OnOpeningContextMenu(object sender, System.ComponentModel.CancelEventArgs e)
        {
            IVelocityRegion[] velRegions = SelectedVelocityRegions;

            _MenuItem_OpenItemFolder.Enabled = velRegions.Length > 0 ? true : false;
            _MenuItem_ExecuteItemFolder.Enabled = velRegions.Length == 1 ? true : false;
        }

        private void OnChangeFilePath(object select, EventArgs e)
        {
            ChangeFilePath();
            UpdateInfos();
            Invalidate();
        }

        private void OnDivideKeyRegion(object select, EventArgs e)
        {
            DivideKeyRegions();
            UpdateInfos();
            Invalidate();
        }

        private void OnDivideVelocityRegion(object select, EventArgs e)
        {
            DivideVelocityRegions();
            UpdateInfos();
            Invalidate();
        }

        private void OnRemove(object sender, EventArgs e)
        {
            RemoveRegions();
            UpdateInfos();
            Invalidate();
            OnSelectChanged(EventArgs.Empty);
        }

        private void OnOpenItemFolder(object sender, EventArgs e)
        {
            foreach (IVelocityRegion velRegion in SelectedVelocityRegions)
            {
                Process.Start("explorer.exe", "/e,/select, " + velRegion.FilePath);
            }
        }

        private void OnExecuteItemFolder(object sender, EventArgs e)
        {
            ContextMenuHandler?.ExecuteItemFolder(this);
        }

        private void OnSelectChanged(EventArgs e)
        {
            if (SelectChanged != null)
            {
                SelectChanged(this, e);
            }
            Invalidate();
        }

        private void OnItemsSourceChanged(EventArgs e)
        {
            if (ItemsSourceChanged != null)
            {
                ItemsSourceChanged(this, e);
            }
            Invalidate();
        }

        private void ChangeFilePath()
        {
            if (_SelectedRegionInfo.Count == 1 && _SelectedRegionInfo[0] is VelocityRegionInfo)
            {
                ChangeVelocityRegionFilePath(_SelectedRegionInfo[0]);
            }
        }

        private void DivideKeyRegions()
        {
            List<IRegion> regionList = new List<IRegion>();

            foreach (IRegionInfo info in _SelectedRegionInfo)
            {
                if (info is KeyRegionInfo == true)
                {
                    if (regionList.Contains(info.Region) == false)
                    {
                        regionList.Add(info.Region);
                    }
                }
                else if (info is VelocityRegionInfo == true)
                {
                    VelocityRegionInfo vInfo = (VelocityRegionInfo)info;
                    IVelocityRegion iVelRegion = (IVelocityRegion)(vInfo.Region);
                    if (iVelRegion.Parent.Children.Count == 1 &&
                         regionList.Contains(iVelRegion.Parent) == false)
                    {
                        regionList.Add(iVelRegion.Parent);
                    }
                }
            }

            foreach (IRegion region in regionList)
            {
                _ItemsSource.DivideKeyRegion((IKeyRegion)region);
            }
        }

        private void DivideVelocityRegions()
        {
            foreach (IRegionInfo info in _SelectedRegionInfo)
            {
                if (info is VelocityRegionInfo == true)
                {
                    VelocityRegionInfo vInfo = (VelocityRegionInfo)info;
                    _ItemsSource.DivideVelocityRegion((IVelocityRegion)vInfo.Region);
                }
            }
        }

        private void RemoveRegions()
        {
            try
            {
                _ItemsSource.BeginTransaction();
                foreach (IRegionInfo info in _SelectedRegionInfo)
                {
                    if (info is KeyRegionInfo == true)
                    {
                        KeyRegionInfo kInfo = (KeyRegionInfo)info;
                        _ItemsSource.Items.Remove((IKeyRegion)kInfo.Region);
                    }
                    else
                    {
                        VelocityRegionInfo vInfo = (VelocityRegionInfo)info;
                        KeyRegionInfo kInfo = vInfo.KeyRegionInfo;
                        if (kInfo.VelocityRegionInfos.Length == 1)
                        {
                            _ItemsSource.Items.Remove((IKeyRegion)kInfo.Region);
                        }
                        else
                        {
                            if (vInfo.Prev == null)
                            {
                                vInfo.Next.Region.MinimumIntact = vInfo.Region.Minimum;
                            }
                            else
                            {
                                vInfo.Prev.Region.MaximumIntact = vInfo.Region.Maximum;
                            }
                            ((IKeyRegion)kInfo.Region).Children.
                                Remove((IVelocityRegion)vInfo.Region);
                        }
                    }
                }
                _ItemsSource.EndTransaction();
            }
            catch
            {
                _ItemsSource.CancelTransaction();
            }
        }

        private void CreateKeyRegion(int index, int minimum, int maximum, string filePath)
        {
            IKeyRegion kRegion = _ItemsSource.CreateKeyRegion(index, minimum, maximum, filePath);
        }

        private void CreateKeyRegion(IRegionInfo info)
        {
            KeyRegionInfo kInfo = (KeyRegionInfo)info;
            string filePath = this.OnGetFilePath(string.Empty);
            if (string.IsNullOrEmpty(filePath) == true)
            {
                return;
            }
            if (kInfo == null)
            {
                CreateKeyRegion(0, 0, Widths.Length - 1, filePath);
            }
            else
            {
                int index = 0;
                KeyRegionInfo ki = kInfo;
                while (ki != null)
                {
                    if (ki.Region != null) ++index;
                    ki = (KeyRegionInfo)(ki.Prev);
                }
                CreateKeyRegion(index, kInfo.Minimum, kInfo.Maximum, filePath);
            }
        }

        private void ChangeVelocityRegionFilePath(IRegionInfo info)
        {
            VelocityRegionInfo velInfo = (VelocityRegionInfo)info;
            string filePath = ((IVelocityRegion)(velInfo.Region)).FilePath;

            string newFilePath = this.OnGetFilePath(filePath);
            if (string.IsNullOrEmpty(newFilePath) == true)
            {
                return;
            }
            if (filePath != newFilePath)
            {
                ((IVelocityRegion)(velInfo.Region)).FilePath = newFilePath;
                VelocityRegion vel =
                    (VelocityRegion)(((ComponentVelocityRegion)(velInfo.Region)).Target);
                try
                {
                    using (WaveFileReader reader = WaveFileReader.CreateInstance(newFilePath))
                    {
                        vel.WaveFile = reader.Open(vel.FilePath);
                    }
                }
                catch { }
            }
        }

        //

        private class SelectedRegionInfo : List<IRegionInfo>
        {
            public bool ContainsKeyRegionInfo
            {
                get
                {
                    bool ret = false;

                    foreach (IRegionInfo info in this)
                    {
                        if (info is KeyRegionInfo)
                        {
                            ret = true;
                        }
                    }

                    return ret;
                }
            }

            public bool ContainsVelocityRegionInfo
            {
                get
                {
                    bool ret = false;

                    foreach (IRegionInfo info in this)
                    {
                        if (info is VelocityRegionInfo)
                        {
                            if (Contains(((VelocityRegionInfo)info).KeyRegionInfo) == false)
                            {
                                ret = true;
                                break;
                            }
                        }
                    }

                    return ret;
                }
            }

            public IKeyRegion[] SelectedKeyRegions
            {
                get
                {
                    List<IKeyRegion> keyRegions = new List<IKeyRegion>();

                    foreach (IRegionInfo info in this)
                    {
                        if (info.Region is IKeyRegion)
                        {
                            keyRegions.Add((IKeyRegion)info.Region);
                        }
                    }

                    return keyRegions.ToArray();
                }
            }

            public IVelocityRegion[] SelectedVelocityRegions
            {
                get
                {
                    List<IVelocityRegion> velRegions = new List<IVelocityRegion>();

                    foreach (IRegionInfo info in this)
                    {
                        if (info.Region is IVelocityRegion)
                        {
                            velRegions.Add((IVelocityRegion)info.Region);
                        }
                    }

                    return velRegions.ToArray();
                }
            }

            public void Select(IRegionInfo region)
            {
                if (region != null)
                {
                    if (Contains(region) == false)
                    {
                        Add(region);
                    }
                }
            }

            public void UnSelect(IRegionInfo region)
            {
                if (region != null)
                {
                    while (Contains(region) == true)
                    {
                        Remove(region);
                    }
                }
            }

            public bool IsSelected(IRegionInfo region)
            {
                return Contains(region);
            }

            public new bool Contains(IRegionInfo region)
            {
                bool ret = false;

                foreach (IRegionInfo info in this)
                {
                    if (info.Region == region.Region)
                    {
                        ret = true;
                        break;
                    }
                }

                return ret;
            }

            public bool Contains(IRegion region)
            {
                bool ret = false;

                if (region != null)
                {
                    foreach (IRegionInfo info in this)
                    {
                        if (info != null && info.Region != null &&
                            info.Region.Target == region.Target)
                        {
                            ret = true;
                            break;
                        }
                    }
                }

                return ret;
            }
        }

        private void OnLeave(object sender, EventArgs e)
        {
            Invalidate();
        }

        private void OnKeyUp(object sender, KeyEventArgs e)
        {
            Keys key = e.KeyCode;

            if (key == Keys.Apps)
            {
                MouseEventArgs mouseEventArgs =
                    new MouseEventArgs(MouseButtons.None, 0, Width / 2, Height / 2, 0);
                OnTrackContextMenu(mouseEventArgs);
            }
        }

        private bool OnDropped(DroppedEventArgs e)
        {
            bool returnValue = true;

            if (Dropped != null)
            {
                Dropped(this, e);
                returnValue = !(e.Cancel);
            }

            return returnValue;
        }

        private class SampleMapCrtlToolTip : UserControl
        {
            private SampleMapCtrl Controller;
            public SampleMapCrtlToolTip(SampleMapCtrl control)
            {
                this.Controller = control;
                this.Parent = control;
                this.Bounds = new Rectangle(0, 0, 26, 13);
                this.Hide();
            }
            private int _Value = 0;
            public int Value
            {
                get
                {
                    return _Value;
                }
                set
                {
                    if (value < 0)
                    {
                        _Value = 0;
                    }
                    else if (value > 127)
                    {
                        _Value = 127;
                    }
                    else
                    {
                        _Value = value;
                    }
                }
            }
            public void SetLocation(int x, int y)
            {
                int lx = x + 10;
                int ly = y + 5;
                Size size = Parent.Size;
                if (x < -11)
                {
                    lx = 0;
                }
                if (y < -5 + Controller._KeyRegionVisibleLength)
                {
                    ly = Controller._KeyRegionVisibleLength;
                }
                if (y + 5 + 13 > size.Height)
                {
                    ly = size.Height - 13;
                }
                if (x + 10 + 26 > size.Width)
                {
                    lx = size.Width - 26;
                }
                Location = new Point(lx, ly);
            }
            protected override void OnPaint(PaintEventArgs pe)
            {
                // Call the OnPaint method of the base class.
                base.OnPaint(pe);

                Pen myPen = new Pen(Color.Black);
                Size size = ClientSize;
                Rectangle rect = new Rectangle(0, 0, size.Width - 1, size.Height - 1);
                pe.Graphics.DrawRectangle(myPen, rect);
                SolidBrush mySolidBrush = new SolidBrush(SystemColors.Info);
                pe.Graphics.FillRectangle(mySolidBrush, 1, 1, size.Width - 2, size.Height - 2);
                StringFormat format = new StringFormat(StringFormatFlags.DirectionRightToLeft);
                mySolidBrush.Color = SystemColors.InfoText;
                rect.Inflate(-1, -1);
                RectangleF rectF = new RectangleF(rect.Location, rect.Size);
                pe.Graphics.DrawString(Value.ToString(), Font, mySolidBrush, rectF, format);
            }
        }
    }

    ///------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class SampleMapCollisionResult
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public SampleMapCollisionResult()
        {
            Type = "None";
        }
        public string Type { get; set; }
        public int VelocitySplitterValue { get; set; }
        public IRegionInfo RegionInfo { get; set; }
    }

    ///インターフェース
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IKeyRegionDrawer
    {
        void Draw(SampleMapDrawDescriptor drawDesc);
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IVelocityRegionDrawer
    {
        void Draw(SampleMapDrawDescriptor drawDesc);
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IRegionInfo
    {
        IRegion Region { get; }
        int Minimum { get; }
        int Maximum { get; }
        Rectangle Bounds { get; }
        IRegionInfo Prev { get; }
        IRegionInfo Next { get; }
    }

    ///-----------------------------------------------------------------------------
    ///
    public class KeyRegionInfo : IRegionInfo
    {
        private IRegionInfo _Prev;
        private IRegionInfo _Next;

        public VelocityRegionInfo[] VelocityRegionInfos { get; set; }

        public IRegion Region { get; set; }
        public int Minimum { get; set; }
        public int Maximum { get; set; }
        public Rectangle Bounds { get; set; }
        public IRegionInfo Prev
        {
            get
            {
                return _Prev;
            }
            set
            {
                _Prev = value;
            }
        }
        public IRegionInfo Next
        {
            get
            {
                return _Next;
            }
            set
            {
                _Next = value;
            }
        }
    }

    ///-----------------------------------------------------------------------------
    ///
    public class VelocityRegionInfo : IRegionInfo
    {
        private IRegionInfo _Prev;
        private IRegionInfo _Next;
        public KeyRegionInfo KeyRegionInfo { get; set; }

        public IRegion Region { get; set; }
        public int Minimum { get; set; }
        public int Maximum { get; set; }
        public Rectangle Bounds { get; set; }
        public string FilePath { get; set; }

        public IRegionInfo Prev
        {
            get
            {
                return _Prev;
            }
            set
            {
                _Prev = value;
            }
        }
        public IRegionInfo Next
        {
            get
            {
                return _Next;
            }
            set
            {
                _Next = value;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class SampleMapDrawDescriptor
    {
        private Graphics _Gc = null;
        private Font _Font = null;

        /// <summary>
        ///
        /// </summary>
        public SampleMapDrawDescriptor(Graphics gc, Font font)
        {
            this._Gc = gc;
            this._Font = font;
        }

        /// <summary>
        ///
        /// </summary>
        public Graphics Gc
        {
            get
            {
                return this._Gc;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public Font Font
        {
            get
            {
                return this._Font;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int Height { get; set; }

        /// <summary>
        ///
        /// </summary>
        public Rectangle Bounds { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool Focused { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool Enabled { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool Selected { get; set; }

        /// <summary>
        ///
        /// </summary>
        public string Text { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool Modify { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int VSplitPosition { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int HSplitPosition { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool Drop { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool Blank { get; set; }

        /// <summary>
        ///
        /// </summary>
        public object UserData { get; set; }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class KeyRegionDrawer : IKeyRegionDrawer
    {
        public void Draw(SampleMapDrawDescriptor drawDesc)
        {
            Rectangle bounds = drawDesc.Bounds;

            SolidBrush brush = new SolidBrush(Color.FromArgb(255, 0, 255, 255));

            SolidBrush grayBrush = new SolidBrush(Color.FromArgb
                                                   (80,
                                                     (SystemColors.WindowText.R +
                                                       SystemColors.HighlightText.R) / 2,
                                                     (SystemColors.WindowText.G +
                                                       SystemColors.HighlightText.G) / 2,
                                                     (SystemColors.WindowText.B +
                                                       SystemColors.HighlightText.B) / 2));

            SolidBrush darkGrayBrush =
                new SolidBrush(Color.FromArgb
                                ((SystemColors.ControlDark.R * 160 / 255) +
                                  (Color.White.R * (255 - 160) / 255),
                                  (SystemColors.ControlDark.G * 160 / 255) +
                                  (Color.White.G * (255 - 160) / 255),
                                  (SystemColors.ControlDark.B * 160 / 255) +
                                  (Color.White.B * (255 - 160) / 255)));

            if (drawDesc.Selected == true)
            {
                if (drawDesc.Focused == true)
                {
                    brush = (SolidBrush)SystemBrushes.Highlight;
                }
                else
                {
                    brush = darkGrayBrush;
                }
            }

            if (drawDesc.Blank == false)
            {
                drawDesc.Gc.FillRectangle(brush,
                                           bounds.X, bounds.Y,
                                           bounds.Width, bounds.Height);
            }

            if (drawDesc.Modify == true || drawDesc.Drop == true)
            {
                brush = grayBrush;
            }

            if (drawDesc.Blank == false || drawDesc.Drop == true)
            {
                drawDesc.Gc.FillRectangle(brush,
                                           bounds.X, bounds.Y,
                                           bounds.Width, bounds.Height);
            }
            if (drawDesc.Blank == false)
            {
                drawDesc.Gc.DrawRectangle(new Pen(Color.Black),
                                           bounds.X, bounds.Y,
                                           bounds.Width - 1, bounds.Height - 1);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class VelocityRegionDrawer : IVelocityRegionDrawer
    {
        public void Draw(SampleMapDrawDescriptor drawDesc)
        {
            Rectangle bounds = drawDesc.Bounds;
            SolidBrush brush = new SolidBrush(SystemColors.Window);

            SolidBrush grayBrush = new SolidBrush(Color.FromArgb
                                                   (80,
                                                     (SystemColors.WindowText.R +
                                                       SystemColors.HighlightText.R) / 2,
                                                     (SystemColors.WindowText.G +
                                                       SystemColors.HighlightText.G) / 2,
                                                     (SystemColors.WindowText.B +
                                                       SystemColors.HighlightText.B) / 2));

            SolidBrush darkGrayBrush =
                new SolidBrush(Color.FromArgb
                                ((SystemColors.ControlDark.R * 160 / 255) +
                                  (Color.White.R * (255 - 160) / 255),
                                  (SystemColors.ControlDark.G * 160 / 255) +
                                  (Color.White.G * (255 - 160) / 255),
                                  (SystemColors.ControlDark.B * 160 / 255) +
                                  (Color.White.B * (255 - 160) / 255)));

            if (drawDesc.Selected == true)
            {
                if (drawDesc.Focused == true)
                {
                    brush = (SolidBrush)SystemBrushes.Highlight;
                }
                else
                {
                    brush = darkGrayBrush;
                }
            }

            if (drawDesc.Blank == false)
            {
                drawDesc.Gc.FillRectangle(brush,
                                           bounds.X, bounds.Y,
                                           bounds.Width, bounds.Height);
            }

            if (drawDesc.Modify == true || drawDesc.Drop == true)
            {
                brush = grayBrush;
            }

            if (drawDesc.Blank == false || drawDesc.Drop == true)
            {
                drawDesc.Gc.FillRectangle(brush,
                                           bounds.X, bounds.Y,
                                           bounds.Width, bounds.Height);
            }
            if (drawDesc.Blank == false)
            {
                drawDesc.Gc.DrawString(drawDesc.Text, drawDesc.Font,
                                       (drawDesc.Selected ?
                                         SystemBrushes.HighlightText : SystemBrushes.WindowText),
                                       bounds);
                drawDesc.Gc.DrawRectangle(new Pen(Color.Black),
                                           bounds.X, bounds.Y,
                                           bounds.Width - 1, bounds.Height - 1);
            }
        }
    }
}
