﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Drawing.Drawing2D;
    using System.Drawing.Imaging;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IInplaceOwner
    {
        bool MoveCaret(CaretMove direction);
        void EndEdit();
        bool Focus();
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class InplaceEditorDictionary : Dictionary<string, InplaceEditor>
    {
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IInplaceEditorArgument
    {
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public abstract class InplaceEditor
    {
        ///--------------------------------
        /// <summary>
        /// インプレイス編集用コントロール
        /// </summary>
        public class InplaceTextBox : TextBox
        {
            protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
            {
                // 親ウィンドウによるショートカット処理を無効化する
                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// インプレイス編集用コントロール
        /// </summary>
        public class InplaceComboBox : ComboBox
        {
            protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
            {
                // 親ウィンドウによるショートカット処理を無効化する
                return false;
            }
        }

        ///--------------------------------
        private bool _Permanent = true;
        private bool editing = false;

        //private IInplaceOwner   _Owner = null;
        //private Control         _OwnerCtrl = null;

        //private bool            _Cancel = false;
        //private bool            _RetryEditIfFailure = false;
        //private bool            _BeginByEditBeginKey = false;


        public event EventHandler BeginEditing;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public InplaceEditor(bool permanent)
        {
            _Permanent = permanent;
            RequireInitialValue = true;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        private InplaceEditor()
        {
        }

        ///--------------------------------
        /// <summary>
        /// これはイベント方式に変更予定
        /// </summary>
#if true
        public IInplaceOwner Owner { get; set; }
#else
        //こんな感じ
        public event EventHandler Completed;
        public event EventHandelr CaretMoveRequested;
#endif

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Control OwnerCtrl { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IParameterValue ParameterManipulator { get; set; }

        ///--------------------------------
        /// <summary>
        /// 作成したコントロールの永続性の取得
        /// </summary>
        public bool Permanent
        {
            get { return _Permanent; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual bool CanImmediateEdit
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual IParameterValue InitialValue { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IInplaceEditorArgument Arguments { get; set; }

        ///--------------------------------
        /// <summary>
        /// 編集開始キーによって編集が開始されたのか？
        /// </summary>
        public bool BeginByEditBeginKey { get; set; }

        ///--------------------------------
        /// <summary>
        /// どのキーで編集開始になったか。
        /// </summary>
        public Keys ImmediateEditKey { get; set; }

        /// <summary>
        /// Enterキーによって編集が終了した時にカレットを移動させるのか？
        /// </summary>
        public bool MoveNextLineCaretAfterEndEditByEnter { get; set; }

        ///--------------------------------
        /// <summary>
        /// 入力に失敗した時に再入力を行うのか？
        /// </summary>
        public virtual bool RetryEditIfFailure { get; set; }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// 編集前と編集後が同じ場合などに、falseを返して、編集をキャンセルする
        /// ことができる。そうすることにより、変更になっていないのに Undo対象に
        /// なることを避けることができる。
        /// </summary>
        public virtual bool Modified
        {
            get { return false; }
        }

        /// <summary>
        ///
        /// </summary>
        public string TargetName { get; set; }

        /// <summary>
        ///
        /// </summary>
        public IListItem Item { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool RequireInitialValue { get; set; }

        ///--------------------------------
        /// <summary>
        /// 後片付け
        /// </summary>
        public virtual void Dispose()
        {
        }

        ///--------------------------------
        /// <summary>
        /// 編集の開始
        /// </summary>
        public bool BeginEdit(Rectangle rect, string text, bool readOnly)
        {
            this.Disposing = false;
            if (readOnly == true)
            {
                return false;
            }

            if (this.BeginEditInternal(rect, text, readOnly) == true)
            {
                this.editing = true;
                return true;
            }

            return false;
        }

        ///--------------------------------
        /// <summary>
        /// 編集の開始
        /// </summary>
        protected abstract bool BeginEditInternal(Rectangle rect, string text, bool readOnly);

        /// <summary>
        /// 編集の終了
        /// </summary>
        public bool EndEdit()
        {
            if (this.Disposing == true || this.editing == false)
            {
                return false;
            }

            this.editing = false;

            if (ParameterManipulator != null)
            {
                try
                {
                    var value = GetValue();
                    if (value != null)
                    {
                        var validationResult = ParameterManipulator.ValidateValue(value);
                        if (validationResult.IsValid == true)
                        {
                            if (Modified == true)
                            {
                                ParameterManipulator.Value = value;
                            }
                        }
                        else
                        {
                            if (Modified == true)
                            {
                                MessageBox.Show(validationResult.ToString());
                            }
                        }
                    }
                }
                catch (FormatException)
                {
                    MessageBox.Show(Resources.MessageResource.Message_InvalidInputValue);
                }
                catch (OverflowException)
                {
                    MessageBox.Show(Resources.MessageResource.Message_OverflowInputValue);
                }
                catch
                {
                }
            }

            ParameterManipulator = null;
            Owner.EndEdit();
            return true;
        }

        /// <summary>
        /// 編集のキャンセル
        /// </summary>
        public bool CancelEdit()
        {
            if (this.Disposing == true || this.editing == false)
            {
                return false;
            }

            this.editing = false;

            ParameterManipulator = null;
            Owner.EndEdit();
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集結果の取得
        /// </summary>
        public virtual object GetValue()
        {
            Debug.Assert(false, "GetValue was not done override");
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnMouseDown(object sender, MouseEventArgs e)
        {
            BeginByEditBeginKey = true;
            //EndEdit();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnLeave(object sender, EventArgs e)
        {
            EndEdit();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnPreviewKeyDown(object sender, PreviewKeyDownEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Tab:
                    EndEdit();
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnKeyDown(object sender, KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Enter:
                    //case Keys.Tab:
                    //EndEdit( Modified != false ? false : true);
                    if (EndEdit() == true)
                    {
                        OwnerCtrl.BeginInvoke(new MethodInvoker(delegate () { Owner.Focus(); }));

                        if (MoveNextLineCaretAfterEndEditByEnter != false)
                        {
                            Owner.MoveCaret(e.Shift != false ?
                                             CaretMove.Up :
                                             CaretMove.Down);
                        }
                    }
                    break;

                case Keys.Escape:
                    if (CancelEdit() == true)
                    {
                        OwnerCtrl.BeginInvoke(new MethodInvoker(delegate () { Owner.Focus(); }));
                    }
                    break;

                default:
                    return;
            }

            e.Handled = true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnKeyPress(object sender, KeyPressEventArgs e)
        {
            if (e.KeyChar == (int)Keys.Enter ||
                e.KeyChar == (int)Keys.Tab ||
                e.KeyChar == (int)Keys.Escape)
            {
                e.Handled = true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual bool CanCut
        {
            get { return false; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual bool CanCopy
        {
            get { return false; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual bool CanPaste
        {
            get { return false; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual bool CanDelete
        {
            get { return false; }
        }

        /// <summary>
        ///
        /// </summary>
        protected bool Disposing { get; set; }

        /// <summary>
        ///
        /// </summary>
        protected void ExecuteEventBeginEditing()
        {
            if (BeginEditing != null)
            {
                BeginEditing(this, EventArgs.Empty);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// テキスト、整数、少数の編集用の基本クラス
    /// </summary>
    public class InputInplaceEditor : InplaceEditor
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public InputInplaceEditor(bool permanent) : base(permanent)
        {
        }

        ///--------------------------------
        /// <summary>
        /// pppp
        /// </summary>
        protected override void OnKeyDown(object sender, KeyEventArgs e)
        {
            Keys key = e.KeyCode;

#if false
            //
            switch( key ) {
#if false  // カーソルキーによって編集が終了することはなくなりました。
            case Keys.Left:
            case Keys.Right:
            case Keys.Up:
            case Keys.Down:
                if( BeginByEditBeginKey != false ) {
                    return;
                }
                break;
#endif

#if false
            case Keys.Enter:
                if( BeginByEditBeginKey == false ) {
                    key = e.Shift != false ? Keys.Up : Keys.Down;
                } else {
                    key = Keys.Enter;
                }
                break;
#endif
            }
#endif

            //
            switch (key)
            {
                case Keys.Tab:
                    EndEdit(Keys.Tab,
                             (e.Shift != false ?
                               CaretMove.Left :
                               CaretMove.Right));
                    break;

#if false  // カーソルキーによって編集が終了することはなくなりました。
            case Keys.Left:
                EndEdit( Keys.Left, CaretMove.Left);
                break;

            case Keys.Right:
                EndEdit( Keys.Right, CaretMove.Right);
                break;

            case Keys.Up:
                EndEdit( Keys.Up, CaretMove.Up);
                break;

            case Keys.Down:
                EndEdit( Keys.Down, CaretMove.Down);
                break;
#endif

                case Keys.Enter:
#if false
                EndEdit();
#else
                    EndEdit(Keys.Enter,
                             (e.Shift != false ?
                               CaretMove.Up :
                               CaretMove.Down));
#endif
                    break;

                case Keys.Escape:
                    CancelEdit();
                    break;

                default:
                    return;
            }

            e.Handled = true;
        }

        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void EndEdit(Keys key, CaretMove direction)
        {
            if (key == Keys.Tab)
            {
                if (EndEdit() == true)
                {
                    Owner.MoveCaret(direction);
                }
                return;
            }

            if (key == Keys.Enter)
            {
                if (EndEdit() == true)
                {
                    if (MoveNextLineCaretAfterEndEditByEnter != false)
                    {
                        Owner.MoveCaret(direction);
                    }
                }
                return;
            }

            if (BeginByEditBeginKey == false)
            {
                if (EndEdit() == true)
                {
                    Owner.MoveCaret(direction);
                }
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// テキスト
    /// </summary>
    public class TextInplaceEditor : InputInplaceEditor
    {
        private TextBox _TextBox = null;
        private string _InitialText = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TextInplaceEditor() : base(true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            OwnerCtrl.Controls.Remove(_TextBox);

            this.Disposing = true;
            _TextBox.Dispose();
            _TextBox = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            return _TextBox.Text;
        }

        ///--------------------------------
        /// <summary>
        /// 編集の開始
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            TextBox textBox = null;

            textBox = new InplaceTextBox();
            textBox.AcceptsTab = true;
            textBox.Multiline = true;
            textBox.Bounds = rect;
            textBox.Parent = OwnerCtrl;
            textBox.WordWrap = false;

            textBox.KeyDown += OnKeyDown;
            textBox.KeyPress += OnKeyPress;
            textBox.MouseDown += OnMouseDown;
            textBox.Leave += OnLeave;

            textBox.Show();
            textBox.Focus();

            //textBox.Capture = true;

            //
            _InitialText = InitialValue.Value.ToString();

            if (text == null)
            {
                textBox.Text = _InitialText;
                textBox.SelectAll();
            }
            else
            {

                textBox.Text = text;
                textBox.SelectionStart = text.Length;
                textBox.SelectionLength = 1;

                if (BeginByEditBeginKey != false)
                {
                    textBox.SelectAll();
                }
            }

            _TextBox = textBox;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                if (_TextBox == null) { return false; }
                return ControlText != _InitialText ? true : false;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected string ControlText
        {
            get { return _TextBox.Text; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 整数値
    /// </summary>
    public class IntegerInplaceEditor : InputInplaceEditor
    {
        private static int _InitialWheelValue = 1;

        private TextBox _TextBox = null;
        private int _WheelValue = _InitialWheelValue;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IntegerInplaceEditor() : base(true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            OwnerCtrl.Controls.Remove(_TextBox);

            this.Disposing = true;
            _TextBox.Dispose();
            _TextBox = null;
        }

        ///--------------------------------
        /// <summary>
        /// 編集結果の取得
        /// </summary>
        public override object GetValue()
        {
            return int.Parse(_TextBox.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            TextBox textBox = null;

            textBox = new InplaceTextBox();
            textBox.AcceptsTab = true;
            textBox.Multiline = true;
            textBox.Bounds = rect;
            textBox.Parent = OwnerCtrl;

            textBox.KeyDown += OnKeyDown;
            textBox.KeyPress += OnKeyPress;
            textBox.Leave += OnLeave;
            textBox.MouseDown += OnMouseDown;
            textBox.MouseWheel += OnMouseWheel;

            textBox.Show();
            textBox.Focus();

            //
            if (text == null)
            {
                text = InitialValue.ToString();
            }

            textBox.Text = text;
            textBox.SelectionStart = text.Length;
            textBox.SelectionLength = 1;

            if (BeginByEditBeginKey != false)
            {
                textBox.SelectAll();
            }

            _TextBox = textBox;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                int nowValue = 0;
                int newValue = 0;

                try
                {
                    nowValue = (int)InitialValue.Value;
                    newValue = int.Parse(_TextBox.Text);

                    if (newValue == nowValue)
                    {
                        return false;
                    }
                }
                catch { return false; }
                return true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnMouseWheel(object sender, MouseEventArgs e)
        {
            ValidationResult result = null;
            int relativeValue = e.Delta / 120;
            int value = 0;

            try
            {
                value = int.Parse(_TextBox.Text);
                value += (_WheelValue * relativeValue);

                result = InitialValue.ValidateValue(value);
                if (result.IsValid != false)
                {
                    _TextBox.Text = value.ToString();
                    _TextBox.SelectAll();
                }
            }
            catch { }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 整数値(ULong)
    /// </summary>
    public class ULongInplaceEditor : InputInplaceEditor
    {

        private static ulong _InitialWheelValue = 1;

        private TextBox _TextBox = null;
        private ulong _WheelValue = _InitialWheelValue;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ULongInplaceEditor() : base(true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            OwnerCtrl.Controls.Remove(_TextBox);

            this.Disposing = true;
            _TextBox.Dispose();
            _TextBox = null;
        }

        ///--------------------------------
        /// <summary>
        /// 編集結果の取得
        /// </summary>
        public override object GetValue()
        {
            return ulong.Parse(_TextBox.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            TextBox textBox = null;

            textBox = new InplaceTextBox();
            textBox.AcceptsTab = true;
            textBox.Multiline = true;
            textBox.Bounds = rect;
            textBox.Parent = OwnerCtrl;

            textBox.KeyDown += OnKeyDown;
            textBox.KeyPress += OnKeyPress;
            textBox.Leave += OnLeave;
            textBox.MouseDown += OnMouseDown;
            textBox.MouseWheel += OnMouseWheel;

            textBox.Show();
            textBox.Focus();

            //
            if (text == null)
            {
                text = InitialValue.ToString();
            }

            textBox.Text = text;
            textBox.SelectionStart = text.Length;
            textBox.SelectionLength = 1;

            if (BeginByEditBeginKey != false)
            {
                textBox.SelectAll();
            }

            _TextBox = textBox;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                ulong nowValue = 0;
                ulong newValue = 0;

                try
                {
                    nowValue = (ulong)InitialValue.Value;
                    newValue = ulong.Parse(_TextBox.Text);

                    if (newValue == nowValue)
                    {
                        return false;
                    }
                }
                catch { return false; }
                return true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnMouseWheel(object sender, MouseEventArgs e)
        {
            ValidationResult result = null;
            ulong relativeValue = (ulong)Math.Abs(e.Delta) / 120;
            ulong value = 0;

            try
            {
                value = ulong.Parse(_TextBox.Text);

                if (e.Delta < 0)
                {
                    value -= (_WheelValue * relativeValue);
                }
                else
                {
                    value += (_WheelValue * relativeValue);
                }

                result = InitialValue.ValidateValue(value);
                if (result.IsValid != false)
                {
                    _TextBox.Text = value.ToString();
                    _TextBox.SelectAll();
                }
            }
            catch { }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数値
    /// </summary>
    public class DecimalInplaceEditor : InputInplaceEditor
    {
        private static float _InitialWheelValue = 0.01F;

        private TextBox _TextBox = null;
        private float _WheelValue = _InitialWheelValue;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DecimalInplaceEditor() : base(true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            OwnerCtrl.Controls.Remove(_TextBox);

            this.Disposing = true;
            _TextBox.Dispose();
            _TextBox = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            return float.Parse(_TextBox.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            TextBox textBox = null;

            textBox = new InplaceTextBox();
            textBox.AcceptsTab = true;
            textBox.Multiline = true;
            textBox.Bounds = rect;
            textBox.Parent = OwnerCtrl;

            textBox.KeyDown += OnKeyDown;
            textBox.KeyPress += OnKeyPress;
            textBox.Leave += OnLeave;
            textBox.MouseDown += OnMouseDown;
            textBox.MouseWheel += OnMouseWheel;

            textBox.Show();
            textBox.Focus();

            //
            if (text == null)
            {
                text = InitialValue.ToString();
            }

            textBox.Text = text;
            textBox.SelectionStart = text.Length;
            textBox.SelectionLength = 1;

            if (BeginByEditBeginKey != false)
            {
                textBox.SelectAll();
            }

            _TextBox = textBox;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                float nowValue = 0;
                float newValue = 0;

                try
                {
                    nowValue = (float)InitialValue.Value;
                    newValue = float.Parse(_TextBox.Text);

                    if (newValue == nowValue)
                    {
                        return false;
                    }
                }
                catch { return false; }
                return true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnMouseWheel(object sender, MouseEventArgs e)
        {
            ValidationResult result = null;
            int relativeValue = e.Delta / 120;
            float value = 0;

            try
            {
                value = float.Parse(_TextBox.Text);
                value += (_WheelValue * relativeValue);

                result = InitialValue.ValidateValue(value);
                if (result.IsValid != false)
                {
                    _TextBox.Text = value.ToString();
                    _TextBox.SelectAll();
                }
            }
            catch { }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数値(double)
    /// </summary>
    public class DoubleInplaceEditor : InputInplaceEditor
    {
        private static double _InitialWheelValue = 0.01;

        private TextBox _TextBox = null;
        private double _WheelValue = _InitialWheelValue;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DoubleInplaceEditor() : base(true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            OwnerCtrl.Controls.Remove(_TextBox);

            this.Disposing = true;
            _TextBox.Dispose();
            _TextBox = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            return double.Parse(_TextBox.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            TextBox textBox = null;

            textBox = new InplaceTextBox();
            textBox.AcceptsTab = true;
            textBox.Multiline = true;
            textBox.Bounds = rect;
            textBox.Parent = OwnerCtrl;

            textBox.KeyDown += OnKeyDown;
            textBox.KeyPress += OnKeyPress;
            textBox.Leave += OnLeave;
            textBox.MouseDown += OnMouseDown;
            textBox.MouseWheel += OnMouseWheel;

            textBox.Show();
            textBox.Focus();

            //
            if (text == null)
            {
                text = InitialValue.ToString();
            }

            textBox.Text = text;
            textBox.SelectionStart = text.Length;
            textBox.SelectionLength = 1;

            if (BeginByEditBeginKey != false)
            {
                textBox.SelectAll();
            }

            _TextBox = textBox;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                double nowValue = 0;
                double newValue = 0;

                try
                {
                    nowValue = (double)InitialValue.Value;
                    newValue = double.Parse(_TextBox.Text);

                    if (newValue == nowValue)
                    {
                        return false;
                    }
                }
                catch { return false; }
                return true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnMouseWheel(object sender, MouseEventArgs e)
        {
            ValidationResult result = null;
            int relativeValue = e.Delta / 120;
            double value = 0;

            try
            {
                value = double.Parse(_TextBox.Text);
                value += (_WheelValue * relativeValue);

                result = InitialValue.ValidateValue(value);
                if (result.IsValid != false)
                {
                    _TextBox.Text = value.ToString();
                    _TextBox.SelectAll();
                }
            }
            catch { }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数値(float)
    /// </summary>
    public class FloatInplaceEditor : InputInplaceEditor
    {
        private static float _InitialWheelValue = 0.01f;

        private TextBox _TextBox = null;
        private float _WheelValue = _InitialWheelValue;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public FloatInplaceEditor()
            : base(true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            OwnerCtrl.Controls.Remove(_TextBox);

            this.Disposing = true;
            _TextBox.Dispose();
            _TextBox = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            return float.Parse(_TextBox.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            TextBox textBox = null;

            textBox = new InplaceTextBox();
            textBox.AcceptsTab = true;
            textBox.Multiline = true;
            textBox.Bounds = rect;
            textBox.Parent = OwnerCtrl;

            textBox.KeyDown += OnKeyDown;
            textBox.KeyPress += OnKeyPress;
            textBox.Leave += OnLeave;
            textBox.MouseDown += OnMouseDown;
            textBox.MouseWheel += OnMouseWheel;

            textBox.Show();
            textBox.Focus();

            //
            if (text == null)
            {
                text = InitialValue.ToString();
            }

            textBox.Text = text;
            textBox.SelectionStart = text.Length;
            textBox.SelectionLength = 1;

            if (BeginByEditBeginKey != false)
            {
                textBox.SelectAll();
            }

            _TextBox = textBox;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                float nowValue = 0;
                float newValue = 0;

                try
                {
                    nowValue = (float)InitialValue.Value;
                    newValue = float.Parse(_TextBox.Text);

                    if (newValue == nowValue)
                    {
                        return false;
                    }
                }
                catch { return false; }
                return true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnMouseWheel(object sender, MouseEventArgs e)
        {
            ValidationResult result = null;
            int relativeValue = e.Delta / 120;
            float value = 0;

            try
            {
                value = float.Parse(_TextBox.Text);
                value += (_WheelValue * relativeValue);

                result = InitialValue.ValidateValue(value);
                if (result.IsValid != false)
                {
                    _TextBox.Text = value.ToString();
                    _TextBox.SelectAll();
                }
            }
            catch { }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ファイルパス
    /// </summary>
    public class FilePathInplaceEditor : TextInplaceEditor
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public FilePathInplaceEditor()
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            return ControlText;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ファイルパスダイアログ
    /// </summary>
    public class OpenFilePathDialogInplaceEditor : InplaceEditor
    {
        public delegate bool OpenDialogHandler(string title, IParameterValue initialValue, Argument argument, out string result);

        ///
        public class Argument : IInplaceEditorArgument
        {
            public string Filter { get; set; }
            public OpenDialogHandler OpenDialog { get; set; }
        }

        ///
        private string _Title = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public OpenFilePathDialogInplaceEditor() : base(false)
        {
        }

        /// <summary>
        ///
        /// </summary>
        public override bool CanImmediateEdit
        {
            get { return false; }
        }

        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            return ControlText;
        }


        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                return this.ControlText != InitialValue.Value.ToString();
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            OpenDialogHandler openDialog = OpenDialog;
            Argument arg = Arguments as Argument;
            if (arg != null)
            {
                if (arg.OpenDialog != null)
                {
                    openDialog = arg.OpenDialog;
                }
            }

            string resultText = null;
            if (openDialog(_Title, InitialValue, arg, out resultText) == false) // 返値は Cancel の値、false が成功
            {
                ControlText = resultText;
                return true;
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual bool OpenDialog(string title, IParameterValue initialValue, Argument arg, out string result)
        {
            string filter = null;
            bool cancel = true;

            if (arg != null)
            {
                filter = arg.Filter;
            }

            OpenFileDialog _Dialog = new OpenFileDialog();
            _Dialog.Title = title;
            _Dialog.FileName = Path.GetFileName(initialValue.Value.ToString());

            if ((string)(initialValue.Value) != string.Empty)
            {
                _Dialog.InitialDirectory = Path.GetDirectoryName(initialValue.Value.ToString());
            }

            _Dialog.Filter = filter;
            _Dialog.FilterIndex = 0;

            result = null;
            if (_Dialog.ShowDialog() == DialogResult.OK)
            {
                if (initialValue.Value.ToString() != _Dialog.FileName)
                {
                    cancel = false;
                    result = _Dialog.FileName;
                }
            }

            return cancel;
        }

        /// <summary>
        ///
        /// </summary>
        protected string ControlText
        {
            get;
            set;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// チェックボックス
    /// </summary>
    public class CheckBoxInplaceEditor : InplaceEditor
    {
        private bool _Value = false;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public CheckBoxInplaceEditor() : base(false)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            return _Value;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                return _Value != (bool)InitialValue.Value;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 編集の開始
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            if (MoveNextLineCaretAfterEndEditByEnter == true &&
                ImmediateEditKey == Keys.Enter)
            {
                return false;
            }

            IParameterValue value = InitialValue;
            bool b = false;

            if (value != null)
            {
                b = (bool)value.Value;
            }
            else
            {
                if (text == null)
                {
                    return false;
                }

                b = bool.Parse(text);
            }
            _Value = !b;

            return true;
        }
    }

    /// <summary>
    /// ドロップダウンのベースクラス
    /// </summary>
    public class DropDownBaseInplaceEditor : InplaceEditor
    {
        private Cursor preservedCursor = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DropDownBaseInplaceEditor(bool permanent) : base(permanent)
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected void PreserveCursor()
        {
            this.preservedCursor = Cursor.Current;
        }

        /// <summary>
        ///
        /// </summary>
        protected void OnDropDown(object sender, EventArgs e)
        {
            Cursor.Current = this.preservedCursor;
        }

        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            Debug.Assert(false, "It must be redefined in the inheritance class.");
            return false;
        }
    }

    /// <summary>
    /// ドロップダウン
    /// </summary>
    public class DropDownInplaceEditor : DropDownBaseInplaceEditor
    {
        ///
        public class Argument : IInplaceEditorArgument
        {
            private int _Index = 0;

            public Argument(IParameterValue[] items)
            {
                Items = items;
                Index = 0;
            }

            public IParameterValue[] Items { get; set; }
            public bool EnabledIndex { get; set; }

            public int Index
            {
                get { return _Index; }
                set
                {
                    _Index = value;
                    EnabledIndex = true;
                }
            }
        }

        ///--------------------------------
        ///
        private ComboBox _ComboBox = null;
        private IParameterValue[] _Values = null;
        private string _InitialText = null;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DropDownInplaceEditor() : base(true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            OwnerCtrl.Controls.Remove(_ComboBox);

            this.Disposing = true;
            _ComboBox.Dispose();
            _ComboBox = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            return this.ParameterManipulator.ParseValue(_ComboBox.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            //
            PreserveCursor();

            //
            ComboBox comboBox = null;

            comboBox = CreateControl();
            comboBox.Bounds = rect;
            comboBox.Parent = OwnerCtrl;
            comboBox.MaxDropDownItems = 12;
            comboBox.DropDownStyle = ComboBoxStyle.DropDown;

            comboBox.PreviewKeyDown += OnPreviewKeyDown;
            comboBox.KeyDown += OnKeyDown;
            comboBox.KeyPress += OnKeyPress;
            comboBox.Leave += OnLeave;
            comboBox.MouseDown += OnMouseDown;
            comboBox.DropDown += OnDropDown;

            //以下を有効にすると、ドロップダウンの選択で直ぐに入力を決定することが
            //できるが、キー入力において、矢印キーの入力でドロップダウンの項目選択
            //が出来なくなる
            //comboBox.SelectionChangeCommitted +=
            //new EventHandler( SelectionChangeCommitted);

            Argument arg = Arguments as Argument;
            Debug.Assert(arg != null, "Type of argument does not match");
            _Values = arg.Items;
            comboBox.DataSource = _Values;
            comboBox.DroppedDown = true;
            comboBox.Show();
            comboBox.Focus();

            //
            if (arg.EnabledIndex != false)
            {
                if (arg.Index >= 0 && arg.Index < _Values.Length)
                {
                    comboBox.SelectedIndex = arg.Index;
                }
            }
            else
            {

                foreach (IParameterValue value in _Values)
                {
                    if (value.Value.ToString() == InitialValue.Value.ToString())
                    {
                        comboBox.SelectedItem = value;
                        break;
                    }
                }
            }

            //
            if (text == null)
            {
                text = InitialValue.Value.ToString();
                comboBox.Text = text;
                comboBox.SelectAll();
            }
            else
            {

                comboBox.Text = text;
                comboBox.SelectionStart = text.Length;
                comboBox.SelectionLength = 1;

                if (BeginByEditBeginKey != false)
                {
                    comboBox.SelectAll();
                }
            }

            _InitialText = InitialValue.Value.ToString();
            _ComboBox = comboBox;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                if (_ComboBox == null) { return false; }
                return _ComboBox.Text != _InitialText ? true : false;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual ComboBox CreateControl()
        {
            return new InplaceComboBox();
        }

#if false  // 使用される可能性がありますので残しておきます。
        /// <summary>
        ///
        /// </summary>
        private void SelectionChangeCommitted( object sender, EventArgs e)
        {
            bool      cancel = false;
            string    text = null;

            text = _ComboBox.SelectedItem as string;
            Debug.Assert( text != null, "Text is null");
            if( text == _InitialText  ) {
                cancel = true;
            } else {
                _ComboBox.Text = text;
            }

            EndEdit( cancel);
        }
#endif
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ドロップダウン(入力補完機能付き)
    /// </summary>
    public class AssistDropDownInplaceEditor : DropDownBaseInplaceEditor
    {
        private ComboBox _ComboBox = null;

        private string _InitialText = null;

        private bool _SuspendUpdateDropDownList = false;
        private bool _SuspendDropDownListFilter = false;

        private bool _PressedArrowKey = false;

        private IParameterValue[] _Values = null;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AssistDropDownInplaceEditor() : base(true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            OwnerCtrl.Controls.Remove(_ComboBox);

            this.Disposing = true;
            _ComboBox.Dispose();
            _ComboBox = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            return this.ParameterManipulator.ParseValue(_ComboBox.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            //
            PreserveCursor();

            //
            ComboBox comboBox = null;

            DropDownInplaceEditor.Argument arg = Arguments as DropDownInplaceEditor.Argument;
            Debug.Assert(arg != null, "Type of argument does not match");
            _Values = arg.Items;

            comboBox = CreateControl();
            comboBox.Bounds = rect;

            comboBox.PreviewKeyDown += OnPreviewKeyDown;
            comboBox.KeyDown += OnKeyDown;
            comboBox.KeyPress += OnKeyPress;
            comboBox.Leave += OnLeave;
            comboBox.MouseDown += OnMouseDown;
            comboBox.DropDown += OnDropDown;

            //以下を有効にすると、ドロップダウンの選択で直ぐに入力を決定することが
            //できるが、キー入力において、矢印キーの入力でドロップダウンの項目選択
            //が出来なくなる
            //comboBox.SelectionChangeCommitted +=
            //new EventHandler( SelectionChangeCommitted);

            comboBox.Parent = OwnerCtrl;
            comboBox.MaxDropDownItems = 12;
            comboBox.DropDownStyle = ComboBoxStyle.DropDown;

            //DataSourceにすると Items.Clearなどができないのでアイテム単位で追加する
            foreach (IParameterValue value in _Values)
            {
                comboBox.Items.Add(value.ToString());
            }

            comboBox.DroppedDown = true;
            comboBox.Show();
            comboBox.Focus();

            comboBox.TextChanged += new EventHandler(OnTextChanged);

            //最初はドロップダウンのフィルタ処理を行なわないようにする
            //_ComboBox.Text = ?で UpdateDropDownListが呼ばれるので
            //ここでフィルタ処理を一時停止にする
            _SuspendDropDownListFilter = true;
            _ComboBox = comboBox;

            //
            if (arg.EnabledIndex != false)
            {
                if (arg.Index >= 0 && arg.Index < _Values.Length)
                {
                    comboBox.SelectedIndex = arg.Index;
                }
            }
            else
            {

                foreach (IParameterValue value in _Values)
                {
                    if (value.Value.ToString() == InitialValue.Value.ToString())
                    {
                        comboBox.SelectedItem = value;
                        break;
                    }
                }
            }

            //
            if (text == null)
            {
                text = InitialValue.Value.ToString();
                comboBox.Text = text;
                comboBox.SelectAll();
            }
            else
            {

                comboBox.Text = text;
                comboBox.SelectionStart = text.Length;
                comboBox.SelectionLength = 1;

                if (BeginByEditBeginKey != false)
                {
                    comboBox.SelectAll();
                }
            }

            //ドロップダウンのフィルタ処理の再開
            _SuspendDropDownListFilter = false;

            _InitialText = InitialValue.Value.ToString();
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IParameterValue[] Values
        {
            get { return _Values; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual ComboBox CreateControl()
        {
            return new InplaceComboBox();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnKeyDown(object sender, KeyEventArgs e)
        {
            base.OnKeyDown(sender, e);

            //
            switch (e.KeyCode)
            {
                //case Keys.Left:
                //case Keys.Right:
                case Keys.Up:
                case Keys.Down:
                    _PressedArrowKey = true;
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnKeyPress(object sender, KeyPressEventArgs e)
        {
            base.OnKeyPress(sender, e);

            if (e.KeyChar == (int)Keys.RWin)
            {
                e.Handled = true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateDropDownList()
        {
            if (_SuspendUpdateDropDownList != false) { return; }
            _SuspendUpdateDropDownList = true;

            string text = _ComboBox.Text;
            int selectionStart = _ComboBox.SelectionStart;
            int selectionLength = _ComboBox.SelectionLength;

            List<string> list = new List<string>();
            Regex regex = null;
            string conditionText = @"^" + Regex.Escape(text.ToLower());

            if (_SuspendDropDownListFilter != false)
            {
                conditionText = ".*";
            }

            try
            {
                regex = new Regex(conditionText);

                foreach (IParameterValue value in _Values)
                {
                    if (regex.IsMatch(value.ToString().ToLower()) != false)
                    {
                        list.Add(value.ToString());
                    }
                }
            }

            catch { }

            //アイテムの無い ComboBoxは許されないのでダミーを追加する
            if (list.Count <= 0) { list.Add(String.Empty); }

            //_ComboBox.DataSource = list.ToArray();
            _ComboBox.Items.Clear();
            _ComboBox.Items.AddRange(list.ToArray());

            _ComboBox.SelectedIndex = -1;
            _ComboBox.SelectedItem = null;
            //_ComboBox.DroppedDown = false;
            _ComboBox.DroppedDown = true;

            _ComboBox.Text = text;
            _ComboBox.SelectionStart = selectionStart;
            _ComboBox.SelectionLength = selectionLength;

            _SuspendUpdateDropDownList = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTextChanged(Object sender, EventArgs e)
        {
            if (_PressedArrowKey == false)
            {
                UpdateDropDownList();
            }

            _PressedArrowKey = false;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                if (_ComboBox == null || this.ParameterManipulator == null)
                {
                    return false;
                }

                var editValue = this.ParameterManipulator.ParseValue(_ComboBox.Text);
                var initValue = this.ParameterManipulator.ParseValue(_InitialText);

                return editValue.ToString() != initValue.ToString();
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ドロップダウンリスト
    /// </summary>
    public class DropDownListInplaceEditor : DropDownBaseInplaceEditor
    {
        ///
        public class Argument : IInplaceEditorArgument
        {
            private int _Index = 0;

            public Argument(IParameterValue[] items)
            {
                Items = items;
            }

            public IParameterValue[] Items { get; set; }
            public bool EnabledIndex { get; set; }

            public int Index
            {
                get { return _Index; }
                set
                {
                    _Index = value;
                    EnabledIndex = true;
                }
            }
        }

        ///--------------------------------
        ///
        private ComboBox _ComboBox = null;
        private IParameterValue[] _Values = null;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DropDownListInplaceEditor() : base(true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            OwnerCtrl.Controls.Remove(_ComboBox);

            this.Disposing = true;
            _ComboBox.Dispose();
            _ComboBox = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object GetValue()
        {
            IParameterValue value = _ComboBox.SelectedItem as IParameterValue;
            return value != null ? value.Value : null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool BeginEditInternal(Rectangle rect, string text, bool readOnly)
        {
            ComboBox comboBox = null;

            comboBox = CreateControl();
            comboBox.Bounds = rect;
            comboBox.Parent = OwnerCtrl;
            comboBox.MaxDropDownItems = 12;
            comboBox.DropDownStyle = ComboBoxStyle.DropDownList;

            comboBox.PreviewKeyDown += OnPreviewKeyDown;
            comboBox.KeyDown += OnKeyDown;
            comboBox.KeyPress += OnKeyPress;
            comboBox.Leave += OnLeave;
            comboBox.SelectionChangeCommitted += SelectionChangeCommitted;

            Argument arg = Arguments as Argument;
            Debug.Assert(arg != null, "Type of argument does not match");
            _Values = arg.Items;
            comboBox.DataSource = _Values;

            //
            if (arg.EnabledIndex != false)
            {
                if (arg.Index >= 0 && arg.Index < _Values.Length)
                {
                    comboBox.SelectedIndex = arg.Index;
                }
            }
            else
            {

                foreach (IParameterValue value in _Values)
                {
                    if (value.Value.ToString() == InitialValue.Value.ToString())
                    {
                        comboBox.SelectedItem = value;
                        break;
                    }
                }
            }

            //
            comboBox.DroppedDown = true;
            comboBox.Show();
            comboBox.Focus();

            _ComboBox = comboBox;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集が行われたのか調べる
        /// </summary>
        public override bool Modified
        {
            get
            {
                if (_ComboBox == null) { return false; }

                var editValue = (_ComboBox.SelectedItem as IParameterValue).Value;
                var initValue = InitialValue.Value;

                return editValue.ToString() != initValue.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual ComboBox CreateControl()
        {
            return new InplaceComboBox();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SelectionChangeCommitted(object sender, EventArgs e)
        {
            EndEdit();
        }
    }
}
