﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Windows.Forms.Windowless;

    public class DockingPage : UserControl, IDockingPage
    {
        #region ** フィールド

        private DockingFormContainer _container = null;
        private string _pageID = string.Empty;
        private string _text = null;
        private Image _image = null;
        private DockingPageState _dockState = DockingPageState.Docked;
        private bool _dockVisible = false;
        private bool _windowActive = false;
        private bool _active = false;
        private string _toolTipText = string.Empty;
        private Rectangle _floatingBounds = new Rectangle(0, 0, 150, 100);

        #endregion

        public DockingPage(string pageID)
        {
            _pageID = (null == pageID) ? string.Empty : pageID;
        }

        private DockingPage() : this(string.Empty) { }

        #region ** プロパティ

        public string PageID
        {
            get { return _pageID; }
        }

        public new virtual string Text
        {
            get { return _text; }
            set { _text = (null == value) ? string.Empty : value; }
        }

        /// <summary>
        /// イメージを取得または設定します。
        /// </summary>
        public virtual Image Image
        {
            get { return _image; }
            set { _image = value; }
        }

        /// <summary>
        /// ツールヒントのテキストを取得または設定します。
        /// </summary>
        [DefaultValue("")]
        public string ToolTipText
        {
            get { return _toolTipText; }
            set { _toolTipText = value; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        public bool WindowActive
        {
            get { return _windowActive; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        public bool Active
        {
            get { return _active; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        public DockingFormContainer DockContainer
        {
            get { return _container; }
            set
            {
                if (value == _container) { return; }

                if (null != _container)
                {
                    _container.Pages.Remove(this);
                }

                _container = value;

                if (null != _container && !_container.Pages.Contains(this))
                {
                    _container.Pages.Add(this);
                }
            }
        }

        public Control FocusControl
        {
            get; set;
        }

        protected virtual bool Closable
        {
            get { return true; }
        }

        protected DockingPageState DockState
        {
            get { return _dockState; }
            set
            {
                if (value == _dockState) { return; }

                _dockState = value;

                OnDockStateChanged(new EventArgs());
            }
        }

        protected bool DockVisible
        {
            get { return _dockVisible; }
            set
            {
                if (value == _dockVisible) { return; }

                _dockVisible = value;

                OnDockVisibleChanged(new EventArgs());
            }
        }

        protected Rectangle FloatingBounds
        {
            get { return _floatingBounds; }
            set
            {
                if (value == _floatingBounds) { return; }

                _floatingBounds = value;

                OnFloatingBoundsChanged(new EventArgs());
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override Point ScrollToControl(Control activeControl)
        {
            return new Point(-this.HorizontalScroll.Value, -this.VerticalScroll.Value);
        }

        /// <summary>
        ///
        /// </summary>
        [Browsable(false)]
        [ReadOnly(true)]
        private NTabControl OwnerTabControl
        {
            get { return Parent as NTabControl; }
        }

        DockingPageState IDockingPage.DockState
        {
            get { return DockState; }
            set { DockState = value; }
        }

        bool IDockingPage.DockVisible
        {
            get { return DockVisible; }
            set { DockVisible = value; }
        }

        Rectangle IDockingPage.FloatingBounds
        {
            get { return FloatingBounds; }
            set { FloatingBounds = value; }
        }

        bool IDockingPage.Closable
        {
            get { return Closable; }
        }

        bool IEmbeddedPage.WindowActive
        {
            get { return _windowActive; }
            set
            {
                if (value == _windowActive) { return; }

                _windowActive = value;
                OnWindowActiveChanged(new EventArgs());
            }
        }

        bool IEmbeddedPage.Active
        {
            get { return _active; }
            set
            {
                if (value == _active) { return; }

                _active = value;
                OnActiveChanged(new EventArgs());
            }
        }

        /// <summary>
        /// タブコントロールに関連付けれているかどうかを取得します。
        /// </summary>
        bool NWTabControl.ITabPage.Attached
        {
            get { return (null != Parent); }
        }

        #endregion

        #region ** イベント

        public EventHandler WindowActiveChanged;

        public EventHandler ActiveChanged;

        event EventHandler IDockingPage.DockStateChanged
        {
            add { _dockStateChanged += value; }
            remove { _dockStateChanged -= value; }
        }

        event EventHandler IDockingPage.DockVisibleChanged
        {
            add { _dockVisibleChanged += value; }
            remove { _dockVisibleChanged -= value; }
        }

        event EventHandler NWTabControl.ITabPage.ImageChanged
        {
            add { _imageChanged += value; }
            remove { _imageChanged -= value; }
        }

        private EventHandler _imageChanged;
        private EventHandler _dockStateChanged;
        private EventHandler _dockVisibleChanged;

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnWindowActiveChanged(EventArgs e)
        {
            if (null != WindowActiveChanged)
            {
                WindowActiveChanged(this, e);
            }
        }

        protected virtual void OnActiveChanged(EventArgs e)
        {
            if (null != ActiveChanged)
            {
                ActiveChanged(this, e);
            }
        }

        protected virtual void OnImageChanged(EventArgs e)
        {
            if (null != _imageChanged)
            {
                _imageChanged(this, e);
            }
        }

        protected virtual void OnDockStateChanged(EventArgs e)
        {
            if (null != _dockStateChanged)
            {
                _dockStateChanged(this, new EventArgs());
            }
        }

        protected virtual void OnDockVisibleChanged(EventArgs e)
        {
            if (null != _dockVisibleChanged)
            {
                _dockVisibleChanged(this, new EventArgs());
            }
        }

        protected virtual void OnFloatingBoundsChanged(EventArgs e) { }

        #endregion

        #region ** メソッド

        protected virtual bool IsEnableDockingState(DockingPageState state)
        {
            return true;
        }

        bool IDockingPage.IsEnableDockingState(DockingPageState state)
        {
            return IsEnableDockingState(state);
        }

        void NWTabControl.ITabPage.Attach(NWControlHost host)
        {
            Attach(host);
        }

        void NWTabControl.ITabPage.Detach()
        {
            Detach();
        }

        void NWTabControl.ITabPage.Select()
        {
            if (FocusControl != null && FocusControl.CanFocus == true)
            {
                FocusControl.Focus();
                return;
            }

            foreach (Control control in Controls)
            {
                if (!control.CanFocus) { continue; }

                control.Focus();
                break;
            }
        }

        void NWTabControl.ITabPage.Show()
        {
            if (null == OwnerTabControl) { return; }

            if (NTabControlStyle.MinimizedStyleBegin <= OwnerTabControl.TabControlStyle &&
                OwnerTabControl.TabControlStyle <= NTabControlStyle.MinimizedStyleEnd)
            {
                return;
            }

            Show();
            (this as NWTabControl.ITabPage).Select();
        }

        private void Attach(NWControlHost host)
        {
            if (null == Parent)
            {
                Parent = host.Owner;
            }

            Debug.Assert(Parent == host.Owner);
        }

        private void Detach()
        {
            Parent = null;
        }

        #endregion
    }

    public enum DockingPageState
    {
        Docked = 0,	// ドッキング
        Floating = 1,	// フローティング
    }

    public interface IDockingPage : INTabPage, IEmbeddedPage
    {
        #region ** プロパティ

        DockingFormContainer DockContainer { get; set; }

        DockingPageState DockState { get; set; }

        bool DockVisible { get; set; }

        Rectangle FloatingBounds { get; set; }

        bool Closable { get; }

        #endregion

        #region ** イベント

        event EventHandler DockStateChanged;

        event EventHandler DockVisibleChanged;

        #endregion

        #region ** メソッド

        bool IsEnableDockingState(DockingPageState state);

        #endregion
    }
}
