﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using NintendoWare.SoundFoundation.Windows.Forms.Windowless;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public partial class DockingFormContainerTabControl : NTabControl
    {
        #region ** 固定値

        // イメージインデックス
        protected const int CloseImageIndex = 5;	// 閉じるボタン

        #endregion

        #region ** フィールド

        private StyleDictionary _styles = new DockingTabStyleDictionary();

        #endregion

        public DockingFormContainerTabControl()
        {
            CoreControlHost.RootInitialized += OnCoreControlInitialized;
            CoreControlHost.RootUninitialize += OnCoreControlUninitialize;

            InitializeComponent();

            AllowDrop = true;
            ShowTabCloseButton = false;
        }

        #region ** プロパティ

        public bool Minimized
        {
            get
            {
                return (NTabControlStyle.MinimizedStyleBegin <= TabControlStyle &&
                            TabControlStyle <= NTabControlStyle.MinimizedStyleEnd);
            }
        }

        protected override NTabControlStyle DefaultTabControlStyle
        {
            get { return NTabControlStyle.NormalMinimizableBottomTab; }
        }

        protected override StyleDictionary Styles
        {
            get { return _styles; }
        }

        #endregion

        #region ** イベント

        public event EventHandler Restored;
        public event EventHandler CloseButtonClick;
        public event NTabControlEventHandler TabDoubleClick;

        #endregion

        #region ** イベントハンドラ

        #region ** コアコントロールイベント

        private void OnCoreControlInitialized(object sender, EventArgs e)
        {
            #region // コメントアウト
#if HasCloseButton
            // DockingTabPanel.CloseButton.Click
            if( CoreControl is DockingTabControlCore ) {
                DockingTabControlCore core = CoreControl as DockingTabControlCore;
                core.TabPanel.CloseButton.Click += OnCloseButtonClick;
            }
#endif
            #endregion

            // NWTab.MouseDoubleClick
            foreach (NWTabControl.NWTab tab in CoreControl.TabPanel.Controls)
            {
                tab.MouseDoubleClick += OnTabMouseDoubleClicked;
            }

            // NWTabPanel.ControlInserted
            CoreControl.TabPanel.ControlInserted += OnTabInserted;
            CoreControl.TabPanel.ControlRemoved += OnTabRemoved;

            #region // コメントアウト
#if HasCloseButton
            // アイコン設定
            CoreControl.SetResource( DockingTabControlCore.CloseButton.ControlName, Icons.Images[ CloseImageIndex ] );
#endif
            #endregion
        }

        private void OnCoreControlUninitialize(object sender, EventArgs e)
        {
            #region // コメントアウト
#if HasCloseButton
            // DockingTabPanel.CloseButton.Click
            if( CoreControl is DockingTabControlCore ) {
                DockingTabControlCore core = CoreControl as DockingTabControlCore;
                core.TabPanel.CloseButton.Click -= OnCloseButtonClick;
            }
#endif
            #endregion

            // NWTab.MouseDoubleClick
            foreach (NWTabControl.NWTab tab in CoreControl.TabPanel.Controls)
            {
                tab.MouseDoubleClick -= OnTabMouseDoubleClicked;
            }

            // NWTabPanel.ControlInserted
            CoreControl.TabPanel.ControlInserted -= OnTabInserted;
            CoreControl.TabPanel.ControlRemoved -= OnTabRemoved;
        }

        private void OnTabInserted(object sender, NWControlEventArgs e)
        {
            Debug.Assert(e.Control is NWTabControl.NWTab);
            e.Control.MouseDoubleClick += OnTabMouseDoubleClicked;
        }

        private void OnTabRemoved(object sender, NWControlEventArgs e)
        {
            Debug.Assert(e.Control is NWTabControl.NWTab);
            e.Control.MouseDoubleClick -= OnTabMouseDoubleClicked;
        }

        private void OnTabMouseDoubleClicked(object sender, MouseEventArgs e)
        {
            if (MouseButtons.Left != e.Button) { return; }

            NWTabControl.NWTab tab = sender as NWTabControl.NWTab;
            Debug.Assert(null != tab);

            OnTabDoubleClick(new NTabControlEventArgs(tab.Model.Page as INTabPage,
                                                        CoreControl.Model.TabPages.IndexOf(tab.Model),
                                                        NTabControlAction.Selected));
        }

        #region ** 拡張ボタンイベント

        protected override void OnRestoreButtonClick(object sender, EventArgs e)
        {
            if (null != Restored)
            {
                Restored(this, new EventArgs());
            }
        }

        private void OnCloseButtonClick(object sender, EventArgs e)
        {
            if (null != CloseButtonClick)
            {
                CloseButtonClick(this, new EventArgs());
            }
        }

        #endregion

        #region ** タブイベント

        protected virtual void OnTabDoubleClick(NTabControlEventArgs e)
        {
            if (null != TabDoubleClick)
            {
                TabDoubleClick(this, e);
            }
        }

        #endregion

        #endregion

        #region ** タブ操作イベント

        protected override void OnSelectNextTab(CancelEventArgs e)
        {
            e.Cancel = true;
        }

        protected override void OnSelectPrevTab(CancelEventArgs e)
        {
            e.Cancel = true;
        }

        protected override void OnCloseSelectedTab(CancelEventArgs e)
        {
            e.Cancel = true;
        }

        #endregion

        #endregion

        public class DockingTabControlCore : NTabControlCore
        {
            public DockingTabControlCore(NWControlHost host) : base(host) { }

            public DockingTabControlCore(NWControlHost host, Core model, TabLayoutData layoutData)
                : base(host, model, layoutData) { }

            #region ** プロパティ

            public new DockingTabPanel TabPanel
            {
                get { return base.TabPanel as DockingTabPanel; }
            }

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new DockingTabControlTemplate(this); }
            }

            #endregion

            #region ** コントロール

            #region // コメントアウト
#if HasCloseButton
            /// <summary>
            /// 閉じるボタン
            /// </summary>
            public class CloseButton : NWNCButton
            {
                public const string ControlName = "CloseButton";

                public CloseButton(NWControlHost host) : base( host, ControlName ) { }

            #region ** プロパティ

                protected override INWControlTemplate DefaultTemplate
                {
                    get { return new ImageButtonTemplate( this ); }
                }

                public override bool Visible
                {
                    get
                    {
                        if( NTabControlStyle.MinimizedStyleBegin <= ( Owner as NTabControl ).TabControlStyle &&
                            ( Owner as NTabControl ).TabControlStyle <= NTabControlStyle.MinimizedStyleEnd ) {
                            return base.Visible;
                        }

                        return false;
                    }
                }

            #endregion
            }
#endif
            #endregion

            public class DockingTabPanel : NTabPanel
            {
                public DockingTabPanel(NWControlHost host) : base(host) { }

                #region ** プロパティ

                #region // コメントアウト
#if HasCloseButton
                public CloseButton CloseButton
                {
                    get { return NcControls[ CloseButton.ControlName ] as CloseButton; }
                }
#endif
                #endregion

                protected override INWControlTemplate DefaultTemplate
                {
                    get { return DockingTabPanelTemplateBottom.CreateInstance(this); }
                }

                #endregion

                #region ** イベントハンドラ

                protected override void OnInitialized(EventArgs e)
                {
                    base.OnInitialized(e);
                    UpdateButtonImages();
                }

                protected override void OnUpdateResource(NWControl control, string key)
                {
                    base.OnUpdateResource(control, key);
                    UpdateButtonImages();
                }

                #endregion

                #region ** メソッド

                private void UpdateButtonImages()
                {
                    // ボタンイメージを設定する
                    MinimizeButton.Image = GetResource(MinimizeButton.ControlName) as Image;
                    RestoreButton.Image = GetResource(RestoreButton.ControlName) as Image;
                    #region // コメントアウト
#if HasCloseButton
                    CloseButton.Image    = GetResource( CloseButton.ControlName ) as Image;
#endif
                    #endregion
                }

                #endregion
            }

            #endregion

            #region ** テンプレート

            public class DockingTabControlHostTemplate : NTabControlHostTemplate
            {
                public DockingTabControlHostTemplate(NWControlHost host) : base(host) { }

                public new static INWComponentTemplate CreateInstance(NWControlHost host)
                {
                    return new DockingTabControlHostTemplate(host);
                }

                #region ** メソッド

                protected override NWControlRoot CreateRootControl(NWControlHost host)
                {
                    return new DockingTabControlCore(host);
                }

                #endregion
            }

            public class DockingTabControlTemplate : NTabControlCoreTemplateBottom
            {
                public DockingTabControlTemplate(NWControl control) : base(control) { }

                public new static INWControlTemplate CreateInstance(NWControl control)
                {
                    return new DockingTabControlTemplate(control);
                }

                #region ** メソッド

                public override void InitializeComponent()
                {
                    Control.Controls.Add(new DockingTabPanel(Host));
                    Control.Controls.Add(new NWTabPagePanel(Host));
                }

                protected override void PerformLayoutInternal(object layoutData)
                {
                    base.PerformLayoutInternal(layoutData);

                    NWTabControl tabControl = Control as NWTabControl;
                    if (null == tabControl) { return; }

                    if (1 == tabControl.Model.TabPages.Count)
                    {
                        TabPanel.Bounds = Rectangle.Empty;
                        TabPagePanel.Bounds = Control.ClientRectangle;
                    }
                }

                #endregion
            }

            #region ** Horizontal

            public class DockingTabPanelTemplateHorizontal : NTabPanelTemplateHorizontal
            {
                #region ** 固定値

                private const int MinimizeButtonSize = 14;
                private const int CloseButtonSize = 14;

                #endregion

                protected DockingTabPanelTemplateHorizontal(NWControl control, INWControlDrawer drawer)
                    : base(control, drawer) { }

                #region ** プロパティ

                protected new DockingTabPanel Control
                {
                    get { return base.Control as DockingTabPanel; }
                }

                #endregion

                #region ** メソッド

                #region ** 初期化

                public override void InitializeComponent()
                {
                    base.InitializeComponent();

                    Control.NcControls.Add(new MinimizeButton(Host));
                    Control.NcControls.Add(new RestoreButton(Host));
                    #region // コメントアウト
#if HasCloseButton
                    Control.NcControls.Add( new CloseButton( Host ) );
#endif
                    #endregion
                }

                #endregion

                #region ** レイアウト

                public override void PerformNcLayout(Rectangle newBounds, ref Rectangle clientRect)
                {
                    if ((Control.Root as NTabControlCore).Minimizable)
                    {
                        clientRect.Width -= ControlSpacing + MinimizeButtonSize + ControlSpacing;
                    }

                    #region // コメントアウト
#if HasCloseButton
                    if( 0 < Control.Controls.Count ) {
                        clientRect.Width -= ControlSpacing + CloseButtonSize + ControlSpacing;
                    }
#endif
                    #endregion

                    if (1 < Control.Controls.Count)
                    {

                        NWTab lastTab = Control.Controls[Control.Controls.Count - 1] as NWTab;
                        Debug.Assert(null != lastTab);

                        if (clientRect.Width < lastTab.Right)
                        {
                            clientRect.X += ScrollButtonWidth + ControlSpacing;
                            clientRect.Width -= (ScrollButtonWidth + ControlSpacing) * 2;
                        }

                    }

                    PerformLayoutScrollButtons(clientRect);
                    PerformLayoutMinimizeButton(newBounds);
                    #region // コメントアウト
#if HasCloseButton
                    PerformLayoutCloseButton( newBounds );
#endif
                    #endregion
                }

                private void PerformLayoutScrollButtons(Rectangle newClientRect)
                {
                    // 表示する必要がない場合は Empty
                    if (0 == newClientRect.Left)
                    {
                        Control.ScrollUpButton.Bounds = Rectangle.Empty;
                        Control.ScrollDownButton.Bounds = Rectangle.Empty;
                        return;
                    }

                    // ScrollUp ボタンを再配置する
                    Rectangle scrollUpButtonRect = Rectangle.Empty;
                    scrollUpButtonRect.X = 0;
                    scrollUpButtonRect.Y = Padding.Top;
                    scrollUpButtonRect.Width = ScrollButtonWidth;
                    scrollUpButtonRect.Height = newClientRect.Height - Padding.Vertical;

                    Control.ScrollUpButton.Bounds = scrollUpButtonRect;

                    // ScrollDown ボタンを再配置する
                    Rectangle scrollDownButtonRect = Rectangle.Empty;
                    scrollDownButtonRect.X = Math.Max(newClientRect.Right + ControlSpacing,
                                                            scrollUpButtonRect.Right);
                    scrollDownButtonRect.Y = Padding.Top;
                    scrollDownButtonRect.Width = ScrollButtonWidth;
                    scrollDownButtonRect.Height = newClientRect.Height - Padding.Vertical;

                    Control.ScrollDownButton.Bounds = scrollDownButtonRect;
                }

                private void PerformLayoutMinimizeButton(Rectangle newBounds)
                {
                    if (!(Control.Root as NTabControlCore).Minimizable)
                    {
                        Control.MinimizeButton.Bounds = Rectangle.Empty;
                        Control.RestoreButton.Bounds = Rectangle.Empty;
                        return;
                    }

                    Rectangle bounds = Rectangle.Empty;
                    bounds.X = newBounds.Width - MinimizeButtonSize - ControlSpacing;
                    bounds.Y = (newBounds.Height - MinimizeButtonSize) / 2;
                    bounds.Width = MinimizeButtonSize;
                    bounds.Height = MinimizeButtonSize;

                    #region // コメントアウト
#if HasCloseButton
                    if( 0 < Control.Controls.Count ) {
                        bounds.X -= CloseButtonSize + ControlSpacing;
                    }
#endif
                    #endregion

                    Control.MinimizeButton.Bounds = bounds;
                    Control.RestoreButton.Bounds = bounds;
                }

                #region // コメントアウト
#if HasCloseButton
                private void PerformLayoutCloseButton(Rectangle newBounds)
                {
                    // 表示する必要がない場合は Empty
                    if( 0 >= Control.Controls.Count ) {
                        Control.DropDownButton.Bounds = Rectangle.Empty;
                        return;
                    }

                    Rectangle bounds = Rectangle.Empty;
                    bounds.X      = newBounds.Width	- CloseButtonSize - ControlSpacing;
                    bounds.Y      = ( newBounds.Height - CloseButtonSize ) / 2;
                    bounds.Width  = CloseButtonSize;
                    bounds.Height = CloseButtonSize;

                    Control.CloseButton.Bounds = bounds;
                }
#endif
                #endregion

                #endregion

                #endregion
            }

            #region ** Top

            public class DockingTabPanelTemplateTop : DockingTabPanelTemplateHorizontal
            {
                protected DockingTabPanelTemplateTop(NWControl control)
                    : base(control, new NWTabPanelDrawerTop()) { }

                public static INWControlTemplate CreateInstance(NWControl control)
                {
                    return new DockingTabPanelTemplateTop(control);
                }

                #region ** メソッド

                #region ** 描画情報

                protected override NWControlDrawInfo CreateControlDrawInfo()
                {
                    Rectangle workRect = Control.AbsoluteBounds;
                    workRect.Y = workRect.Bottom - Padding.Bottom;
                    workRect.Height = Padding.Bottom;

                    NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                    drawInfo.BorderRect = workRect;

                    return drawInfo;
                }

                #endregion

                #endregion
            }

            #endregion

            #region ** Bottom

            public class DockingTabPanelTemplateBottom : DockingTabPanelTemplateHorizontal
            {
                protected DockingTabPanelTemplateBottom(NWControl control)
                    : base(control, new NWTabPanelDrawerBottom()) { }

                public static INWControlTemplate CreateInstance(NWControl control)
                {
                    return new DockingTabPanelTemplateBottom(control);
                }

                #region ** メソッド

                #region ** 描画情報

                protected override NWControlDrawInfo CreateControlDrawInfo()
                {
                    Rectangle workRect = Control.AbsoluteBounds;
                    workRect.Height = Padding.Top;

                    NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                    drawInfo.BorderRect = workRect;

                    return drawInfo;
                }

                #endregion

                #endregion
            }

            #endregion

            #region ** Minimized

            public class DockingTabControlTemplateHorizontalMinimized : NTabControlCoreTemplateHorizontalMinimized
            {
                public DockingTabControlTemplateHorizontalMinimized(NWControl control) : base(control) { }

                public new static INWControlTemplate CreateInstance(NWControl control)
                {
                    return new DockingTabControlTemplateHorizontalMinimized(control);
                }

                #region ** メソッド

                public override void InitializeComponent()
                {
                    Control.Controls.Add(new DockingTabPanel(Host));
                    Control.Controls.Add(new NWTabPagePanel(Host));
                }

                protected override void PerformLayoutInternal(object layoutData)
                {
                    base.PerformLayoutInternal(layoutData);
                }

                #endregion
            }

            public class DockingTabPanelTemplateHorizontalMinimized : DockingTabPanelTemplateHorizontal
            {
                protected DockingTabPanelTemplateHorizontalMinimized(NWControl control)
                    : base(control, new NWControlDrawer()) { }

                public static INWControlTemplate CreateInstance(NWControl control)
                {
                    return new DockingTabPanelTemplateHorizontalMinimized(control);
                }
            }

            #endregion

            #endregion

            #region ** Vertical

            public class DockingTabPanelTemplateVertical : NTabPanelTemplateVertical
            {
                #region ** 固定値

                private const int MinimizeButtonSize = 14;
                private const int CloseButtonSize = 14;

                #endregion

                protected DockingTabPanelTemplateVertical(NWControl control) : base(control, new NWControlDrawer()) { }

                #region ** プロパティ

                protected new DockingTabPanel Control
                {
                    get { return base.Control as DockingTabPanel; }
                }

                #endregion

                #region ** メソッド

                #region ** 初期化

                public override void InitializeComponent()
                {
                    base.InitializeComponent();

                    Control.NcControls.Add(new MinimizeButton(Host));
                    Control.NcControls.Add(new RestoreButton(Host));
                    #region // コメントアウト
#if HasCloseButton
                    Control.NcControls.Add( new CloseButton( Host ) );
#endif
                    #endregion
                }

                #endregion

                #region ** レイアウト

                public override void PerformNcLayout(Rectangle newBounds, ref Rectangle clientRect)
                {
                    if ((Control.Root as NTabControlCore).Minimizable)
                    {
                        int height = ControlSpacing + MinimizeButtonSize + ControlSpacing;
                        clientRect.Y += height;
                        clientRect.Height -= height;
                    }

                    if (0 < Control.Controls.Count)
                    {
                        clientRect.Height -= ControlSpacing + CloseButtonSize + ControlSpacing;
                    }

                    if (1 < Control.Controls.Count)
                    {

                        NWTab lastTab = Control.Controls[Control.Controls.Count - 1] as NWTab;
                        Debug.Assert(null != lastTab);

                        if (clientRect.Height < lastTab.Bottom)
                        {
                            clientRect.Y += ScrollButtonHeight + ControlSpacing;
                            clientRect.Height -= (ScrollButtonHeight + ControlSpacing) * 2;
                        }

                    }

                    PerformLayoutScrollButtons(clientRect);
                    PerformLayoutMinimizeButton(newBounds);
                    #region // コメントアウト
#if HasCloseButton
                    PerformLayoutCloseButton( newBounds );
#endif
                    #endregion
                }

                private void PerformLayoutScrollButtons(Rectangle newClientRect)
                {
                    // 表示する必要がない場合は Empty
                    if (1 >= Control.Controls.Count)
                    {
                        Control.ScrollUpButton.Bounds = Rectangle.Empty;
                        Control.ScrollDownButton.Bounds = Rectangle.Empty;
                        return;
                    }

                    NWTab lastTab = Control.Controls[Control.Controls.Count - 1] as NWTab;
                    Debug.Assert(null != lastTab);

                    if (newClientRect.Height >= lastTab.Bottom)
                    {
                        Control.ScrollUpButton.Bounds = Rectangle.Empty;
                        Control.ScrollDownButton.Bounds = Rectangle.Empty;
                        return;
                    }


                    // ScrollUp ボタンを再配置する
                    Rectangle scrollUpButtonRect = Rectangle.Empty;
                    scrollUpButtonRect.X = Padding.Left;
                    scrollUpButtonRect.Y = newClientRect.Top - ScrollButtonHeight - ControlSpacing;
                    scrollUpButtonRect.Width = newClientRect.Width - Padding.Horizontal;
                    scrollUpButtonRect.Height = ScrollButtonHeight;

                    Control.ScrollUpButton.Bounds = scrollUpButtonRect;

                    // ScrollDown ボタンを再配置する
                    Rectangle scrollDownButtonRect = Rectangle.Empty;
                    scrollDownButtonRect.X = Padding.Left;
                    scrollDownButtonRect.Y = Math.Max(newClientRect.Bottom + ControlSpacing,
                                                            scrollUpButtonRect.Bottom);
                    scrollDownButtonRect.Width = newClientRect.Width - Padding.Horizontal;
                    scrollDownButtonRect.Height = ScrollButtonHeight;

                    Control.ScrollDownButton.Bounds = scrollDownButtonRect;
                }

                private void PerformLayoutMinimizeButton(Rectangle newBounds)
                {
                    if (!(Control.Root as NTabControlCore).Minimizable)
                    {
                        Control.MinimizeButton.Bounds = Rectangle.Empty;
                        Control.RestoreButton.Bounds = Rectangle.Empty;
                        return;
                    }

                    Rectangle bounds = Rectangle.Empty;
                    bounds.X = (newBounds.Width - MinimizeButtonSize) / 2;
                    bounds.Y = ControlSpacing;
                    bounds.Width = MinimizeButtonSize;
                    bounds.Height = MinimizeButtonSize;

                    Control.MinimizeButton.Bounds = bounds;
                    Control.RestoreButton.Bounds = bounds;
                }

                #region // コメントアウト
#if HasCloseButton
                private void PerformLayoutCloseButton(Rectangle newBounds)
                {
                    // 表示する必要がない場合は Empty
                    if( 1 >= Control.Controls.Count ) {
                        Control.DropDownButton.Bounds = Rectangle.Empty;
                        return;
                    }

                    Rectangle bounds = Rectangle.Empty;
                    bounds.X      = ( newBounds.Width - CloseButtonSize ) / 2;
                    bounds.Y      = newBounds.Height - CloseButtonSize - ControlSpacing;
                    bounds.Width  = CloseButtonSize;
                    bounds.Height = CloseButtonSize;

                    Control.CloseButton.Bounds = bounds;
                }
#endif
                #endregion

                #endregion

                #endregion
            }

            #region ** Left

            public class DockingTabPanelTemplateLeft : DockingTabPanelTemplateHorizontal
            {
                protected DockingTabPanelTemplateLeft(NWControl control)
                    : base(control, new NWTabPanelDrawerLeft()) { }

                public static INWControlTemplate CreateInstance(NWControl control)
                {
                    return new DockingTabPanelTemplateLeft(control);
                }

                #region ** メソッド

                #region ** 描画情報

                protected override NWControlDrawInfo CreateControlDrawInfo()
                {
                    Rectangle workRect = Control.AbsoluteBounds;
                    workRect.X = workRect.Right - Padding.Right;
                    workRect.Width = Padding.Right;

                    NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                    drawInfo.BorderRect = workRect;

                    return drawInfo;
                }

                #endregion

                #endregion
            }

            #endregion

            #region ** Right

            public class DockingTabPanelTemplateRight : DockingTabPanelTemplateHorizontal
            {
                protected DockingTabPanelTemplateRight(NWControl control)
                    : base(control, new NWTabPanelDrawerRight()) { }

                public static INWControlTemplate CreateInstance(NWControl control)
                {
                    return new DockingTabPanelTemplateRight(control);
                }

                #region ** メソッド

                #region ** 描画情報

                protected override NWControlDrawInfo CreateControlDrawInfo()
                {
                    Rectangle workRect = Control.AbsoluteBounds;
                    workRect.Width = Padding.Left;

                    NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                    drawInfo.BorderRect = workRect;

                    return drawInfo;
                }

                #endregion

                #endregion
            }

            #endregion

            #region ** Minimized

            public class DockingTabControlTemplateVerticalMinimized : NTabControlCoreTemplateVerticalMinimized
            {
                public DockingTabControlTemplateVerticalMinimized(NWControl control) : base(control) { }

                public new static INWControlTemplate CreateInstance(NWControl control)
                {
                    return new DockingTabControlTemplateVerticalMinimized(control);
                }

                #region ** メソッド

                public override void InitializeComponent()
                {
                    Control.Controls.Add(new DockingTabPanel(Host));
                    Control.Controls.Add(new NWTabPagePanel(Host));
                }

                #endregion
            }

            public class DockingTabPanelTemplateVerticalMinimized : DockingTabPanelTemplateVertical
            {
                public DockingTabPanelTemplateVerticalMinimized(NWControl control) : base(control) { }

                public static INWControlTemplate CreateInstance(NWControl control)
                {
                    return new DockingTabPanelTemplateVerticalMinimized(control);
                }
            }

            #endregion

            #endregion

            #endregion
        }

        #region ** スタイル

        protected class DockingTabStyleDictionary : StyleDictionary
        {
            public DockingTabStyleDictionary()
            {
                // 通常スタイル
                Add(NTabControlStyle.NormalMinimizableTopTab, new DockingTabStyleTop());
                Add(NTabControlStyle.NormalMinimizableBottomTab, new DockingTabStyleBottom());

                // 最小化スタイル
                Add(NTabControlStyle.MinimizedTopTab, new DockingTabStyleMinimizedTop());
                Add(NTabControlStyle.MinimizedBottomTab, new DockingTabStyleMinimizedBottom());
                Add(NTabControlStyle.MinimizedLeftTab, new DockingTabStyleMinimizedLeft());
                Add(NTabControlStyle.MinimizedRightTab, new DockingTabStyleMinimizedRight());
            }
        }

        #region ** 通常スタイル

        public class DockingTabStyleTop : StyleMinimizableTop
        {
            public DockingTabStyleTop()
            {
                this[typeof(NWControlHost)] =
                    new NWControlHostTemplateFactory(DockingTabControlCore.DockingTabControlHostTemplate.CreateInstance);
                this[typeof(DockingTabControlCore)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabControlTemplate.CreateInstance);
                this[typeof(DockingTabControlCore.DockingTabPanel)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabPanelTemplateTop.CreateInstance);
            }
        }

        public class DockingTabStyleBottom : StyleMinimizableBottom
        {
            public DockingTabStyleBottom()
            {
                this[typeof(NWControlHost)] =
                    new NWControlHostTemplateFactory(DockingTabControlCore.DockingTabControlHostTemplate.CreateInstance);
                this[typeof(DockingTabControlCore)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabControlTemplate.CreateInstance);
                this[typeof(DockingTabControlCore.DockingTabPanel)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabPanelTemplateBottom.CreateInstance);
            }
        }

        public class DockingTabStyleLeft : StyleMinimizableLeft
        {
            public DockingTabStyleLeft()
            {
                this[typeof(NWControlHost)] =
                    new NWControlHostTemplateFactory(DockingTabControlCore.DockingTabControlHostTemplate.CreateInstance);
                this[typeof(DockingTabControlCore)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabControlTemplate.CreateInstance);
                this[typeof(DockingTabControlCore.DockingTabPanel)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabPanelTemplateLeft.CreateInstance);
            }
        }

        #endregion

        #region ** 最小化スタイル

        public class DockingTabStyleMinimizedTop : StyleMinimizedTop
        {
            public DockingTabStyleMinimizedTop()
            {
                this[typeof(NWControlHost)] =
                    new NWControlHostTemplateFactory(DockingTabControlCore.DockingTabControlHostTemplate.CreateInstance);
                this[typeof(DockingTabControlCore)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabControlTemplateHorizontalMinimized.CreateInstance);
                this[typeof(DockingTabControlCore.DockingTabPanel)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabPanelTemplateHorizontalMinimized.CreateInstance);
            }
        }

        public class DockingTabStyleMinimizedBottom : StyleMinimizedBottom
        {
            public DockingTabStyleMinimizedBottom()
            {
                this[typeof(NWControlHost)] =
                    new NWControlHostTemplateFactory(DockingTabControlCore.DockingTabControlHostTemplate.CreateInstance);
                this[typeof(DockingTabControlCore)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabControlTemplateHorizontalMinimized.CreateInstance);
                this[typeof(DockingTabControlCore.DockingTabPanel)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabPanelTemplateHorizontalMinimized.CreateInstance);
            }
        }

        public class DockingTabStyleMinimizedLeft : StyleMinimizedLeft
        {
            public DockingTabStyleMinimizedLeft()
            {
                this[typeof(NWControlHost)] =
                    new NWControlHostTemplateFactory(DockingTabControlCore.DockingTabControlHostTemplate.CreateInstance);
                this[typeof(DockingTabControlCore)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabControlTemplateVerticalMinimized.CreateInstance);
                this[typeof(DockingTabControlCore.DockingTabPanel)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabPanelTemplateVerticalMinimized.CreateInstance);
            }
        }

        public class DockingTabStyleMinimizedRight : StyleMinimizedRight
        {
            public DockingTabStyleMinimizedRight()
            {
                this[typeof(NWControlHost)] =
                    new NWControlHostTemplateFactory(DockingTabControlCore.DockingTabControlHostTemplate.CreateInstance);
                this[typeof(DockingTabControlCore)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabControlTemplateVerticalMinimized.CreateInstance);
                this[typeof(DockingTabControlCore.DockingTabPanel)] =
                    new NWControlTemplateFactory(DockingTabControlCore.DockingTabPanelTemplateVerticalMinimized.CreateInstance);
            }
        }

        #endregion

        #endregion
    }
}
