﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Diagnostics;
using System.Globalization;
using System.Linq;
using System.Text;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Projects;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    ///-----------------------------------------------------------------------------
    ///
    public class ComponentTreeItem : ITreeItem, IDisposable
    {
        private ComponentTreeItemCollection _Children = null;
        private ComponentTreeAdapter _Adapter = null;
        private Component _Target = null;

        private bool _SuspendCollectionChanged = false;
        private bool _SuspendTargetCollectionChanged = false;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentTreeItem(Component component)
        {
            Target = component;

            SubItemName = String.Empty;
            Length = 16;

            Enabled = true;
            Visible = true;
            CanEdit = true;

            _Children = new ComponentTreeItemCollection();
            _Children.CollectionChanged += OnCollectionChanged;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IDisposable.Dispose()
        {
            Target = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        ITreeItem ITreeItem.Parent
        {
            get { return Parent; }
            set { Parent = value as ComponentTreeItem; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        ITreeItemCollection ITreeItem.Children
        {
            get { return Children; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentTreeItem Parent { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ITreeItemCollection Children
        {
            get { return _Children; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ITreeItem PreviousSibling
        {
            get
            {
                if (Parent == null)
                {
                    return null;
                }

                int index = Parent.Children.IndexOf(this);

                if (index - 1 < 0)
                {
                    return null;
                }

                return Parent.Children[index - 1] as ComponentTreeItem;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ITreeItem NextSibling
        {
            get
            {
                if (Parent == null)
                {
                    return null;
                }

                int index = Parent.Children.IndexOf(this);

                if (index + 1 >= Parent.Children.Count)
                {
                    return null;
                }

                return Parent.Children[index + 1] as ComponentTreeItem;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual Component Target
        {
            get { return _Target; }
            set
            {
                if (_Target != null)
                {
                    _Target.Children.CollectionChanged -= OnTargetCollectionChanged;
                }

                _Target = value;

                if (_Target != null)
                {
                    _Target.Children.CollectionChanged += OnTargetCollectionChanged;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentTreeAdapter Adapter
        {
            get
            {
                if (_Adapter == null && Parent != null)
                {
                    ComponentTreeItem parent = Parent as ComponentTreeItem;
                    Debug.Assert(parent != null, "Parent is not ComponentTreeItem");
                    return parent.Adapter;
                }
                return _Adapter;
            }
            set { _Adapter = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Enabled { get; set; }
        public bool Visible { get; set; }

        public bool Container { get; set; }
        public virtual bool Expanded { get; set; }
        public bool Expander { get; set; }

        public string ExpandIconName { get; set; }
        public string CollapseIconName { get; set; }

        public bool CanEdit { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string SubItemName { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Length { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Depth
        {
            get
            {
                ITreeItem item = Parent;
                int depth = 0;

                while (item != null)
                {
                    item = item.Parent;
                    depth++;
                }
                return depth;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IParameterValue GetValue(string name)
        {
            if (CanEdit == false)
            {
                return null;
            }

            ComponentTreeAdapter adapter = null;

            if ((adapter = Adapter) != null)
            {
                name = adapter.NameConvert(name);
            }

            if (Target.Parameters.ContainsKey(name) == false)
            {
                return null;
            }

            ParameterEditor parameterEditor = null;
            parameterEditor = new ParameterEditor(name, GetParameterValue(name));
            parameterEditor.ValueChanged += OnValueChanged;

            return parameterEditor;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void GetConstValues(Dictionary<string, string> parameters)
        {
            parameters["Caption"] = Target.Name;
            parameters["Expanded"] = Expanded.ToString();

            parameters["ExpandIcon"] = ExpandIconName;
            parameters["CollapseIcon"] = CollapseIconName;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual IParameterValue GetParameterValue(string name)
        {
            return Target.Parameters[name];
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnTargetCollectionChanged(NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (Component newItem in e.NewItems)
                    {
                        AddItem(newItem);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (Component oldItem in e.OldItems)
                    {
                        RemoveItem(oldItem);
                    }
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnValueChanged(EventArgs e)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValueChanged(object sender, EventArgs e)
        {
            ParameterEditor parameterEditor = sender as ParameterEditor;
            ComponentTreeAdapter adapter = null;

            if ((adapter = Adapter) == null)
            {
                return;
            }

            adapter.SetValue(this, parameterEditor.Name, parameterEditor.Value as String);
            OnValueChanged(e);
        }

        /// ComponentTreeItemの追加によって Componentの追加が必要無いのならば
        /// AddComponent、RemoveComponentは必要無い。
        /// この場合、TreeCtrlは Componentの完全な Viewとして動作する

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void AddComponent(ITreeItem item)
        {
            ComponentTreeAdapter adapter = null;

            if ((adapter = Adapter) == null)
            {
                return;
            }

            adapter.AddComponent(FirstParentHavingTarget, item as ComponentTreeItem);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RemoveComponent(ITreeItem item)
        {
            ComponentTreeAdapter adapter = null;

            if ((adapter = Adapter) == null)
            {
                return;
            }

            adapter.RemoveComponent(FirstParentHavingTarget, item as ComponentTreeItem);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void AddItem(Component component)
        {
            ComponentTreeAdapter adapter = null;

            if ((adapter = Adapter) == null)
            {
                return;
            }

            adapter.AddItem(component);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RemoveItem(Component component)
        {
            ComponentTreeAdapter adapter = null;

            if ((adapter = Adapter) == null)
            {
                return;
            }

            adapter.RemoveItem(component);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentTreeItem FirstParentHavingTarget
        {
            get
            {
                ComponentTreeItem item = this;

                while (item != null)
                {
                    if (item.Target != null)
                    {
                        return item as ComponentTreeItem;
                    }
                    item = item.Parent as ComponentTreeItem;
                }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            _SuspendTargetCollectionChanged = true;

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (ITreeItem newItem in e.NewItems)
                    {
                        newItem.Parent = this;

                        if (_SuspendCollectionChanged == false)
                        {
                            AddComponent(newItem);
                        }
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (ITreeItem oldItem in e.OldItems)
                    {
                        if (_SuspendCollectionChanged == false)
                        {
                            RemoveComponent(oldItem);
                        }

                        oldItem.Parent = null;
                    }
                    break;
            }

            _SuspendTargetCollectionChanged = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTargetCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (_SuspendTargetCollectionChanged != false)
            {
                return;
            }

            _SuspendCollectionChanged = true;
            OnTargetCollectionChanged(e);
            _SuspendCollectionChanged = false;
        }
    }
}
