﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Linq;
    using System.Text;
    using NintendoWare.SoundFoundation.Projects;

    public class ComponentKeyRegion : IKeyRegion, IDisposable
    {
        private ComponentVelocityRegionCollection _Children = new ComponentVelocityRegionCollection();
        private KeyRegion _Target;
        private int _Minimum;
        private int _Maximum;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentKeyRegion(KeyRegion target)
        {
            _Target = target;
            _Target.Parameters.ParameterValueChanged += OnParameterValueChanged;
            _Target.Children.CollectionChanged += OnComponentCollectionChanged;
            Children.CollectionChanged += OnCollectionChanged;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Dispose()
        {
            if (_Target != null)
            {
                _Target.Parameters.ParameterValueChanged -= OnParameterValueChanged;
                _Target.Children.CollectionChanged -= OnComponentCollectionChanged;
            }
            if (_Children != null)
            {
                foreach (ComponentVelocityRegion componentVelocityRegion in _Children)
                {
                    componentVelocityRegion.Dispose();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IVelocityRegionCollection IKeyRegion.Children
        {
            get { return Children; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IVelocityRegionCollection Children
        {
            get { return _Children; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ISampleMapItemsSource Parent { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Component Target
        {
            get
            {
                return _Target;
            }
            set
            {
                _Target = (KeyRegion)value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int MinimumIntact
        {
            set
            {
                ChangeMinimum(value);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int MaximumIntact
        {
            set
            {
                ChangeMaximum(value);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Minimum
        {
            get
            {
                return _Minimum;
            }
            set
            {
                int min = value;

                if (Parent != null && Parent.Items != null)
                {
                    ComponentKeyRegionCollection Items =
                        (ComponentKeyRegionCollection)Parent.Items;
                    ComponentKeyRegion prevItem = Items.PrevItem(this);
                    if (prevItem != null && prevItem.Maximum == Minimum - 1)
                    {
                        int prevMax = value - 1;
                        if (prevMax < 0)
                        {
                            prevMax = 0;
                            min = prevMax + 1;
                        }
                        if (127 < min)
                        {
                            min = 127;
                            prevMax = min - 1;
                        }
                        prevItem.ChangeMaximum(prevMax);
                    }
                }

                if (min < 0)
                {
                    min = 0;
                }
                if (127 < min)
                {
                    min = 127;
                }

                ChangeMinimum(min);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Maximum
        {
            get
            {
                return _Maximum;
            }
            set
            {
                int max = value;

                if (Parent != null && Parent.Items != null)
                {
                    ComponentKeyRegionCollection Items =
                        (ComponentKeyRegionCollection)Parent.Items;
                    ComponentKeyRegion nextItem = Items.NextItem(this);
                    if (nextItem != null && nextItem.Minimum == Maximum + 1)
                    {
                        int nextMin = value + 1;
                        if (127 < nextMin)
                        {
                            nextMin = 127;
                            max = nextMin - 1;
                        }
                        if (max < 0)
                        {
                            max = 0;
                            nextMin = max + 1;
                        }
                        nextItem.ChangeMinimum(nextMin);
                    }
                }

                if (max < 0)
                {
                    max = 0;
                }
                if (127 < max)
                {
                    max = 127;
                }

                ChangeMaximum(max);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ChangeMinimum(int value)
        {
            if (Parent != null)
            {
                ComponentSampleMapAdapter Adapter = (ComponentSampleMapAdapter)Parent;
                Adapter.SetValue(Target, ProjectParameterNames.KeyRegion.KeyMin, value);
                _Minimum = ((KeyRegion)Target).KeyMin;
            }
            else
            {
                _Minimum = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ChangeMaximum(int value)
        {
            if (Parent != null)
            {
                ComponentSampleMapAdapter Adapter = (ComponentSampleMapAdapter)Parent;
                Adapter.SetValue(Target, ProjectParameterNames.KeyRegion.KeyMax, value);
                _Maximum = ((KeyRegion)Target).KeyMax;
            }
            else
            {
                _Maximum = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    if (e.NewItems.Count > 0 &&
                        e.NewItems[0] is ComponentVelocityRegion)
                    {
                        ComponentVelocityRegion velRegion = (ComponentVelocityRegion)e.NewItems[0];
                        velRegion.Parent = this;

                        //this.Target.Children.Insert( e.NewStartingIndex, velRegion.Target );
                        ((ComponentSampleMapAdapter)Parent).
                            InsertComponent(Target, e.NewStartingIndex, velRegion.Target);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    if (e.OldItems.Count > 0 &&
                         e.OldItems[0] is ComponentVelocityRegion)
                    {
                        ComponentVelocityRegion velRegion = (ComponentVelocityRegion)e.OldItems[0];

                        //this.Target.Children.Remove( velRegion.Target );
                        ((ComponentSampleMapAdapter)Parent).
                            RemoveComponent(velRegion.Target);
                    }
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnComponentCollectionChanged(object sender,
                                                     NotifyCollectionChangedEventArgs e)
        {
            ComponentKeyRegion keyRegion = (ComponentKeyRegion)this;

            ((ComponentVelocityRegionCollection)(keyRegion.Children)).
                 SuspendVelocityRegionCollectionEvent();
            try
            {
                switch (e.Action)
                {
                    case NotifyCollectionChangedAction.Add:
                        if (e.NewItems.Count > 0 &&
                            e.NewItems[0] is VelocityRegion)
                        {
                            VelocityRegion velRegion = (VelocityRegion)e.NewItems[0];
                            ComponentVelocityRegion region =
                                GetVelocityRegionByModel(velRegion);
                            if (region == null)
                            {
                                region =
                                    ((ComponentSampleMapAdapter)(keyRegion.Parent)).
                                    CreateVelocityRegion(velRegion, keyRegion);
                                this.Children.Insert(e.NewStartingIndex, region);
                            }
                        }
                        break;

                    case NotifyCollectionChangedAction.Remove:
                        if (e.OldItems.Count > 0 &&
                             e.OldItems[0] is VelocityRegion)
                        {
                            VelocityRegion velRegion = (VelocityRegion)e.OldItems[0];
                            ComponentVelocityRegion region =
                                GetVelocityRegionByModel(velRegion);
                            if (region != null)
                            {
                                this.Children.Remove(region);
                            }
                        }
                        break;
                }
            }
            finally
            {
                ((ComponentVelocityRegionCollection)(keyRegion.Children)).
                    ResumeVelocityRegionCollectionEvent();
            }

            ((ComponentSampleMapAdapter)Parent).Update();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnParameterValueChanged(object sender, EventArgs e)
        {
            _Minimum = _Target.KeyMin;
            _Maximum = _Target.KeyMax;
            if (Parent != null)
            {
                ((ComponentSampleMapAdapter)(this.Parent)).Update();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentVelocityRegion GetVelocityRegionByModel(VelocityRegion velRegion)
        {
            ComponentVelocityRegion comVelRegion = null;

            foreach (ComponentVelocityRegion region in this.Children)
            {
                if (region.Target == velRegion)
                {
                    comVelRegion = region;
                    break;
                }
            }

            return comVelRegion;
        }
    }
}
