﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public class VelocitySelectorControl : UserControl
    {
        private const int VELOCITY_MIN = 1;
        private const int VELOCITY_MAX = 127;
        private const int UNIT_COUNT = VELOCITY_MAX - VELOCITY_MIN + 1;

        private int mSelectedVelocity = VELOCITY_MAX;

        public int SelectedVelocity
        {
            get { return mSelectedVelocity; }
            set
            {
                int velocity = value;
                velocity = Math.Min(velocity, VELOCITY_MAX);
                velocity = Math.Max(velocity, VELOCITY_MIN);
                if (velocity != mSelectedVelocity)
                {
                    mSelectedVelocity = velocity;
                    OnVelocityChanged(new VelocitySelectorEventArgs(velocity));
                    Invalidate();
                }
            }
        }

        public class VelocitySelectorEventArgs : EventArgs
        {
            public int Velocity { get; set; }
            public VelocitySelectorEventArgs(int velocity)
            {
                Velocity = velocity;
            }
        }
        public delegate void VelocitySelectorEventHandler(object s, VelocitySelectorEventArgs e);
        public event VelocitySelectorEventHandler VelocityChanged;
        protected void OnVelocityChanged(VelocitySelectorEventArgs e)
        {
            if (VelocityChanged != null)
            {
                VelocityChanged(this, e);
            }
        }

        public VelocitySelectorControl()
        {
            InitializeComponent();

            SetStyle(ControlStyles.AllPaintingInWmPaint
                      | ControlStyles.DoubleBuffer, true);
        }

        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // VelocitySelectorControl
            //
            this.AutoScaleDimensions = new System.Drawing.SizeF(6F, 11F);
            this.AutoScaleMode = System.Windows.Forms.AutoScaleMode.Font;
            this.Name = "VelocitySelectorControl";
            this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.VelocitySelectorControl_MouseDown);
            this.Paint += new System.Windows.Forms.PaintEventHandler(this.VelocitySelectorControl_Paint);
            this.MouseUp += new System.Windows.Forms.MouseEventHandler(this.VelocitySelectorControl_MouseUp);
            this.MouseMove += new System.Windows.Forms.MouseEventHandler(this.VelocitySelectorControl_MouseMove);
            this.ResumeLayout(false);

        }

        private int GetYFromVelocity(int velocity)
        {
            int clHeight = ClientRectangle.Height - 2;

            int y = (VELOCITY_MAX - mSelectedVelocity) * (clHeight - 1) / (VELOCITY_MAX - VELOCITY_MIN) + 1;
            return y;
        }

        private void VelocitySelectorControl_Paint(object sender, PaintEventArgs e)
        {
            Graphics gfx = e.Graphics;
            gfx.DrawRectangle(Pens.Black, 0, 0, ClientRectangle.Width - 1, ClientRectangle.Height - 1);

            int top = GetYFromVelocity(mSelectedVelocity);
            int bottom;
            if (mSelectedVelocity > VELOCITY_MIN)
            {
                bottom = GetYFromVelocity(mSelectedVelocity - 1);
            }
            else
            {
                bottom = ClientRectangle.Height - 1;
            }

            int height = Math.Max(bottom - top, 1);
            gfx.FillRectangle(
                              Brushes.Black,
                              1,
                              top,
                              ClientRectangle.Width - 2,
                              height
                              );
        }

        private int GetVelocityFromXY(int x, int y)
        {
            int clHeight = ClientRectangle.Height - 2;

            int velocity = VELOCITY_MAX - (y - 1) * (VELOCITY_MAX - VELOCITY_MIN) / (clHeight - 1);
            velocity = Math.Min(velocity, VELOCITY_MAX);
            velocity = Math.Max(velocity, VELOCITY_MIN);

            return velocity;
        }

        private void VelocitySelectorControl_MouseDown(object sender, MouseEventArgs e)
        {
            if ((e.Button & MouseButtons.Left) != 0)
            {
                int velocity = GetVelocityFromXY(e.X, e.Y);
                SelectedVelocity = velocity;
                Capture = true;
            }
        }

        private void VelocitySelectorControl_MouseUp(object sender, MouseEventArgs e)
        {
            Capture = false;
        }

        private void VelocitySelectorControl_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (ContainsFocus == false)
            {
                Capture = false;
            }
            else if ((e.Button & MouseButtons.Left) != 0)
            {
                int velocity = GetVelocityFromXY(e.X, e.Y);
                SelectedVelocity = velocity;
                Capture = true;
            }
        }
    }
}
