﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public partial class NLogTextBox : Control
    {
        #region ** 固定値

        protected const int InvalidLineNumber = -1;
        protected const int InvalidHeight = -1;

        #endregion

        #region ** フィールド

        private ScrollBar _hScrollBar = new HScrollBar();
        private ScrollBar _vScrollBar = new VScrollBar();

        private LinkedList<Block> _blocks = new LinkedList<Block>();
        private bool _scaned = false;
        private int _lineHeight = InvalidHeight;

        private LinkedListNode<Block> _firstVisibledBlockNode = null;
        private LinkedListNode<Block> _lastVisibledBlockNode = null;
        private LinkedListNode<Block> _focusedBlockNode = null;
        private BlockCollection _selectedBlocks = new BlockCollection();

        private Brush _textBrush = null;
        private Brush _backBrush = null;
        private StringFormat _stringFormat = new StringFormat();

        // パラメータ
        private BorderStyle _borderStyle = BorderStyle.Fixed3D;
        private int _maximumLines = 0;

        #endregion

        public NLogTextBox()
        {
            // スタイルを設定する
            SetStyle(ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint | ControlStyles.Selectable, true);

            DoubleBuffered = true;

            // デフォルトパラメータを設定する
            BackColor = SystemColors.Window;

            InitializeComponent();

            // メニューイベント
            _menuItemCopy.Click += OnMenuItemCopyClick;

            // 横スクロールバー
            _hScrollBar.Anchor = AnchorStyles.Left | AnchorStyles.Bottom | AnchorStyles.Right;
            _hScrollBar.Location = new Point(1, ClientRectangle.Height - _hScrollBar.Height - 1);
            _hScrollBar.Width = ClientRectangle.Width - 2;
            _hScrollBar.Visible = false;
            Controls.Add(_hScrollBar);

            // 縦スクロールバー
            _vScrollBar.Anchor = AnchorStyles.Top | AnchorStyles.Bottom | AnchorStyles.Right;
            _vScrollBar.Location = new Point(ClientRectangle.Width - _vScrollBar.Width - 1, 1);
            _vScrollBar.Height = ClientRectangle.Height - 2;
            _vScrollBar.Visible = false;
            _vScrollBar.Scroll += OnVScroll;
            Controls.Add(_vScrollBar);
        }

        #region ** プロパティ

        [Category("Appearance")]
        [DefaultValue(BorderStyle.Fixed3D)]
        public BorderStyle BorderStyle
        {
            get { return _borderStyle; }
            set { _borderStyle = value; }
        }

        [Category("Behavior")]
        [DefaultValue(0)]
        public int MaximumLines
        {
            get { return _maximumLines; }
            set { _maximumLines = value; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        public Item[] SelectedItems
        {
            get
            {
                List<Item> items = new List<Item>();

                foreach (LinkedListNode<Block> blockNode in _selectedBlocks)
                {
                    items.Add(blockNode.Value);
                }

                return items.ToArray();
            }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        protected LinkedList<Block> Blocks
        {
            get { return _blocks; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        protected BlockCollection SelectedBlocks
        {
            get { return _selectedBlocks; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        protected Rectangle InnerRectangle
        {
            get
            {
                Rectangle innerRect = ClientRectangle;
                innerRect.X += Padding.Left;
                innerRect.Y += Padding.Top;
                innerRect.Width -= Padding.Horizontal;
                innerRect.Height -= Padding.Vertical;

                if (_borderStyle != BorderStyle.None)
                {
                    innerRect.Inflate(-1, -1);
                }

                return innerRect;
            }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        protected Rectangle TextRectangle
        {
            get
            {
                Rectangle textRect = InnerRectangle;
                textRect.Width -= (_vScrollBar.Visible) ? _vScrollBar.Width : 0;
                textRect.Height -= (_hScrollBar.Visible) ? _hScrollBar.Height : 0;

                return textRect;
            }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        protected int LineHeight
        {
            get { return _lineHeight; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        protected int LineCount
        {
            get
            {
                int lines = 0;

                foreach (Block block in _blocks)
                {
                    if (!block.Scaned) { continue; }
                    lines += block.Lines.Count;
                }

                return lines;
            }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        protected Point ScrollLocation
        {
            get { return new Point(0, _vScrollBar.Value * LineHeight); }
        }

        #region ** Control からのオーバーライド

        [DefaultValue(typeof(Color), "Window")]
        public override Color BackColor
        {
            get { return base.BackColor; }
            set { base.BackColor = value; }
        }

        #endregion

        #endregion

        #region ** イベント

        public event EventHandler SelectedItemsChanged;

        #endregion

        #region ** イベントハンドラ

        #region ** 描画イベント

        protected override void OnPaint(PaintEventArgs pe)
        {
            base.OnPaint(pe);

            Scan(!_scaned, pe.Graphics);

            UpdateVisibledBlocks();
            UpdateDrawComponents();

            DrawBackground(pe.Graphics, ClientRectangle);
            DrawBorder(pe.Graphics, ClientRectangle);

            pe.Graphics.SetClip(TextRectangle);
            DrawItems(pe.Graphics, TextRectangle);
        }

        protected override void OnPaintBackground(PaintEventArgs pevent)
        {
            // 何もしない
        }

        #endregion

        #region ** 描画パラメータ変更イベント

        protected override void OnFontChanged(EventArgs e)
        {
            base.OnFontChanged(e);

            InvalidateScan();
        }

        protected override void OnForeColorChanged(EventArgs e)
        {
            base.OnForeColorChanged(e);

            if (null != _textBrush)
            {
                _textBrush.Dispose();
                _textBrush = null;
            }
        }

        protected override void OnBackColorChanged(EventArgs e)
        {
            base.OnBackColorChanged(e);

            if (null != _backBrush)
            {
                _backBrush.Dispose();
                _backBrush = null;
            }
        }

        protected override void OnSystemColorsChanged(EventArgs e)
        {
            base.OnSystemColorsChanged(e);

            if (null != _textBrush)
            {
                _textBrush.Dispose();
                _textBrush = null;
            }

            if (null != _backBrush)
            {
                _backBrush.Dispose();
                _backBrush = null;
            }
        }

        #endregion

        #region ** スクロールイベント

        private void OnVScroll(object sender, ScrollEventArgs e)
        {
            switch (e.Type)
            {
                case ScrollEventType.ThumbTrack:
                case ScrollEventType.EndScroll:
                    InvalidateVisibledBlock();
                    break;
            }

            Invalidate();
        }

        #endregion

        #region ** マウスイベント

        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            LinkedListNode<Block> blockNode = GetBlockFromPoint(e.Location);

            if (null != blockNode)
            {

                switch (e.Button)
                {
                    case MouseButtons.Left:
                        ClearSelections();
                        SelectBlock(blockNode, false);
                        break;

                    case MouseButtons.Right:

                        if (!_selectedBlocks.Contains(blockNode))
                        {
                            ClearSelections();
                            SelectBlock(blockNode, false);
                        }

                        // メニューを表示
                        if (null != ContextMenuStrip)
                        {
                            ContextMenuStrip.Show(this, e.Location);
                        }

                        break;
                }

            }
            else
            {
                ClearSelections();
            }

            Focus();
        }

        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            int move = -_vScrollBar.SmallChange * e.Delta / 120 * 3;
            int range = _vScrollBar.Maximum - _vScrollBar.LargeChange + 1;

            if (_vScrollBar.Value + move < _vScrollBar.Minimum)
            {
                _vScrollBar.Value = _vScrollBar.Minimum;
            }
            else if (_vScrollBar.Value + move > range)
            {
                _vScrollBar.Value = range;
            }
            else
            {
                _vScrollBar.Value += move;
            }

            InvalidateVisibledBlock();
            Invalidate();
        }

        #endregion

        #region ** メニューイベント

        private void OnMenuItemCopyClick(object sender, EventArgs e)
        {
            CopySelectedLogText();
        }

        #endregion

        #region ** 選択イベント

        protected virtual void OnSelectedItemsChanged(EventArgs e)
        {
            if (null != SelectedItemsChanged)
            {
                SelectedItemsChanged(this, e);
            }
        }

        #endregion

        #endregion

        #region ** メソッド

        public void Append(string text)
        {
            Append(text, null);
        }

        public void Append(string text, object userData)
        {
            if (null == text || 0 == text.Length) { return; }

            if (_maximumLines > 0)
            {
                while (_blocks.Count >= _maximumLines)
                {
                    _blocks.RemoveFirst();
                }
            }

            _blocks.AddLast(new Block(this, text, userData));

            _lastVisibledBlockNode = null;
            Invalidate();
        }

        public void Clear()
        {
            bool raiseEvent = (0 < _selectedBlocks.Count);

            _blocks.Clear();
            _selectedBlocks.Clear();
            _focusedBlockNode = null;

            InvalidateScan();

            if (raiseEvent)
            {
                OnSelectedItemsChanged(new EventArgs());
            }
        }

        #region ** テキスト解析

        private void InvalidateScan()
        {
            _lineHeight = InvalidHeight;
            _scaned = false;

            InvalidateVisibledBlock();
            Invalidate();
        }

        private void InvalidateVisibledBlock()
        {
            _firstVisibledBlockNode = null;
            _lastVisibledBlockNode = null;
        }

        private void Scan(bool rescan)
        {
            using (Graphics g = CreateGraphics())
            {
                Scan(rescan, g);
            }
        }

        private void Scan(bool rescan, Graphics g)
        {
            if (!Visible) { return; }

            ScanBlocks(g, rescan);

            if (InvalidHeight == _lineHeight)
            {
                _lineHeight = GetLineHeight(g);
            }

            // ドキュメントの高さを取得する
            int lineCount = (int)LineCount;

            if (!_vScrollBar.Visible)
            {

                if (ClientRectangle.Height < lineCount * LineHeight)
                {
                    _vScrollBar.Visible = true;
                    ScanBlocks(g, true);

                    lineCount = LineCount;
                }

            }
            else
            {

                if (ClientRectangle.Height >= lineCount * LineHeight)
                {
                    _vScrollBar.Visible = false;
                    ScanBlocks(g, true);

                    lineCount = LineCount;
                }

            }

            // 縦スクロールバーの更新
            UpdateScrollBar(lineCount);

            _scaned = true;
        }

        private void ScanBlocks(Graphics g, bool rescan)
        {
            Debug.Assert(null != g);

            Rectangle textRect = TextRectangle;

            foreach (Block block in _blocks)
            {

                if (rescan || !block.Scaned)
                {
                    block.Scan(g, Font, textRect.Width, _stringFormat);
                }

            }
        }

        private int GetLineHeight()
        {
            return GetLineHeight(CreateGraphics());
        }

        private int GetLineHeight(Graphics g)
        {
            return (int)g.MeasureString(" ", Font, 0, _stringFormat).Height;
        }

        private int GetBlockLineNumber(Block block)
        {
            int lines = 0;

            foreach (Block workBlock in _blocks)
            {
                if (block == workBlock) { return lines; }
                lines += workBlock.Lines.Count;
            }

            return lines;
        }

        private LinkedListNode<Block> GetBlockFromLineNumber(int lineNumber)
        {
            if (0 == _blocks.Count) { return null; }
            if (0 > lineNumber) { return null; }

            int currentLines = 0;

            LinkedListNode<Block> blockNode = _blocks.First;

            while (null != blockNode)
            {

                if (lineNumber < currentLines + blockNode.Value.Lines.Count) { return blockNode; }

                currentLines += blockNode.Value.Lines.Count;
                blockNode = blockNode.Next;

            }

            return null;
        }

        private Rectangle GetBlockRectangle(Block block)
        {
            int blockLineNumber = GetBlockLineNumber(block);

            return new Rectangle(TextRectangle.Left, TextRectangle.Top + GetLineHeight() * blockLineNumber,
                                  TextRectangle.Width, block.Height);
        }

        private Rectangle GetLineRectangle(Block block, Line line)
        {
            Debug.Assert(null != block);
            Debug.Assert(null != line);
            return block.GetLineRectangle(line, GetBlockRectangle(block));
        }

        #endregion

        #region ** 選択

        protected void SelectBlock(LinkedListNode<Block> blockNode, bool toggle)
        {
            if (null == blockNode) { return; }

            if (_selectedBlocks.Contains(blockNode.Value))
            {

                if (toggle)
                {
                    _selectedBlocks.Remove(blockNode.Value);
                }
                else
                {
                    _menuItemCopy.Enabled = true;
                    return;
                }

            }
            else
            {

                _selectedBlocks.Add(blockNode);

                if (1 == _selectedBlocks.Count)
                {

                    _focusedBlockNode = blockNode;
                    EnsureVisible(blockNode.Value);

                    _menuItemCopy.Enabled = true;

                }

            }

            Invalidate();

            OnSelectedItemsChanged(new EventArgs());
        }

        private void SelectNextBlock()
        {
            if (null == _focusedBlockNode) { return; }
            if (null == _focusedBlockNode.Next) { return; }

            ClearSelections();
            SelectBlock(_focusedBlockNode.Next, false);
        }

        private void SelectPreviousBlock()
        {
            if (null == _focusedBlockNode) { return; }
            if (null == _focusedBlockNode.Previous) { return; }

            ClearSelections();
            SelectBlock(_focusedBlockNode.Previous, false);
        }

        private void ClearSelections()
        {
            _selectedBlocks.Clear();
            _menuItemCopy.Enabled = false;
            Invalidate();

            OnSelectedItemsChanged(new EventArgs());
        }

        #endregion

        #region ** 座標操作

        public Item GetItemFromPoint(Point point)
        {
            LinkedListNode<Block> itemNode = GetBlockFromPoint(point);
            return (null == itemNode) ? null : itemNode.Value;
        }

        private LinkedListNode<Block> GetBlockFromPoint(Point point)
        {
            return GetBlockFromPoint(point, _firstVisibledBlockNode);
        }

        private LinkedListNode<Block> GetBlockFromPoint(Point point, LinkedListNode<Block> startBlockNode)
        {
            Rectangle workBlockRect = new Rectangle();
            return GetBlockFromPoint(point, startBlockNode, out workBlockRect);
        }

        private LinkedListNode<Block> GetBlockFromPoint(Point point, LinkedListNode<Block> startBlockNode, out Rectangle blockRect)
        {
            blockRect = Rectangle.Empty;

            if (0 == _blocks.Count) { return null; }


            Point controlPoint = point;
            controlPoint.Offset(ScrollLocation);

            LinkedListNode<Block> currentNode = (null == startBlockNode) ? _blocks.First : startBlockNode;

            while (null != currentNode)
            {

                if (!currentNode.Value.Scaned)
                {
                    break;
                }

                Rectangle currentBlockRect = GetBlockRectangle(currentNode.Value);
                if (currentBlockRect.Contains(controlPoint))
                {
                    blockRect = currentBlockRect;
                    return currentNode;
                }

                currentNode = currentNode.Next;

            }

            return null;
        }

        private int GetLineNumberFromPoint(Point point, out Block block)
        {
            Line line = null;
            return GetLineNumberFromPoint(point, out block, out line);
        }

        private int GetLineNumberFromPoint(Point point, out Block block, out Line line)
        {
            block = null;
            line = null;

            Point controlPoint = point;
            controlPoint.Offset(ScrollLocation);

            LinkedListNode<Block> workNode = GetBlockFromPoint(controlPoint);
            if (null == workNode) { return InvalidLineNumber; }


            int blockLineNumber = GetLineNumberFromPoint(controlPoint, workNode.Value);

            block = workNode.Value;
            line = workNode.Value.Lines[blockLineNumber];

            return GetBlockLineNumber(block) + blockLineNumber;
        }

        private int GetLineNumberFromPoint(Point point, Block block)
        {
            Point controlPoint = point;
            controlPoint.Offset(ScrollLocation);

            Rectangle blockRect = GetBlockRectangle(block);
            int line = (controlPoint.Y - blockRect.Top) / GetLineHeight();

            if (line < 0 || block.Lines.Count <= line)
            {
                throw new Exception();
            }

            return line;
        }

        #endregion

        #region ** スクロール操作

        public void ScrollToBottom()
        {
            Scan(!_scaned);

            if (!_vScrollBar.Visible) { return; }
            _vScrollBar.Value = (1 > _vScrollBar.LargeChange) ?
                                    _vScrollBar.Maximum : _vScrollBar.Maximum - _vScrollBar.LargeChange + 1;

            InvalidateVisibledBlock();
            Invalidate();
        }

        private void UpdateScrollBar(int lineCount)
        {
            if (_vScrollBar.Visible)
            {

                int oldValue = _vScrollBar.Value;
                int oldRange = _vScrollBar.Maximum - _vScrollBar.LargeChange + 1;

                _vScrollBar.SmallChange = 1;
                _vScrollBar.LargeChange = (0 == LineHeight) ? 0 : TextRectangle.Height / LineHeight;
                _vScrollBar.Minimum = 0;
                _vScrollBar.Maximum = lineCount - 1;

                int newRange = (1 > _vScrollBar.LargeChange) ?
                                    _vScrollBar.Maximum : _vScrollBar.Maximum - _vScrollBar.LargeChange + 1;

                _vScrollBar.Value = oldValue * newRange / oldRange;

            }
            else
            {
                _vScrollBar.SmallChange = 0;
                _vScrollBar.LargeChange = 0;
                _vScrollBar.Minimum = 0;
                _vScrollBar.Maximum = 0;
            }
        }

        private void EnsureVisible(Block block)
        {
            if (null == block) { return; }
            if (0 == _blocks.Count) { return; }

            if (!_vScrollBar.Visible) { return; }

            int blockLineNumber = GetBlockLineNumber(block);

            // 表示領域より上にある場合は上スクロール
            if (_vScrollBar.Value > blockLineNumber)
            {
                _vScrollBar.Value = blockLineNumber;
                InvalidateVisibledBlock();
            }
            // 表示領域より下にある場合は下スクロール
            else if (_vScrollBar.Value + _vScrollBar.LargeChange - 1 < blockLineNumber + block.Lines.Count - 1)
            {
                _vScrollBar.Value = blockLineNumber + block.Lines.Count - 1 - _vScrollBar.LargeChange + 1;
                InvalidateVisibledBlock();
            }
        }

        #endregion

        #region ** 描画情報の更新

        private void UpdateVisibledBlocks()
        {
            if (0 == _blocks.Count)
            {
                InvalidateVisibledBlock();
                return;
            }

            if (null == _firstVisibledBlockNode)
            {

                _firstVisibledBlockNode = GetBlockFromLineNumber(_vScrollBar.Value);

                if (null == _firstVisibledBlockNode)
                {
                    _lastVisibledBlockNode = null;
                    return;
                }

            }

            if (null == _lastVisibledBlockNode)
            {

                int lastLine = _vScrollBar.Value + _vScrollBar.LargeChange + 1;

                if (lastLine < _vScrollBar.Maximum)
                {
                    _lastVisibledBlockNode = GetBlockFromLineNumber(_vScrollBar.Value + _vScrollBar.LargeChange + 1);
                }

                if (null == _lastVisibledBlockNode)
                {
                    _lastVisibledBlockNode = _blocks.Last;
                }

            }

            //Debug.WriteLine( string.Format( "line[ {0} - {1} ] {2} : {3}",
            //    GetBlockLineNumber( _firstVisibledBlock ), GetBlockLineNumber( _lastVisibledBlock ),
            //    _firstVisibledBlock.Lines[0].Text.Substring( 0, 4 ), _lastVisibledBlock.Lines[0].Text.Substring( 0, 4 ) ) );
        }

        private void UpdateDrawComponents()
        {
            if (null == _textBrush)
            {
                _textBrush = new SolidBrush(ForeColor);
            }

            if (null == _backBrush)
            {
                _backBrush = new SolidBrush(BackColor);
            }
        }

        #endregion

        #region ** 描画

        protected virtual IBlockDrawer GetDrawer(Block block)
        {
            return new DefaultBlockDrawer();
        }

        private void DrawBackground(Graphics g, Rectangle rect)
        {
            Debug.Assert(null != g);
            g.FillRectangle(_backBrush, rect);
        }

        private void DrawBorder(Graphics g, Rectangle rect)
        {
            Debug.Assert(null != g);

            rect.Width -= 1;
            rect.Height -= 1;

            g.DrawRectangle(SystemPens.ControlDark, rect);
        }

        private void DrawItems(Graphics g, Rectangle rect)
        {
            Debug.Assert(null != g);

            LinkedListNode<Block> currentNode = (null == _firstVisibledBlockNode) ? null : _firstVisibledBlockNode;

            while (null != currentNode)
            {

                if (0 >= currentNode.Value.Height) { break; }

                Rectangle blockRect = GetBlockRectangle(currentNode.Value);
                blockRect.Offset(0, -_vScrollBar.Value * LineHeight);

                Brush textBrush = _textBrush;
                Brush backBrush = _backBrush;

                if (_selectedBlocks.Contains(currentNode.Value))
                {
                    textBrush = SystemBrushes.HighlightText;
                    backBrush = (ContainsFocus) ? SystemBrushes.Highlight : SystemBrushes.ControlDark;
                }

                //Debug.WriteLine( string.Format( "draw[ {0} ] {1}",
                //    GetBlockLineNumber( currentNode.Value ), currentNode.Value.Lines[0].Text.Substring( 0, 4 ) ) );

                GetDrawer(currentNode.Value).Draw(currentNode.Value, g, Font, textBrush, backBrush, blockRect, _stringFormat);

                if (_lastVisibledBlockNode == currentNode) { break; }
                currentNode = currentNode.Next;

            }
        }

        #endregion

        #region ** クリップボード操作

        private void CopySelectedLogText()
        {
            if (1 != _selectedBlocks.Count) { return; }
            Clipboard.SetDataObject(_selectedBlocks[0].Value.Text, true);
        }

        #endregion

        #region ** Control からのオーバーライド

        protected override void SetBoundsCore(int x, int y, int width, int height, BoundsSpecified specified)
        {
            bool rescan = (Width != width);

            base.SetBoundsCore(x, y, width, height, specified);

            if (rescan)
            {
                InvalidateScan();
            }
            else
            {
                InvalidateVisibledBlock();
                Invalidate();
            }
        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            switch (keyData)
            {
                case Keys.Control | Keys.C:
                    CopySelectedLogText();
                    return true;

                case Keys.Down:
                    SelectNextBlock();
                    return true;

                case Keys.Up:
                    SelectPreviousBlock();
                    return true;
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        #endregion

        #endregion

        #region ** モデルクラス

        public class TextComponent
        {
            NLogTextBox _owner = null;

            public TextComponent(NLogTextBox owner)
            {
                Debug.Assert(null != owner);
                _owner = owner;
            }

            protected NLogTextBox Owner
            {
                get { return _owner; }
            }
        }

        public class Item : TextComponent
        {
            #region ** フィールド

            private string _text = string.Empty;
            private object _userData = null;

            #endregion

            public Item(NLogTextBox owner, string text, object userData) : base(owner)
            {
                Debug.Assert(null != text);
                Debug.Assert(0 < text.Length);

                _text = text;
                _userData = userData;
            }

            #region ** プロパティ

            public string Text
            {
                get { return _text; }
            }

            public object UserData
            {
                get { return _userData; }
            }

            #endregion
        }

        protected class Block : Item
        {
            #region ** フィールド

            Collection<Line> _lines = new Collection<Line>();

            #endregion

            public Block(NLogTextBox owner, string text, object userData) : base(owner, text, userData) { }

            #region ** プロパティ

            public ReadOnlyCollection<Line> Lines
            {
                get { return new ReadOnlyCollection<Line>(_lines); }
            }

            public bool Selected
            {
                get { return Owner.SelectedBlocks.Contains(this); }
            }

            public bool Scaned
            {
                get { return (0 == Text.Length || 0 < _lines.Count); }
            }

            public int Height
            {
                get
                {
                    if (!Scaned) { return InvalidHeight; }
                    return _lines.Count * Owner.LineHeight;
                }
            }

            public int LineHeight
            {
                get { return Owner.LineHeight; }
            }

            #endregion

            #region ** メソッド

            public void Scan(Graphics g, Font font, int width, StringFormat stringFormat)
            {
                Debug.Assert(null != g);
                Debug.Assert(null != font);
                Debug.Assert(null != stringFormat);


                StringFormat measureFormat = new StringFormat(stringFormat);
                measureFormat.FormatFlags |= StringFormatFlags.NoWrap;

                _lines.Clear();

                string[] lineTexts = Text.Split('\n');

                foreach (string lineText in lineTexts)
                {

                    if (0 == lineText.Length) { continue; }

                    int currentChar = 0;
                    int lines = 0;

                    while (currentChar < lineText.Length)
                    {

                        int chars = 0;
                        string text = lineText.Substring(currentChar);

                        g.MeasureString(text, font, new SizeF(width, 0), measureFormat,
                                         out chars, out lines);

                        _lines.Add(new Line(Owner, _lines.Count, text.Substring(0, chars)));

                        currentChar += chars;

                    }

                }
            }

            public Rectangle GetLineRectangle(Line line, Rectangle blockRect)
            {
                if (null == line) { throw new ArgumentNullException("line"); }

                Rectangle lineRect = blockRect;
                lineRect.Height = (int)Owner.LineHeight;
                lineRect.Y += line.Number * lineRect.Height;

                return lineRect;
            }

            #endregion
        }

        protected class Line : TextComponent
        {
            #region ** フィールド

            private int _number = InvalidLineNumber;
            private string _text = string.Empty;

            #endregion

            public Line(NLogTextBox owner, int number, string text) : base(owner)
            {
                Debug.Assert(null != owner);
                Debug.Assert(InvalidLineNumber < number);

                _number = number;
                _text = (null == text) ? string.Empty : text;
            }

            #region ** プロパティ

            public int Number
            {
                get { return _number; }
            }

            public string Text
            {
                get { return _text; }
            }

            #endregion
        }

        protected class BlockCollection : KeyedCollection<Block, LinkedListNode<Block>>
        {
            protected override Block GetKeyForItem(LinkedListNode<Block> item)
            {
                return item.Value;
            }
        }

        #endregion

        #region ** キャレットクラス

        private class Caret
        {
            private Control _owner = null;
            private Bitmap _image = null;
            private Size _size = Size.Empty;

            public Caret(Control owner)
            {
                Debug.Assert(null != owner);
                _owner = owner;

                _owner.GotFocus += OnOwnerGotFocus;
                _owner.LostFocus += OnOwnerLostFocus;

                _size.Width = 1;
                _size.Height = _owner.Font.Height;
            }

            #region ** プロパティ

            public Control Owner
            {
                get { return _owner; }
            }

            public Bitmap Image
            {
                get { return _image; }
                set
                {
                    if (value == _image) { return; }

                    _image = value;
                    Initialize();
                }
            }

            public Point Location
            {
                set
                {
                    SetCaretPos(value.X, value.Y);
                }
            }

            public Size Size
            {
                get { return _size; }
                set
                {
                    if (value == _size) { return; }

                    _size = value;
                    Initialize();
                }
            }

            public bool Visible
            {
                set
                {
                    if (value)
                    {
                        ShowCaret(_owner.Handle);
                    }
                    else
                    {
                        HideCaret(_owner.Handle);
                    }
                }
            }

            #endregion

            #region ** イベントハンドラ

            private void OnOwnerGotFocus(object sender, EventArgs e)
            {
                Initialize();
                Visible = true;
            }

            private void OnOwnerLostFocus(object sender, EventArgs e)
            {
                Visible = false;
                Unintialize();
            }

            #endregion

            #region ** メソッド

            private void Initialize()
            {
                Unintialize();
                CreateCaret(_owner.Handle, (null != _image) ? _image.GetHbitmap() : IntPtr.Zero, _size.Width, _size.Height);
            }

            private void Unintialize()
            {
                DestroyCaret();
            }

            #endregion

            #region ** DllImport

            [DllImport("user32.dll")]
            public static extern bool CreateCaret(IntPtr hWnd, IntPtr hBitmap, int nWidth, int nHeight);

            [DllImport("user32.dll")]
            public static extern bool DestroyCaret();

            [DllImport("user32.dll")]
            public static extern bool GetCaretPos(IntPtr lpPoint);

            [DllImport("user32.dll")]
            public static extern bool SetCaretPos(int x, int y);

            [DllImport("user32.dll")]
            public static extern bool ShowCaret(IntPtr hWnd);

            [DllImport("user32.dll")]
            public static extern bool HideCaret(IntPtr hWnd);

            #endregion
        }

        #endregion

        #region ** 描画クラス

        protected interface IBlockDrawer
        {
            void Draw(Block block, Graphics g, Font font, Brush textBrush, Brush backBrush, Rectangle rect, StringFormat format);
        }

        protected class DefaultBlockDrawer : IBlockDrawer
        {
            public virtual void Draw(Block block, Graphics g, Font font, Brush textBrush, Brush backBrush, Rectangle rect, StringFormat format)
            {
                Rectangle lineRect = rect;
                lineRect.Height = (int)block.LineHeight;

                g.FillRectangle(backBrush, rect);

                foreach (Line line in block.Lines)
                {
                    g.DrawString(line.Text, font, textBrush, lineRect, format);
                    lineRect.Y += lineRect.Height;
                }
            }
        }

        #endregion
    }
}
