﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Windows.Forms;

    public class KeyboardControl : UserControl
    {
        private static readonly KeyShape[] KeyShapes;
        private static readonly int OctaveLength;

        private int KeyMin;
        private int KeyLen;
        private int KeyMax;

        private int[] WhiteKeys; // 白鍵のキー番号の配列(白黒ミックスでは配列のインデックスにKeyMinを加えた値がそのままキー番号になるが、白鍵は飛び飛びの値をとるため、事前に用意しておく)

        static KeyboardControl()
        {
            // 上側を均等に割る設定。
            // 白鍵側を均等に割る場合はコメントアウトしてある方を使う
            // (あまり違いはないのですが、厳密にはここの設定を変えると E と F の間が1オクターブの 1/84ぐらいずれます)
            // ちなみに、本物のピアノは白鍵が均等に割られています。
            const int WS = 20; // 35
            const int WL = 21; // 35
            const int MS = 12; // 21
            const int ML = 12; // 20
            Debug.Assert(WS * 3 == MS * 5 && WL * 4 == ML * 7, "?");
            // C〜Bの上辺と下辺のローカル座標です。
            KeyShapes = new KeyShape[12];
            KeyShapes[0] = new KeyShape(0, MS, 0, WS);
            KeyShapes[1] = new KeyShape(MS, MS, WS, 0);
            KeyShapes[2] = new KeyShape(MS * 2, MS, WS, WS);
            KeyShapes[3] = new KeyShape(MS * 3, MS, WS * 2, 0);
            KeyShapes[4] = new KeyShape(MS * 4, MS, WS * 2, WS);
            KeyShapes[5] = new KeyShape(MS * 5, ML, WS * 3, WL);
            KeyShapes[6] = new KeyShape(MS * 5 + ML, ML, WS * 3 + WL, 0);
            KeyShapes[7] = new KeyShape(MS * 5 + ML * 2, ML, WS * 3 + WL, WL);
            KeyShapes[8] = new KeyShape(MS * 5 + ML * 3, ML, WS * 3 + WL * 2, 0);
            KeyShapes[9] = new KeyShape(MS * 5 + ML * 4, ML, WS * 3 + WL * 2, WL);
            KeyShapes[10] = new KeyShape(MS * 5 + ML * 5, ML, WS * 3 + WL * 3, 0);
            KeyShapes[11] = new KeyShape(MS * 5 + ML * 6, ML, WS * 3 + WL * 3, WL);
            OctaveLength = WS * 3 + WL * 4;
        }

        /// <summary>
        /// キーボードコントロールを作成します。左右のキーが黒鍵の場合、表示が難しいので、白鍵でなければならないという制約を設けてあります。
        /// </summary>
        /// <param name="keyMin">左端のキーです。白鍵でなければなりません。</param>
        /// <param name="keyLen">キーの個数です。右端は白鍵でなければなりません。</param>
        public KeyboardControl() : this(0, 128) { }
        public KeyboardControl(int keyMin, int keyLen)
        {
            KeyMin = keyMin;
            KeyLen = keyLen;
            KeyMax = KeyMin + KeyLen;
            //Debug.Assert(IsBlack(KeyMin) == false && IsBlack(KeyMax - 1) == false);

            ArrayList twk = new ArrayList();
            for (int i = KeyMin; i < KeyMax; i++)
            {
                if (IsBlack(i) == false)
                {
                    twk.Add(i);
                }
            }
            WhiteKeys = twk.ToArray(typeof(int)) as int[];

            InitializeComponent();

            SetStyle(ControlStyles.AllPaintingInWmPaint
                      | ControlStyles.DoubleBuffer, true);
        }

        ///
        public int[] GetKeyboardWidthArray()
        {
            UpdateVisibleWidths(ClientRectangle.Width);
            return VisibleWidths;
        }

        // マウスでのキーボード操作
        public class KeyboardEventArgs : EventArgs
        {
            public int Key { get; set; }
            public KeyboardEventArgs(int key)
            {
                Key = key;
            }
        }
        public delegate void KeyboardEventHandler(object s, KeyboardEventArgs e);
        public event KeyboardEventHandler KeyOn;
        public void OnKeyOn(KeyboardEventArgs e)
        {
            if (KeyOn != null)
                KeyOn(this, e);
        }
        public event KeyboardEventHandler KeyOff;
        public void OnKeyOff(KeyboardEventArgs e)
        {
            if (KeyOff != null)
                KeyOff(this, e);
        }

        // MouseDown の前に実行されます。
        public class MouseDownBeforeEventArgs : CancelEventArgs
        {
            public MouseDownBeforeEventArgs(MouseEventArgs e)
            {
                this.Button = e.Button;
                this.Clicks = e.Clicks;
                this.Delta = e.Delta;
                this.Location = e.Location;
                this.X = e.X;
                this.Y = e.Y;
                this.Cancel = false;
            }

            public MouseButtons Button { get; private set; }
            public int Clicks { get; private set; }
            public int Delta { get; private set; }
            public Point Location { get; private set; }
            public int X { get; private set; }
            public int Y { get; private set; }
        }
        public delegate void MouseDownBeforeEventHandler(object sender, MouseDownBeforeEventArgs e);
        public event MouseDownBeforeEventHandler MouseDownBefore;
        public void OnMouseDownBefore(object sender, MouseDownBeforeEventArgs e)
        {
            if (MouseDownBefore != null)
            {
                MouseDownBefore(sender, e);
            }
        }

        private void InitializeComponent()
        {
            //
            // KeyboardControl
            //
            this.Name = "KeyboardControl";
            this.MouseUp += new System.Windows.Forms.MouseEventHandler(this.KeyboardControl_MouseUp);
            this.Paint += new System.Windows.Forms.PaintEventHandler(this.KeyboardControl_Paint);
            this.MouseMove += new System.Windows.Forms.MouseEventHandler(this.KeyboardControl_MouseMove);
            this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.KeyboardControl_MouseDown);

        }
        private struct KeyShape
        {
            public int UpperX;
            public int UpperW;
            public int LowerX;
            public int LowerW;
            public KeyShape(int upperX, int upperW, int lowerX, int lowerW)
            {
                UpperX = upperX;
                UpperW = upperW;
                LowerX = lowerX;
                LowerW = lowerW;
            }
        }
        /// <summary>
        /// 黒鍵かどうかを調べます。
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        private bool IsBlack(int key)
        {
            switch (key % 12)
            {
                case 1:
                case 3:
                case 6:
                case 8:
                case 10:
                    return true;
                default:
                    return false;
            }
        }
        private int GetLocalUpperX(int key)
        {
            return (key / 12) * OctaveLength + KeyShapes[key % 12].UpperX;
        }
        private int GetLocalUpperW(int key)
        {
            return KeyShapes[key % 12].UpperW;
        }
        private int GetLocalLowerX(int key)
        {
            return (key / 12) * OctaveLength + KeyShapes[key % 12].LowerX;
        }
        private int GetLocalLowerW(int key)
        {
            return KeyShapes[key % 12].LowerW;
        }

        private int VisibleWidth = -1;
        private int[] VisibleWidths; // 白黒ミックスでの表示幅の配列
        private int[] WhiteVisibleWidths; // 白鍵の表示幅の配列

        private void UpdateVisibleWidths(int visibleWidth)
        {
            if (VisibleWidth != visibleWidth)
            {
                VisibleWidth = visibleWidth;
                int localX = GetLocalLowerX(KeyMin);
                int localWidth = GetLocalLowerX(KeyMax) - localX;
                VisibleWidths = new int[KeyLen];
                int old = GetLocalLowerX(KeyMin); // A など、左端の白鍵の左側に黒鍵が本来はあり、それが表示されない場合にずれないようにするため、GetLocalUpperX() ではなく GetLocalLowerX() を使う。
                for (int i = 0; i < KeyLen; i++)
                {
                    int next = (i < KeyLen - 1) ? GetLocalUpperX(i + KeyMin + 1) : GetLocalLowerX(i + KeyMin + 1); // 上と同様の理由で右端も。
                    VisibleWidths[i] = next * visibleWidth / localWidth - old * visibleWidth / localWidth;
                    old = next;
                }
                WhiteVisibleWidths = new int[WhiteKeys.Length];
                for (int i = 0; i < WhiteKeys.Length; i++)
                {
                    int x = GetLocalLowerX(WhiteKeys[i]);
                    int w = GetLocalLowerW(WhiteKeys[i]);
                    WhiteVisibleWidths[i] = (x + w) * visibleWidth / localWidth - x * visibleWidth / localWidth;
                }
            }
        }
        /// <summary>
        /// 表示上の黒鍵の高さを取得します。
        /// </summary>
        private int VisibleBlackHeight
        {
            get
            {
                return ClientRectangle.Height * 2 / 3;
            }
        }

        private void KeyboardControl_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
        {
            UpdateVisibleWidths(ClientRectangle.Width);
            Image im = new Bitmap(ClientRectangle.Width, ClientRectangle.Height, e.Graphics);
            Graphics g = Graphics.FromImage(im);
            int x = 0;
            StringFormat sf = new StringFormat(StringFormat.GenericDefault);
            sf.Alignment = StringAlignment.Center;
            for (int i = 0; i < WhiteVisibleWidths.Length; i++)
            {
                Brush b;
                if (WhiteKeys[i] == _selectedKey)
                {
                    b = Brushes.Black;
                }
                else
                {
                    b = Brushes.White;
                }
                g.FillRectangle(b, x, 0, WhiteVisibleWidths[i], ClientRectangle.Height);
                if (WhiteKeys[i] % 12 == 0 && WhiteKeys[i] != 0)
                {
                    g.DrawString((WhiteKeys[i] / 12 - 1).ToString(), Font, Brushes.Blue, new Rectangle(x, ClientRectangle.Height - Font.Height - 2, WhiteVisibleWidths[i] - 1, Font.Height), sf);
                }
                g.DrawRectangle(Pens.Gray, x, 0, WhiteVisibleWidths[i] - 1, ClientRectangle.Height - 1);
                x += WhiteVisibleWidths[i];
            }
            x = 0;
            int h = VisibleBlackHeight;
            for (int i = 0; i < KeyLen; i++)
            {
                if (IsBlack(i + KeyMin))
                {
                    g.DrawLine(Pens.White, x - 1, 1, x - 1, h);
                    g.DrawLine(Pens.White, x - 1, h, x + VisibleWidths[i], h);
                    g.DrawLine(Pens.White, x + VisibleWidths[i], 1, x + VisibleWidths[i], h);
                    g.FillRectangle(Brushes.Black, x, 0, VisibleWidths[i], h);
                    if (i + KeyMin == _selectedKey)
                    {
                        int tw = VisibleWidths[i] - 2;
                        int th = h - 2;
                        if (tw > 0 && th > 0)
                            g.FillRectangle(Brushes.White, x + 1, 1, tw, th);
                    }
                }
                x += VisibleWidths[i];
            }
            e.Graphics.DrawImage(im, 0, 0);
        }

        private int _activeKey = -1;
        private int _selectedKey = 60;

        private int activeKey
        {
            get
            {
                return _activeKey;
            }
            set
            {
                _activeKey = value;
                Invalidate();
            }
        }

        public int SelectedKey
        {
            get
            {
                return _selectedKey;
            }
            set
            {
                _selectedKey = value;
                Invalidate();
            }
        }

        private void Press(int key)
        {
            OnKeyOn(new KeyboardEventArgs(key));
            activeKey = key;
            _selectedKey = key;
        }

        private void Release()
        {
            if (activeKey != -1)
            {
                OnKeyOff(new KeyboardEventArgs(activeKey));
                activeKey = -1;
            }
        }

        private int GetKey(int x, int y)
        {
            if (y >= VisibleBlackHeight)
            {
                // 白鍵側
                int cx = 0;
                for (int i = 0; i < WhiteVisibleWidths.Length; i++)
                {
                    if (x < cx + WhiteVisibleWidths[i])
                    {
                        return WhiteKeys[i];
                    }
                    cx += WhiteVisibleWidths[i];
                }
                return WhiteKeys[WhiteKeys.Length - 1]; // 右側にはみ出しているので、一番右
            }
            else
            {
                // 上側
                int cx = 0;
                for (int i = 0; i < VisibleWidths.Length; i++)
                {
                    if (x < cx + VisibleWidths[i])
                    {
                        return i + KeyMin;
                    }
                    cx += VisibleWidths[i];
                }
                return KeyMin + VisibleWidths.Length - 1; // 右側にはみ出しているので、一番右
            }
        }

        private void KeyboardControl_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            MouseDownBeforeEventArgs args = new MouseDownBeforeEventArgs(e);
            this.OnMouseDownBefore(sender, args);
            if (args.Cancel == true)
            {
                return;
            }

            if ((e.Button & MouseButtons.Left) != 0)
            {
                int key = GetKey(e.X, e.Y);
                Press(key);
                Capture = true;
            }
        }

        private void KeyboardControl_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if ((e.Button & MouseButtons.Left) != 0 ||
                (e.Button & MouseButtons.Right) != 0)
            {
                Release();
            }
        }

        private void KeyboardControl_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (ContainsFocus == false)
            {
                Release();
                Capture = false;
            }
            else if ((e.Button & MouseButtons.Left) != 0)
            {
                int key = GetKey(e.X, e.Y);
                if (key != activeKey)
                {
                    Release();
                    Press(key);
                }
                Capture = true;
            }
        }

    }
}
