﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms.Windowless
{
    /// <summary>
    /// NTabControl のメインビュークラス
    /// </summary>
    public partial class NWTabControl : NWControlRoot
    {
        #region ** パラメータ

        // コンポーネント
        private Core _model = null;                // モデル

        // UI コンポーネント
        private ToolTip _toolTip = null;            // ツールチップ

        // パラメータ
        private bool _selectionExpression = false;  // 選択タブ強調表示
        private bool _showDropDownButton = true;   // ドロップダウンボタン表示
        private bool _showTabCloseButton = true;   // タブ上の閉じるボタン表示
        private bool _showToolTips = false;  // ツールチップ表示

        // 状態
        private NWControl _active = null;

        private Point _dragPoint = Point.Empty; // ドラッグ開始位置
        private NWTab _dragTargetTab = null;        // ドラッグ対象タブ

        private bool _canRaiseTabEvents = true;

        #endregion

        public NWTabControl(NWControlHost host)
            : this(host, string.Empty) { }

        public NWTabControl(NWControlHost host, string name)
            : base(host, name)
        {
            // モデルを初期化する
            _model = new Core(host);
            _model.SelectedTabChanging += OnCoreSelectedTabChanging;

            // レイアウト情報を設定する
            base.LayoutData = new TabLayoutData();

            // ツールチップを設定する
            _toolTip = new ToolTip();
            _toolTip.AutomaticDelay = 500;
        }

        public NWTabControl(NWControlHost host, string name, Core model, TabLayoutData layoutData)
            : base(host, name)
        {
            if (null == model) { throw new ArgumentNullException("model"); }
            if (null == layoutData) { throw new ArgumentNullException("layoutData"); }

            // モデルを設定する
            Model = model;

            // レイアウト情報を設定する
            LayoutData = layoutData;

            // ツールチップを設定する
            _toolTip = new ToolTip();
            _toolTip.AutomaticDelay = 500;
        }

        #region ** プロパティ

        public Core Model
        {
            get { return _model; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                if (value == _model) { return; }

                _model = value;
                _model.Host = Host;
                _model.SelectedTabChanging += OnCoreSelectedTabChanging;

                PerformLayout();
            }
        }

        public new TabLayoutData LayoutData
        {
            get { return base.LayoutData as TabLayoutData; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                if (value == base.LayoutData) { return; }

                base.LayoutData = value;

                PerformLayout();
            }
        }

        public NWControl ActiveControl
        {
            get { return _active; }
        }

        public bool SelectionExpression
        {
            get { return _selectionExpression; }
            set
            {
                if (value == _selectionExpression) { return; }

                _selectionExpression = value;

                if (null != TabPanel)
                {
                    TabPanel.Invalidate();
                }
            }
        }

        public bool ShowDropDownButton
        {
            get { return _showDropDownButton; }
            set
            {
                if (value == _showDropDownButton) { return; }

                _showDropDownButton = value;
                PerformLayout();
            }
        }

        public bool ShowTabCloseButton
        {
            get { return _showTabCloseButton; }
            set
            {
                if (value == _showTabCloseButton) { return; }

                _showTabCloseButton = value;
                PerformLayout();
            }
        }

        /// <summary>
        /// ツールチップの表示設定を取得または設定します。
        /// </summary>
        public bool ShowToolTips
        {
            get { return _showToolTips; }
            set { _showToolTips = value; }
        }

        protected override INWControlTemplate DefaultTemplate
        {
            get { return new NWTabControlTemplateTop(this); }
        }

        protected override Type TemplateType
        {
            get { return typeof(NWTabControlTemplate); }
        }

        protected internal bool CanRaiseTabEvents
        {
            get { return _canRaiseTabEvents; }
            set { _canRaiseTabEvents = value; }
        }

        protected virtual bool IsHorizontal
        {
            get { return (TabPanel.Template is NWTabPanelTemplateHorizontal); }
        }

        #region ** コントロール

        public NWTabPanel TabPanel
        {
            get { return Controls[NWTabPanel.ControlName] as NWTabPanel; }
        }

        public NWTabPagePanel TabPagePanel
        {
            get { return Controls[NWTabPagePanel.ControlName] as NWTabPagePanel; }
        }

        #endregion

        #endregion

        #region ** イベント

        /// <summary>
        /// オブジェクトがドラッグされると発生します。
        /// </summary>
        public event NWTabControlDragEventHandler Dragged;

        #endregion

        #region ** イベントハンドラ

        #region ** 初期化イベント

        protected override void OnInitialized(EventArgs e)
        {
            base.OnInitialized(e);

            if (null != Model.SelectedTab)
            {
                EnsureVisible(Model.SelectedTab);
            }
        }

        #endregion

        #region ** NWTabControl イベント

        protected virtual void OnDragged(NWTabControlDragEventArgs e)
        {
            if (null != Dragged)
            {
                Dragged(this, e);
            }
        }

        #endregion

        #region ** Core イベント

        private void OnCoreSelectedTabChanging(object sender, NWTabControlEventArgs e)
        {
            if (null == e.Tab) { return; }
            if (null == TabPagePanel) { return; }

            // ページサイズを設定する
            if (!TabPagePanel.AbsoluteBounds.IsEmpty)
            {
                e.Tab.Page.Bounds = TabPagePanel.AbsoluteBounds;
            }
        }

        #endregion

        #region ** レイアウトイベントのオーバーライド

        protected override void OnLayout(NWControlLayoutEventArgs e)
        {
            base.OnLayout(e);

            if (TabPagePanel.AbsoluteBounds.IsEmpty)
            {

                if (null != _model.SelectedTab)
                {
                    _model.SelectedTab.Page.Hide();
                }

                return;

            }

            // ページサイズを設定する
            if (null != _model.SelectedTab)
            {
                _model.SelectedTab.Page.Bounds = TabPagePanel.AbsoluteBounds;

                if (!_model.SelectedTab.Page.Visible)
                {
                    _model.SelectedTab.Page.Show();
                }
            }
        }

        #endregion

        #region ** マウスイベントのオーバーライド

        protected override void OnMouseLeave()
        {
            base.OnMouseLeave();

            _dragTargetTab = null;
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            if (null != _dragTargetTab)
            {

                Rectangle dragRect = Rectangle.Empty;
                dragRect.X = _dragPoint.X - (dragRect.Width / 2);
                dragRect.Y = _dragPoint.Y - (dragRect.Height / 2);
                dragRect.Width = SystemInformation.DragSize.Width;
                dragRect.Height = SystemInformation.DragSize.Height;

                if (!dragRect.Contains(e.Location))
                {

                    SuspendScroll();
                    Owner.DoDragDrop(_dragTargetTab.Model, DragDropEffects.Move);
                    ResumeScroll();

                    EnsureVisible(_model.SelectedTab);

                    _dragTargetTab = null;

                }

                return;
            }


            if ((e.Button & MouseButtons.Left) != 0) { return; }
            if (!ShowToolTips) { return; }


            NWTab tab = MouseActive as NWTab;
            if (MouseActive is NWTabButton)
            {
                tab = MouseActive.Parent as NWTab;
            }

            if (null != tab)
            {
                ShowToolTip(tab);
            }
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            // タブの場合、ドラッグターゲットを設定
            // 次の MouseMove イベントでドラッグを開始する
            if (e.Button == MouseButtons.Left)
            {

                if (MouseActive is NWTab)
                {

                    _dragPoint = e.Location;
                    _dragTargetTab = MouseActive as NWTab;
                    //Debug.WriteLine( string.Format( "[Drag] Start : {0:D}", hr.Index ) );

                }

            }
            else
            {
                Debug.Assert(null == _dragTargetTab, "unexpected error.");
            }
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            _dragTargetTab = null;

            base.OnMouseUp(e);
        }

        #endregion

        #endregion

        #region ** メソッド

        #region ** タブ操作

        public void SelectNextTab()
        {
            _model.SelectNextTab();
        }

        public void SelectPrevTab()
        {
            _model.SelectPrevTab();
        }

        public void CloseSelectedTab()
        {
            _model.CloseSelectedTab();
        }

        #endregion

        public void SetActiveControl(NWControl control)
        {
            if (ActiveControl == control) { return; }

            if (null != ActiveControl)
            {
                ActiveControl.PerformMouseLeave();
            }

            if (null != control)
            {
                control.PerformMouseEnter();
            }

            _active = control;
        }

        private void ShowToolTip(NWTab tab)
        {
            if (null != tab)
            {

                if (_toolTip.GetToolTip(Owner) != tab.Model.ToolTipText)
                {
                    _toolTip.SetToolTip(Owner, tab.Model.ToolTipText);
                }

            }
            else if (_toolTip.GetToolTip(Owner) != string.Empty)
            {
                _toolTip.SetToolTip(Owner, string.Empty);
            }
        }

        #region ** レイアウト操作

        public override void SuspendLayout()
        {
            base.SuspendLayout();

            _model.SuspendRaiseEvent();
            CanRaiseTabEvents = false;
        }

        protected override void ResumeLayout(bool performLayout, bool forced)
        {
            CanRaiseTabEvents = true;
            _model.ResumeRaiseEvent(true);

            base.ResumeLayout(performLayout, forced);
        }

        protected override void PerformLayoutInternal(bool forced)
        {
            if (null == _model) { return; }
            if (null == LayoutData) { return; }

            base.PerformLayoutInternal(forced);
        }

        #endregion

        #region ** ドラッグ＆ドロップ操作

        public void PerformDragEnter(DragEventArgs drgevent)
        {
            drgevent.Effect = DragDropEffects.None;

            NWTabControlDragEventArgs e = new NWTabControlDragEventArgs(drgevent);
            if (e.Tab == null)
            {
                return;
            }

            if ((drgevent.AllowedEffect & DragDropEffects.Move) == 0)
            {
                return;
            }

            OnDragged(e);

            drgevent.Effect = e.Effect;
        }

        public void PerformDragLeave()
        {
            Control c = new Control();

            System.Xml.Serialization.XmlSerializer a = new System.Xml.Serialization.XmlSerializer(typeof(Control));
        }

        // ★NTabPanel で処理すべきだ。
        public void PerformDragOver(DragEventArgs drgevent)
        {
            drgevent.Effect = DragDropEffects.None;

            NWTabControlDragEventArgs e = new NWTabControlDragEventArgs(drgevent);
            if (e.Tab == null)
            {
                return;
            }
            OnDragged(e);

            drgevent.Effect = e.Effect;

            int srcIndex = _model.TabPages.IndexOf(e.Tab);
            Point targetPoint = Owner.PointToClient(new Point(drgevent.X, drgevent.Y));

            NWTab srcTab = null;
            if (InvalidIndex != srcIndex)
            {
                srcTab = TabPanel.Controls[srcIndex] as NWTab;
            }
            else
            {

                drgevent.Effect = DragDropEffects.None;
                return;

                // ★他コントロールからのドラッグなので、その処理を追加する
                //   new NWTab を追加。
                //Debug.Fail( "not implemented" );

            }

            Debug.Assert(null != srcTab, "unexpected error.");


            NWHitResult hr = HitTest<NWTab>(targetPoint);
            NWTab tab = hr.Control as NWTab;

            if (null == tab)
            {
                drgevent.Effect = DragDropEffects.None;
                return;
            }


            drgevent.Effect = DragDropEffects.Move;

            if (srcIndex == hr.Index) { return; }


            if (srcIndex < hr.Index)
            {

                int index = hr.Index;

                if (IsHorizontal)
                {
                    if (tab.AbsoluteBounds.Right - srcTab.Width + 4 > targetPoint.X)
                    {
                        index--;
                    }
                }
                else
                {
                    if (tab.AbsoluteBounds.Bottom - srcTab.Height + 4 > targetPoint.Y)
                    {
                        index--;
                    }
                }

                if (index < 0) { return; }
                if (srcIndex == index) { return; }

                Debug.WriteLine(string.Format("[Drag] + < Remove:{0:D}, Inesrt:{1:D}", srcIndex, index));

                _model.TabPages.MovePage(srcIndex, index);

                Debug.WriteLine(string.Format("[Drag] - < Remove:{0:D}, Inesrt:{1:D}", srcIndex, index));

            }
            else
            {

                int index = hr.Index;

                if (IsHorizontal)
                {
                    if (tab.AbsoluteBounds.Left + srcTab.Width - 4 < targetPoint.X)
                    {
                        index++;
                    }
                }
                else
                {
                    if (tab.AbsoluteBounds.Top + srcTab.Height - 4 < targetPoint.Y)
                    {
                        index++;
                    }
                }

                if (index >= _model.TabPages.Count) { return; }
                if (srcIndex == index) { return; }

                Debug.WriteLine(string.Format("[Drag] + > Remove:{0:D}, Inesrt:{1:D}", srcIndex, index));

                _model.TabPages.MovePage(srcIndex, index);

                Debug.WriteLine(string.Format("[Drag] - > Remove:{0:D}, Inesrt:{1:D}", srcIndex, index));

            }
        }

        public void PerformDragDrop(DragEventArgs drgevent)
        {
        }

        #endregion

        #region ** スクロール操作

        /// <summary>
        /// 指定タブが表示されるように最小限のスクロールを行います。
        /// </summary>
        /// <param name="index">タブのインデックス</param>
        public void EnsureVisible(Tab tab)
        {
            TabPanel.EnsureVisible(tab);
        }

        private new void SuspendScroll()
        {
            TabPanel.SuspendScroll();
        }

        private void ResumeScroll()
        {
            ResumeScroll(true);
        }

        private new void ResumeScroll(bool performScroll)
        {
            TabPanel.ResumeScroll(performScroll);
        }

        #endregion

        #endregion

        #region ** 列挙値

        /// <summary>
        /// NWTabControl イベントを表す値を定義します。
        /// </summary>
        public enum Action
        {
            Inserted = 0,           // NWTabControl.Core.Inserted イベントを表します。
            Removed = 1,            // NWTabControl.Core.Removed イベントを表します。
            Swapped = 2,            // NWTabControl.Core.Swapped イベントを表します。
            Closing = 3,            // NWTabControl.Core.Closing イベントを表します。
            Closed = 4,             // NWTabControl.Core.Closed イベントを表します。
            Selecting = 5,          // NWTabControl.Core.Selecting イベントを表します。
            Selected = 6,           // NWTabControl.Core.Selected イベントを表します。
            Deselecting = 7,        // NWTabControl.Core.Deselecting イベントを表します。
            Deselected = 8,         // NWTabControl.Core.Deselected イベントを表します。
            ContextMenuShown = 9,   // NWTabControl.Core.ContextMenuShown イベントを表します。
        }

        /// <summary>
        ///
        /// </summary>
        public enum ActionTrigger
        {
            Unknown = 0,            // その他
            MouseDown = 1,          // マウスボタン操作
            ShortcutKey = 2,        // ショートカットキー
            TabSelectionMenu = 3,   // タブ選択メニュー
        }

        #endregion

        #region ** コントロール

        /// <summary>
        /// タブ上のボタン
        /// </summary>
        public class NWTabButton : NWButton
        {
            public NWTabButton(NWControlHost host) : base(host) { }
            public NWTabButton(NWControlHost host, string name) : base(host, name) { }

            #region ** プロパティ

            public override bool Visible
            {
                get
                {
                    if (null == Parent) { return false; }
                    if (!Parent.Selected) { return false; }
                    return base.Visible;
                }
            }

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new NWTabButtonTemplate(this); }
            }

            #endregion
        }

        /// <summary>
        /// タブ
        /// </summary>
        public class NWTab : NWControl
        {
            public const string CloseButtonName = "CloseButton";

            #region ** パラメータ

            // コンポーネント
            private Tab _model;

            #endregion

            public NWTab(NWControlHost host, Tab model)
                : base(host)
            {
                Debug.Assert(null != model, "unexpected error.");
                _model = model;
            }

            #region ** プロパティ

            public Tab Model
            {
                get { return _model; }
            }

            public Image Image
            {
                get { return _model.Image; }
            }

            public NWTabButton CloseButton
            {
                get { return Controls[CloseButtonName] as NWTabButton; }
            }

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new NWTabTemplateTop(this); }
            }

            private new NWTabControl Root
            {
                get { return base.Root as NWTabControl; }
            }

            #endregion

            #region ** イベント

            public event EventHandler CloseButtonClick;

            #endregion

            #region ** イベントハンドラ

            #region ** 初期化イベント

            protected override void OnInitialized(EventArgs e)
            {
                base.OnInitialized(e);

                // イベントハンドラを登録する
                CloseButton.Click += OnCloseButtonClicked;

                _model.PageChanged += OnPageChanged;
                _model.SelectedChanged += OnSelectedChanged;

                _model.Page.ImageChanged += OnPageImageChanged;
                _model.Page.TextChanged += OnPageTextChanged;
            }

            protected override void OnUninitialize(EventArgs e)
            {
                base.OnUninitialize(e);

                // イベントハンドラの登録を解除する
                CloseButton.Click -= OnCloseButtonClicked;

                _model.PageChanged -= OnPageChanged;
                _model.SelectedChanged -= OnSelectedChanged;

                _model.Page.ImageChanged -= OnPageImageChanged;
                _model.Page.TextChanged -= OnPageTextChanged;
            }

            #endregion

            #region ** ユーザ操作イベント

            protected override void OnMouseDown(MouseEventArgs e)
            {
                base.OnMouseDown(e);

                switch (e.Button)
                {
                    case MouseButtons.Left:
                        Root.Model.SelectedTab = _model;
                        _model.Page.Select();
                        break;

                    case MouseButtons.Middle:
                        Root.Model.TabPages.Remove(Model);
                        break;
                }
            }

            private void OnCloseButtonClicked(object sender, EventArgs e)
            {
                if (null != CloseButtonClick)
                {
                    CloseButtonClick(this, new EventArgs());
                }

                Root.Model.TabPages.Remove(_model);
            }

            #endregion

            private void OnPageChanged(object sender, EventArgs e)
            {
                Parent.PerformLayout();
            }

            private void OnSelectedChanged(object sender, EventArgs e)
            {
                Selected = _model.Selected;
            }

            private void OnPageImageChanged(object sender, EventArgs e)
            {
                Parent.PerformLayout();
            }

            private void OnPageTextChanged(object sender, EventArgs e)
            {
                Parent.PerformLayout();
            }

            #endregion
        }

        /// <summary>
        /// タブパネル
        /// </summary>
        public class NWTabPanel : NWControl
        {
            public const string ControlName = "TabPanel";

            #region ** フィールド

            private int _originIndex = InvalidIndex;    // 表示の基点となるタブインデックス

            private Dictionary<Tab, NWTab> _tabDictionary = new Dictionary<Tab, NWTab>();

            private NWControl[] _orderedTabs = new NWControl[] { };

            #endregion

            public NWTabPanel(NWControlHost host) : base(host, ControlName) { }

            #region ** プロパティ

            #region ** コントロール

            public ScrollButton ScrollUpButton
            {
                get { return NcControls[NWTabControl.ScrollLeftButton.ControlName] as ScrollButton; }
            }

            public ScrollButton ScrollDownButton
            {
                get { return NcControls[NWTabControl.ScrollRightButton.ControlName] as ScrollButton; }
            }

            public DropDownButton DropDownButton
            {
                get { return NcControls[NWTabControl.DropDownButton.ControlName] as DropDownButton; }
            }

            #endregion

            public int ScrollOriginIndex
            {
                get { return _originIndex; }
                set { SetScrollIndex(value); }
            }

            public new INWTabPanelTemplate Template
            {
                get { return base.Template as INWTabPanelTemplate; }
            }

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new NWTabPanelTemplateTop(this); }
            }

            protected override Type TemplateType
            {
                get { return typeof(INWTabPanelTemplate); }
            }

            protected override IList<NWControl> OrderedControls
            {
                get
                {
                    if (null == Core.SelectedTab || 0 == Controls.Count)
                    {
                        return base.OrderedControls;
                    }

                    // 選択タブを前面にもってくる
                    _orderedTabs[0] = _tabDictionary[Core.SelectedTab];
                    return _orderedTabs;
                }
            }

            private Core Core
            {
                get { return Root.Model; }
            }

            private new NWTabControl Root
            {
                get { return base.Root as NWTabControl; }
            }

            private bool EnabledScrollUp
            {
                get
                {
                    if (!ScrollUpButton.Visible) { return false; }

                    Debug.Assert(Controls.Count > 0, "unexpected error.");
                    return (Controls[0].Left < ScrollPosition.X) || (Controls[0].Top < ScrollPosition.Y);
                }
            }

            private bool EnabledScrollDown
            {
                get
                {
                    if (!ScrollDownButton.Visible) { return false; }
                    if (_originIndex == Controls.Count - 1) { return false; }


                    Debug.Assert(Controls.Count > 0, "unexpected error.");

                    NWTab lastTab = Controls[Controls.Count - 1] as NWTab;
                    Debug.Assert(null != lastTab, "unexpected error.");

                    return (ClientRectangle.Width < lastTab.Right - ScrollPosition.X) ||
                           (ClientRectangle.Height < lastTab.Bottom - ScrollPosition.Y);
                }
            }

            #endregion

            #region ** イベントハンドラ

            #region ** 初期化イベント

            protected override void OnInitialize()
            {
                base.OnInitialize();

                // タブを追加する
                int index = 0;

                foreach (Tab tab in Root.Model.TabPages)
                {
                    AddTab(tab, index);
                    index++;
                }
            }

            protected override void OnInitialized(EventArgs e)
            {
                base.OnInitialized(e);

                // イベントハンドラを登録する
                ScrollUpButton.Click += OnScrollUpButtonClicked;
                ScrollDownButton.Click += OnScrollDownButtonClicked;
                DropDownButton.Click += OnDropDownButtonClicked;

                Root.Model.ItemInserted += OnCoreAdded;
                Root.Model.ItemRemoved += OnCoreRemoved;
                Root.Model.ItemSwapped += OnCoreSwapped;
            }

            protected override void OnUninitialize(EventArgs e)
            {
                base.OnUninitialize(e);

                // イベントハンドラの登録を解除する
                ScrollUpButton.Click -= OnScrollUpButtonClicked;
                ScrollDownButton.Click -= OnScrollDownButtonClicked;
                DropDownButton.Click -= OnDropDownButtonClicked;
            }

            protected override void OnUninitialized(EventArgs e)
            {
                base.OnUninitialized(e);

                Root.Model.ItemInserted -= OnCoreAdded;
                Root.Model.ItemRemoved -= OnCoreRemoved;
                Root.Model.ItemSwapped -= OnCoreSwapped;
            }

            #endregion

            #region ** アイテム操作イベント

            private void OnCoreAdded(object sender, NWTabControlEventArgs e)
            {
                // タブを追加する
                NWTab newTab = AddTab(e.Tab, e.TabIndex);
                newTab.InitializeComponent();

                // レイアウトを更新する
                Parent.PerformLayout();
            }

            private void OnCoreRemoved(object sender, NWTabControlEventArgs e)
            {
                Debug.Assert(_tabDictionary.ContainsKey(e.Tab), "unexpected error.");

                NWTab tab = _tabDictionary[e.Tab];

                // タブを削除する
                Controls.Remove(tab);
                _tabDictionary.Remove(e.Tab);

                tab.UninitializeComponent();


                // レイアウトを更新する
                Parent.PerformLayout();

                // スクロールを更新する
                ValidateScrollOrigin();
            }

            private void OnCoreSwapped(object sender, NWTabControlSwapEventArgs e)
            {
                NWTab tab1 = _tabDictionary[e.Tab];
                NWTab tab2 = _tabDictionary[e.Tab2];

                Controls.Swap(Controls.IndexOf(tab1), Controls.IndexOf(tab2));
                PerformLayout();
            }

            #endregion

            #region ** レイアウトイベント

            protected override void OnNcLayout(NWControlNcLayoutEventArgs e)
            {
                base.OnNcLayout(e);
                UpdateScrollState();
            }

            #endregion

            #region ** スクロールイベント

            private void OnScrollUpButtonClicked(object sender, EventArgs e)
            {
                if (!EnabledScrollUp) { return; }
                SetScrollIndex(_originIndex - 1);
            }

            private void OnScrollDownButtonClicked(object sender, EventArgs e)
            {
                if (!EnabledScrollDown) { return; }
                SetScrollIndex(_originIndex + 1);
            }

            #endregion

            #region ** ドロップダウンイベント

            private void OnDropDownButtonClicked(object sender, EventArgs e)
            {
                if (Core.TabPages.Count == 0) { return; }

                ContextMenuStrip menu = new ContextMenuStrip();

                foreach (Tab tab in Core.TabPages)
                {

                    ToolStripMenuItem menuItem = new ToolStripMenuItem(tab.Text, tab.Image, OnDropDownMenuClicked);
                    menuItem.Tag = tab;

                    menu.Items.Add(menuItem);

                }

                // 表示するメニューを更新する
                DropDownButton.Menu = menu;
            }

            private void OnDropDownMenuClicked(object sender, EventArgs e)
            {
                ToolStripMenuItem menuItem = sender as ToolStripMenuItem;
                Debug.Assert(null != menuItem, "unexpected error.");

                Tab tab = menuItem.Tag as Tab;
                Debug.Assert(null != tab, "unexpected error.");

                Core.SelectedTab = tab;
                EnsureVisible(tab);
            }

            #endregion

            protected override void OnControlInserted(NWControlEventArgs e)
            {
                base.OnControlInserted(e);

                // オーダーを更新する
                UpdateOrderedTabs();
            }

            protected override void OnControlRemoved(NWControlEventArgs e)
            {
                base.OnControlRemoved(e);

                // オーダーを更新する
                UpdateOrderedTabs();
            }

            #endregion

            #region ** メソッド

            /// <summary>
            /// 指定タブが表示されるように最小限のスクロールを行います。
            /// </summary>
            /// <param name="tab">タブ</param>
            public void EnsureVisible(Tab tab)
            {
                EnsureVisible(_tabDictionary[tab]);
            }

            /// <summary>
            /// 指定タブが表示されるように最小限のスクロールを行います。
            /// </summary>
            /// <param name="index">タブのインデックス</param>
            public void EnsureVisible(NWTab tab)
            {
                Debug.Assert(null != tab, "unexpected error.");

                if (IsScrollLocked) { return; }

                Rectangle scrollRect = ScrollRectangle;
                scrollRect.Offset(-ClientRectangle.Left, -ClientRectangle.Top);

                // スクロールする必要がない場合は終了
                if (scrollRect.Left <= tab.Left && tab.Right <= scrollRect.Right) { return; }

                // タブサイズが表示領域より大きい場合は、
                // 対象タブが基点となるようにスクロールする
                if (tab.Width > scrollRect.Width)
                {
                    SetScrollIndex(Root.Model.TabPages.IndexOf(tab.Model));
                    return;
                }

                // 左にスクロールする必要がある場合は、
                // 対象タブが基点となるようにスクロールする
                if (tab.Left < scrollRect.Left)
                {
                    SetScrollIndex(Root.Model.TabPages.IndexOf(tab.Model));
                    return;
                }

                // 以下、右にスクロールする必要がある場合
                // 対象タブが完全に見えるようになるのに最小の移動距離を計算する
                int scrollTargetIndex = _originIndex + 1;

                while (scrollTargetIndex < Controls.Count)
                {
                    if (scrollTargetIndex == _originIndex) { break; }
                    if (tab.Right - Controls[scrollTargetIndex].Left < scrollRect.Width)
                    {
                        break;
                    }

                    scrollTargetIndex++;
                }

                Debug.Assert(scrollTargetIndex < Controls.Count, "unexpected error.");

                SetScrollIndex(scrollTargetIndex);
            }

            private void ValidateScrollOrigin()
            {
                if (0 == Controls.Count)
                {
                    ScrollPosition = Point.Empty;
                    return;
                }

                Rectangle scrollRect = ScrollRectangle;
                scrollRect.Offset(-ClientRectangle.Left, -ClientRectangle.Top);

                int targetIndex = Controls.Count - 1;
                NWTab lastTab = Controls[targetIndex] as NWTab;

                if (lastTab.Width < scrollRect.Width)
                {

                    while (targetIndex > 0)
                    {
                        if (lastTab.Right - Controls[targetIndex - 1].Left > scrollRect.Width)
                        {
                            break;
                        }

                        targetIndex--;
                    }

                }

                Debug.Assert(targetIndex >= 0, "unexpected error.");


                if (_originIndex > targetIndex)
                {
                    SetScrollIndex(targetIndex);
                }
            }

            private void SetScrollIndex(int index)
            {
                if (index == _originIndex) { return; }
                Debug.Assert(index < Controls.Count, "unexpected error.");

                if (IsScrollLocked) { return; }

                _originIndex = index;
                if (InvalidIndex == _originIndex) { return; }

                Point newScrollPosition = new Point(Controls[_originIndex].Left, Controls[_originIndex].Top);
                if (2 >= newScrollPosition.X) { newScrollPosition.X = 0; }
                if (2 >= newScrollPosition.Y) { newScrollPosition.Y = 0; }

                ScrollPosition = newScrollPosition;
                UpdateScrollButtons();
            }

            /// <summary>
            /// スクロール状態を更新します。
            /// </summary>
            private void UpdateScrollState()
            {
                if (!EnabledScrollUp && !EnabledScrollDown)
                {
                    ScrollOriginIndex = (Controls.Count > 0) ? 0 : InvalidIndex;
                }
                else
                {
                    UpdateScrollButtons();
                }
            }

            /// <summary>
            /// スクロールボタンの状態を更新します。
            /// </summary>
            private void UpdateScrollButtons()
            {
                ScrollUpButton.Enabled = EnabledScrollUp;
                ScrollDownButton.Enabled = EnabledScrollDown;
            }

            private NWTab AddTab(Tab tab, int tabIndex)
            {
                if (null == tab) { return null; }

                // 新しいタブを生成する
                NWTab newTab = Template.CreateTabInstance(tab) as NWTab;

                if (tab.Selected)
                {
                    newTab.Selected = true;
                }

                Controls.Insert(tabIndex, newTab);
                _tabDictionary.Add(tab, newTab);

                return newTab;
            }

            private void UpdateOrderedTabs()
            {
                _orderedTabs = new NWControl[Controls.Count + 1];
                Controls.CopyTo(_orderedTabs, 1);
            }

            #endregion
        }

        /// <summary>
        /// タブページパネル
        /// </summary>
        public class NWTabPagePanel : NWControl
        {
            public const string ControlName = "TabPagePanel";

            public NWTabPagePanel(NWControlHost host) : base(host, ControlName) { }
        }

        /// <summary>
        /// ドロップダウンボタン
        /// </summary>
        public class DropDownButton : NWNCButton
        {
            public const string ControlName = "DropDownButton";

            #region ** パラメータ

            // UIコンポーネント
            private ContextMenuStrip _menu = null;

            // 状態
            private bool _menuVisibled = false;

            #endregion

            public DropDownButton(NWControlHost host) : base(host, ControlName) { }

            #region ** プロパティ

            public ContextMenuStrip Menu
            {
                get { return _menu; }
                set
                {
                    if (value == _menu) { return; }

                    if (null != _menu)
                    {
                        _menu.Hide();
                    }

                    _menu = value;

                    if (null != value)
                    {
                        value.VisibleChanged += OnMenuVisibleChanged;
                    }
                }
            }

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new NWDropDownButtonTemplate(this); }
            }

            #endregion

            #region ** イベントハンドラ

            private void OnMenuVisibleChanged(object sender, EventArgs e)
            {
                if (null == _menu || !_menu.Visible)
                {
                    base.SetState(NWControlState.Normal);
                }
            }

            #region ** NWControl イベントハンドラのオーバーライド

            protected override void OnMouseActivating(EventArgs e)
            {
                base.OnMouseActivating(e);

                if (null == _menu)
                {
                    _menuVisibled = false;
                }
                else
                {
                    _menuVisibled = _menu.Visible;
                }
            }

            protected override void OnClicked(EventArgs e)
            {
                //Debug.WriteLine( "DropDownButton.Clicked" );

                base.OnClicked(e);
                if (null == _menu) { return; }

                if (_menuVisibled)
                {
                    base.SetState(NWControlState.Normal);
                    return;
                }

                Rectangle bounds = AbsoluteBounds;
                _menu.Show(Host.Owner, new Point(bounds.Right, bounds.Bottom), ToolStripDropDownDirection.BelowLeft);
                _menuVisibled = true;
            }

            #endregion

            #endregion

            #region ** メソッド

            protected override void SetState(NWControlState state)
            {
                if (null != _menu && _menu.Visible) { return; }
                base.SetState(state);
            }

            #endregion
        }

        /// <summary>
        /// スクロールボタン
        /// </summary>
        public class ScrollButton : NWNCButton
        {
            public ScrollButton(NWControlHost host, string name) : base(host, name) { }

            #region ** プロパティ

            protected override INWControlTemplate DefaultTemplate
            {
                get { return new NWTabScrollButtonTemplate(this); }
            }

            #endregion
        }

        /// <summary>
        /// スクロールダウンボタン
        /// </summary>
        public class ScrollRightButton : ScrollButton
        {
            public const string ControlName = "ScrollDownButton";

            public ScrollRightButton(NWControlHost host) : base(host, ControlName) { }
        }

        /// <summary>
        /// スクロールアップボタン
        /// </summary>
        public class ScrollLeftButton : ScrollButton
        {
            public const string ControlName = "ScrollUpButton";

            public ScrollLeftButton(NWControlHost host) : base(host, ControlName) { }
        }

        /// <summary>
        /// スクロールダウンボタン
        /// </summary>
        public class ScrollDownButton : ScrollButton
        {
            public const string ControlName = "ScrollDownButton";

            public ScrollDownButton(NWControlHost host) : base(host, ControlName) { }
        }

        /// <summary>
        /// スクロールアップボタン
        /// </summary>
        public class ScrollUpButton : ScrollButton
        {
            public const string ControlName = "ScrollUpButton";

            public ScrollUpButton(NWControlHost host) : base(host, ControlName) { }
        }

        #endregion
    }

    #region ** イベント

    public delegate void NWTabControlEventHandler(object sender, NWTabControlEventArgs e);
    public delegate void NWTabControlCancelEventHandler(object sender, NWTabControlCancelEventArgs e);
    public delegate void NWTabControlSwapEventHandler(object sender, NWTabControlSwapEventArgs e);
    public delegate void NWTabControlDragEventHandler(object sender, NWTabControlDragEventArgs e);

    /// <summary>
    /// NWTabControl イベントのデータを提供します。
    /// </summary>
    public class NWTabControlEventArgs : EventArgs
    {
        #region ** パラメータ

        private NWTabControl.Tab _tab = null;
        private int _tabIndex = NWTabControl.InvalidIndex;
        private NWTabControl.Action _action = NWTabControl.Action.Selected;
        private NWTabControl.ActionTrigger _actionTrigger = NWTabControl.ActionTrigger.Unknown;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="tab">イベントの処理対象となるタブ</param>
        /// <param name="tabIndex">イベントの処理対象となるタブのインデックス</param>
        /// <param name="action">アクションイベントの種類</param>
        /// <param name="actionTrigger">イベント発生の切欠となったトリガーの種類</param>
        public NWTabControlEventArgs(NWTabControl.Tab tab, int tabIndex, NWTabControl.Action action,
                                     NWTabControl.ActionTrigger actionTrigger)
        {
            _tab = tab;
            _tabIndex = tabIndex;
            _action = action;
            _actionTrigger = actionTrigger;
        }

        #region ** プロパティ

        /// <summary>
        /// アクションイベントの種類を取得します。
        /// </summary>
        public NWTabControl.Action Action
        {
            get { return _action; }
        }

        /// <summary>
        /// イベント発生の切欠となったアクションの種類を取得します。
        /// </summary>
        public NWTabControl.ActionTrigger ActionTrigger
        {
            get { return _actionTrigger; }
        }

        /// <summary>
        /// 発生しているイベントの対象となるタブページを取得します。
        /// </summary>
        public NWTabControl.Tab Tab
        {
            get { return _tab; }
        }

        /// <summary>
        /// 対象タブページのインデックス
        /// </summary>
        public int TabIndex
        {
            get { return _tabIndex; }
        }

        #endregion
    }

    /// <summary>
    /// キャンセル可能な NWTabControl イベントのデータを提供します。
    /// </summary>
    public class NWTabControlCancelEventArgs : CancelEventArgs
    {
        #region ** パラメータ

        private NWTabControl.Tab _tab = null;
        private int _tabIndex = NWTabControl.InvalidIndex;
        private NWTabControl.Action _action = NWTabControl.Action.Selected;
        private NWTabControl.ActionTrigger _actionTrigger = NWTabControl.ActionTrigger.Unknown;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="tab">イベントの処理対象となるタブ</param>
        /// <param name="tabIndex">イベントの処理対象となるタブのインデックス</param>
        /// <param name="action">アクションイベントの種類</param>
        /// <param name="actionTrigger">イベント発生の切欠となったトリガーの種類</param>
        public NWTabControlCancelEventArgs(NWTabControl.Tab tab, int tabIndex, NWTabControl.Action action,
                                            NWTabControl.ActionTrigger actionTrigger)
            : base(false)
        {
            _tab = tab;
            _tabIndex = tabIndex;
            _action = action;
            _actionTrigger = actionTrigger;
        }

        protected NWTabControlCancelEventArgs(bool cancel) : base(cancel) { }

        #region ** プロパティ

        /// <summary>
        /// アクションイベントの種類を取得します。
        /// </summary>
        public NWTabControl.Action Action
        {
            get { return _action; }
        }

        /// <summary>
        /// イベント発生の切欠となったアクションの種類を取得します。
        /// </summary>
        public NWTabControl.ActionTrigger ActionTrigger
        {
            get { return _actionTrigger; }
        }

        /// <summary>
        /// 発生しているイベントの対象となるタブページを取得します。
        /// </summary>
        public NWTabControl.Tab Tab
        {
            get { return _tab; }
        }

        /// <summary>
        /// 対象タブページのインデックス
        /// </summary>
        public int TabIndex
        {
            get { return _tabIndex; }
        }

        #endregion
    }

    /// <summary>
    /// NWTabControl Swappedイベントのデータを提供します。
    /// </summary>
    public class NWTabControlSwapEventArgs : NWTabControlEventArgs
    {
        #region ** パラメータ

        private NWTabControl.Tab _tab2 = null;
        private int _tab2Index = NWTabControl.InvalidIndex;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="tab1">イベントの処理対象となるタブ1</param>
        /// <param name="tab1Index">イベントの処理対象となるタブ1のインデックス</param>
        /// <param name="tab2">イベントの処理対象となるタブ2</param>
        /// <param name="tab2Index">イベントの処理対象となるタブ2のインデックス</param>
        /// <param name="actionTrigger">イベント発生の切欠となったトリガーの種類</param>
        public NWTabControlSwapEventArgs(NWTabControl.Tab tab1, int tab1Index, NWTabControl.Tab tab2, int tab2Index,
                                         NWTabControl.ActionTrigger actionTrigger)
            : base(tab1, tab1Index, NWTabControl.Action.Swapped, actionTrigger)
        {
            _tab2 = tab2;
            _tab2Index = tab2Index;
        }

        #region ** プロパティ

        /// <summary>
        /// 発生しているイベントの対象となるタブページを取得します。
        /// </summary>
        public NWTabControl.Tab Tab2
        {
            get { return _tab2; }
        }

        /// <summary>
        /// 対象タブページのインデックス
        /// </summary>
        public int Tab2Index
        {
            get { return _tab2Index; }
        }

        #endregion
    }

    public class NWTabControlDragEventArgs : DragEventArgs
    {
        private NWTabControl.Tab _tab = null;

        public NWTabControlDragEventArgs(DragEventArgs e)
            : base(e.Data, e.KeyState, e.X, e.Y, e.AllowedEffect, e.Effect)
        {
            _tab = e.Data.GetData(typeof(NWTabControl.Tab)) as NWTabControl.Tab;

            if (null != _tab)
            {
                Effect = DragDropEffects.Move;
            }
            else
            {
                Effect = DragDropEffects.None;
            }
        }

        #region ** プロパティ

        public NWTabControl.Tab Tab
        {
            get { return _tab; }
            set { _tab = value; }
        }

        #endregion
    }

    #endregion
}
