﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;


namespace NintendoWare.SoundFoundation.Windows.Forms.Windowless
{
    public partial class NWTabControl
    {
        #region ** コアコンポーネント

        public class Core
        {
            #region ** パラメータ

            // リンク
            private NWControlHost _host;

            // コンポーネント
            private TabCollection _tabs = null;
            private ImageList _imageList = null;

            // 状態
            private NWTabControl.ActionTrigger _actionTrigger = NWTabControl.ActionTrigger.Unknown;

            private Tab _selectedTab = null;
            private Tab _incomingTab = null;
            private bool _selectionDirty = false;
            private int _eventLockCount = 0;

            #endregion

            public Core(NWControlHost host)
            {
                Debug.Assert(null != host, "unexpected error.");

                _host = host;
                _tabs = new TabCollection();

                _tabs.ItemInserted += OnTabPageInserted;
                _tabs.ItemRemoving += OnTabPageRemoving;
                _tabs.ItemRemoved += OnTabPageRemoved;
                _tabs.ItemSwapped += OnTabPageSwapped;
            }

            #region ** プロパティ

            public NWControlHost Host
            {
                get { return _host; }
                set { _host = value; }
            }

            public TabCollection TabPages
            {
                get { return _tabs; }
            }

            public ImageList ImageList
            {
                get { return _imageList; }
                set { _imageList = value; }
            }

            public int SelectedIndex
            {
                get
                {
                    if (null == SelectedTab) { return InvalidIndex; }
                    return _tabs.IndexOf(SelectedTab);
                }
            }

            public Tab SelectedTab
            {
                get
                {
                    if (_eventLockCount > 0) { return _incomingTab; }
                    return _selectedTab;
                }
                set
                {
                    Select(value, true);
                }
            }

            public NWTabControl.ActionTrigger CurrentAction
            {
                get { return _actionTrigger; }
                set { _actionTrigger = value; }
            }

            protected internal bool RaiseEventLocked
            {
                get { return (0 < _eventLockCount); }
            }

            #endregion

            #region ** イベント

            public event NWTabControlEventHandler ItemInserted;
            public event NWTabControlEventHandler ItemRemoved;
            public event NWTabControlSwapEventHandler ItemSwapped;

            /// <summary>
            /// タブが選択されたときに発生します。
            /// </summary>
            public event NWTabControlCancelEventHandler ItemClosing;

            /// <summary>
            /// タブが選択されたときに発生します。
            /// </summary>
            public event NWTabControlEventHandler ItemClosed;

            /// <summary>
            /// タブが選択される前に発生し、ハンドラがタブの変更をキャンセルできるようにします。
            /// </summary>
            public event NWTabControlCancelEventHandler ItemSelecting;

            /// <summary>
            /// タブが選択されたときに発生します。
            /// </summary>
            public event NWTabControlEventHandler ItemSelected;

            /// <summary>
            /// タブの選択が解除されたときに発生します。
            /// </summary>
            public event NWTabControlCancelEventHandler ItemDeselecting;

            /// <summary>
            /// タブの選択が解除される前に発生し、ハンドラがタブの変更をキャンセルできるようにします。
            /// </summary>
            public event NWTabControlEventHandler ItemDeselected;

            /// <summary>
            /// SelectedIndex プロパティが変更されると発生します。
            /// </summary>
            public event EventHandler SelectedIndexChanged;

            public event NWTabControlEventHandler SelectedTabChanging;

            public event EventHandler SelectedTabChanged;

            #endregion

            #region ** イベントハンドラ

            #region ** Core イベント

            protected virtual void OnInserted(NWTabControlEventArgs e)
            {
                if (null != ItemInserted)
                {
                    ItemInserted(this, e);
                }
            }

            protected virtual void OnRemoved(NWTabControlEventArgs e)
            {
                if (null != ItemRemoved)
                {
                    ItemRemoved(this, e);
                }
            }

            protected virtual void OnSwapped(NWTabControlSwapEventArgs e)
            {
                if (null != ItemSwapped)
                {
                    ItemSwapped(this, e);
                }
            }

            protected virtual void OnClosing(NWTabControlCancelEventArgs e)
            {
                if (null != ItemClosing)
                {
                    ItemClosing(this, e);
                }
            }

            protected virtual void OnClosed(NWTabControlEventArgs e)
            {
                if (null != ItemClosed)
                {
                    ItemClosed(this, e);
                }
            }

            protected virtual void OnSelecting(NWTabControlCancelEventArgs e)
            {
                if (null != ItemSelecting)
                {
                    ItemSelecting(this, e);
                }
            }

            protected virtual void OnSelected(NWTabControlEventArgs e)
            {
                if (null != ItemSelected)
                {
                    ItemSelected(this, e);
                }
            }

            protected virtual void OnDeselecting(NWTabControlCancelEventArgs e)
            {
                if (null != ItemDeselecting)
                {
                    ItemDeselecting(this, e);
                }
            }

            protected virtual void OnDeselected(NWTabControlEventArgs e)
            {
                if (null != ItemDeselected)
                {
                    ItemDeselected(this, e);
                }
            }

            protected virtual void OnSelectedIndexChanged(EventArgs e)
            {
                if (null != SelectedIndexChanged)
                {
                    SelectedIndexChanged(this, e);
                }
            }

            protected virtual void OnSelectedTabChanging(NWTabControlEventArgs e)
            {
                if (null != SelectedTabChanging)
                {
                    SelectedTabChanging(this, e);
                }
            }

            protected virtual void OnSelectedTabChanged(EventArgs e)
            {
                if (null != SelectedTabChanged)
                {
                    SelectedTabChanged(this, e);
                }
            }

            #endregion

            #region ** コレクションイベント

            private void OnTabPageInserted(object sender, CoreEventArgs e)
            {
                e.Tab.Page.Attach(Host);

                SuspendRaiseEvent();

                if (InvalidIndex == SelectedIndex)
                {
                    AutoSelect(null, InvalidIndex);
                }

                OnInserted(new NWTabControlEventArgs(e.Tab, e.TabIndex,
                                                        NWTabControl.Action.Inserted, _actionTrigger));

                ResumeRaiseEvent(true);
            }

            private void OnTabPageRemoving(object sender, CoreCancelEventArgs e)
            {
                Debug.Assert(_tabs.Count > 0, "unexpected error.");

                // Closing イベント
                NWTabControlCancelEventArgs closingEvent = new NWTabControlCancelEventArgs(e.Tab, e.TabIndex,
                                                                                            NWTabControl.Action.Closing,
                                                                                            _actionTrigger);
                OnClosing(closingEvent);

                e.Cancel = closingEvent.Cancel;
                if (e.Cancel) { return; }


                SuspendRaiseEvent();

                // 最後のアイテムの場合、選択解除する（キャンセル不可）
                if (_tabs.Count == 1)
                {
                    Select(null, true);
                }
            }

            private void OnTabPageRemoved(object sender, CoreEventArgs e)
            {
                e.Tab.Page.Detach();

                int oldSelectedIndex = InvalidIndex;

                // 選択タブが削除された場合、他のタブを自動選択する
                if (e.Tab == _selectedTab)
                {

                    e.Tab.Selected = false;
                    AutoSelect(e.Tab, e.TabIndex);

                    oldSelectedIndex = e.TabIndex;

                }

                OnRemoved(new NWTabControlEventArgs(e.Tab, e.TabIndex,
                                                        NWTabControl.Action.Removed, _actionTrigger));

                ResumeRaiseEvent(oldSelectedIndex, true);


                // Close完了イベント
                NWTabControlEventArgs closingEvent = new NWTabControlEventArgs(e.Tab, e.TabIndex,
                                                                                NWTabControl.Action.Closed,
                                                                                _actionTrigger);
                OnClosed(closingEvent);
            }

            private void OnTabPageSwapped(object sender, CoreSwapEventArgs e)
            {
                OnSwapped(new NWTabControlSwapEventArgs(e.Tab, e.TabIndex, e.Tab2, e.Tab2Index, _actionTrigger));
            }

            #endregion

            #endregion

            #region ** メソッド

            #region ** イベントロック

            protected internal void SuspendRaiseEvent()
            {
                _eventLockCount++;
                _incomingTab = _selectedTab;
            }

            protected internal void ResumeRaiseEvent(bool raiseTabEvent)
            {
                ResumeRaiseEvent(InvalidIndex, raiseTabEvent);
            }

            private void ResumeRaiseEvent(int oldIndex, bool raiseTabEvent)
            {
                if (_eventLockCount > 0)
                {

                    _eventLockCount--;
                    if (_eventLockCount > 0) { return; }

                }

                if (!_selectionDirty) { return; }

                Select(_incomingTab, oldIndex, raiseTabEvent);

                if (null != _incomingTab)
                {
                    _incomingTab.Page.Show();
                }
            }

            #endregion

            #region ** タブ操作

            public void SelectNextTab()
            {
                if (_tabs.Count <= 1) { return; }

                int nextIndex = SelectedIndex + 1;

                if (nextIndex >= TabPages.Count)
                {
                    nextIndex = 0;
                }

                SelectedTab = _tabs[nextIndex];
            }

            public void SelectPrevTab()
            {
                if (_tabs.Count <= 1) { return; }

                int prevIndex = SelectedIndex - 1;

                if (prevIndex < 0)
                {
                    prevIndex = _tabs.Count - 1;
                }

                SelectedTab = _tabs[prevIndex];
            }

            public void CloseSelectedTab()
            {
                if (_tabs.Count == 0) { return; }
                Debug.Assert(InvalidIndex != SelectedIndex, "unexpected error.");

                _tabs.RemoveAt(SelectedIndex);
            }

            /// <summary>
            /// タブを選択します。
            /// </summary>
            /// <param name="index">選択するタブのインデックス</param>
            /// <param name="raiseTabEvent">タブ変更イベントを発行する場合は true、発行しない場合は false。</param>
            /// <returns>指定タブが選択された場合は true、キャンセルされた場合は false。</returns>
            private bool Select(Tab tab, bool raiseTabEvent)
            {
                return Select(tab, InvalidIndex, raiseTabEvent);
            }

            /// <summary>
            /// タブを選択します。
            /// </summary>
            /// <param name="index">選択するタブのインデックス</param>
            /// <param name="oldTabIndex">古い選択タブのインデックス</param>
            /// <param name="raiseTabEvent">タブ変更イベントを発行する場合は true、発行しない場合は false。</param>
            /// <returns>指定タブが選択された場合は true、キャンセルされた場合は false。</returns>
            private bool Select(Tab tab, int oldTabIndex, bool raiseTabEvent)
            {
                // ロック中は処理しない
                if (_eventLockCount > 0)
                {
                    _selectionDirty = true;
                    _incomingTab = tab;
                    return true;
                }

                _selectionDirty = false;

                if (_selectedTab == tab) { return true; }


                Tab oldTab = _selectedTab;
                int oldIndex = (InvalidIndex == oldTabIndex) ? SelectedIndex : oldTabIndex;

                if (raiseTabEvent)
                {

                    if (!RaiseSelectingEvent(oldTab, tab))
                    {
                        _incomingTab = _selectedTab;
                        return false;
                    }

                }

                // 選択
                SelectTab(tab);

                if (raiseTabEvent)
                {
                    RaiseSelectedEvent(oldTab, oldIndex, tab);
                }

                return true;
            }

            private void AutoSelect(Tab oldTab, int oldIndex)
            {
                if (0 == _tabs.Count) { return; }

                if (1 == _tabs.Count || InvalidIndex == oldIndex)
                {
                    Select(_tabs[0], true);
                    return;
                }

                if (_tabs.Count <= oldIndex)
                {
                    Select(_tabs[_tabs.Count - 1], true);
                }
                else
                {
                    Select(_tabs[oldIndex], true);
                }
            }

            /// <summary>
            /// 指定タブを選択します。
            /// </summary>
            /// <param name="tab">選択するタブ</param>
            /// <param name="tabIndex">選択するタブのインデックス</param>
            /// <param name="raiseIndexEvent">インデックス変更イベントを発行する場合は true、発行しない場合は false。</param>
            /// <param name="raiseTabEvent">タブ変更イベントを発行する場合は true、発行しない場合は false。</param>
            /// <returns>指定タブが選択された場合は true、キャンセルされた場合は false。</returns>
            private void SelectTab(Tab tab)
            {
                Debug.Assert(0 == _eventLockCount, "unexpected error.");

                Tab oldTab = _selectedTab;

                if (null != _selectedTab)
                {
                    _selectedTab.Selected = false;
                }

                _selectedTab = tab;
                if (null == _selectedTab) { return; }


                _selectedTab.Selected = true;

                // タブページを表示する
                _selectedTab.Page.Show();

                if (null != oldTab)
                {
                    oldTab.Page.Hide();
                }
            }

            private bool RaiseSelectingEvent(Tab oldTab, Tab newTab)
            {
                int oldIndex = (null == oldTab) ? InvalidIndex : _tabs.IndexOf(oldTab);
                int newIndex = (null == newTab) ? InvalidIndex : _tabs.IndexOf(newTab);

                // 選択解除イベント
                NWTabControlCancelEventArgs deselectingEvent = new NWTabControlCancelEventArgs(oldTab, oldIndex,
                                                                                NWTabControl.Action.Deselecting,
                                                                                _actionTrigger);
                OnDeselecting(deselectingEvent);
                if (deselectingEvent.Cancel) { return false; }


                // 選択イベント
                if (null != newTab)
                {
                    NWTabControlCancelEventArgs selectingEvent = new NWTabControlCancelEventArgs(newTab, newIndex,
                                                                                    NWTabControl.Action.Selecting,
                                                                                    _actionTrigger);
                    OnSelecting(selectingEvent);
                    if (selectingEvent.Cancel && _tabs.Count > 1) { return false; }
                }


                // 選択タブ変更イベント
                OnSelectedTabChanging(new NWTabControlEventArgs(newTab, newIndex,
                                                        NWTabControl.Action.Selecting, _actionTrigger));

                return true;
            }

            private bool RaiseSelectedEvent(Tab oldTab, int oldIndex, Tab newTab)
            {
                int newIndex = (null == newTab) ? InvalidIndex : _tabs.IndexOf(newTab);

                // 選択解除イベント
                NWTabControlEventArgs deselectedEvent = new NWTabControlEventArgs(oldTab, oldIndex,
                                                                                NWTabControl.Action.Deselected,
                                                                                _actionTrigger);
                OnDeselected(deselectedEvent);


                // 選択イベント
                if (null != newTab)
                {
                    NWTabControlEventArgs selectedEvent = new NWTabControlEventArgs(newTab, newIndex,
                                                                                    NWTabControl.Action.Selected,
                                                                                    _actionTrigger);
                    OnSelected(selectedEvent);
                }


                // 選択タブ変更イベント
                OnSelectedTabChanged(new EventArgs());


                // 選択インデックス変更イベント
                if (oldIndex != newIndex)
                {
                    OnSelectedIndexChanged(new EventArgs());
                }

                return true;
            }

            #endregion

            #region ** デバッグ
            private void WriteInformations()
            {
#if DEBUG && false
                Debug.WriteLine( "** NTabControl Informations ------------------" );
                Debug.WriteLine( string.Format( "* [Selected    ] {0}", _selectedIndex ) );

                Debug.WriteLine( "* - tabs -" );

                int index = 0;

                foreach( INTabPage tab in _tabPages ) {

                    Debug.WriteLine( string.Format( "*  [{0:D}] {1, -8} {2, -36} {3}",
                                                    index,
                                                    tab.Tab.State.ToString(),
                                                    tab.Tab.AbsoluteBounds.ToString(),
                                                    tab.Tab.Page.Text ) );

                    index++;

                }

                Debug.WriteLine( "** --------------------------------------------" );
#endif
            }

            #endregion

            #endregion
        }

        public interface ITabPage
        {
            #region ** プロパティ

            /// <summary>
            /// タブコントロールに関連付けれているかどうかを取得します。
            /// </summary>
            bool Attached { get; }

            /// <summary>
            /// タブページの領域を取得または設定します。
            /// </summary>
            Rectangle Bounds { get; set; }

            /// <summary>
            /// タブに表示されるテキストを取得します。
            /// </summary>
            string Text { get; }

            /// <summary>
            /// このタブのツールヒントのテキストを取得します。
            /// </summary>
            string ToolTipText { get; }

            /// <summary>
            /// イメージを取得します。
            /// </summary>
            Image Image { get; }

            /// <summary>
            /// タブページの表示状態を取得します。
            /// </summary>
            bool Visible { get; }

            #endregion

            #region ** イベント

            event EventHandler TextChanged;
            event EventHandler ImageChanged;

            #endregion

            #region ** メソッド

            void Attach(NWControlHost host);
            void Detach();

            void Select();

            void Show();
            void Hide();

            #endregion
        }

        public class Tab
        {
            private ITabPage _page = null;

            // 状態
            private bool _selected = false;

            public Tab(ITabPage page)
            {
                if (null == page) { throw new Exception(); }
                _page = page;
            }

            #region ** プロパティ

            public ITabPage Page
            {
                get { return _page; }
                set
                {
                    if (value == _page) { return; }

                    _page = value;
                    OnPageChanged(new EventArgs());
                }
            }

            public string Text
            {
                get { return _page.Text; }
            }

            public string ToolTipText
            {
                get { return _page.ToolTipText; }
            }

            public Image Image
            {
                get { return _page.Image; }
            }

            public bool Selected
            {
                get { return _selected; }
                set
                {
                    if (value == _selected) { return; }

                    _selected = value;
                    OnSelectedChanged(new EventArgs());
                }
            }

            #endregion

            #region ** イベント

            public event EventHandler PageChanged;
            public event EventHandler SelectedChanged;

            #endregion

            #region ** イベントハンドラ

            protected virtual void OnPageChanged(EventArgs e)
            {
                if (null != PageChanged)
                {
                    PageChanged(this, e);
                }
            }

            protected virtual void OnSelectedChanged(EventArgs e)
            {
                if (null != SelectedChanged)
                {
                    SelectedChanged(this, e);
                }
            }

            #endregion
        }

        #endregion

        #region ** イベントクラスとデリゲート

        public delegate void CoreEventHandler(object sender, CoreEventArgs e);
        public delegate void CoreCancelEventHandler(object sender, CoreCancelEventArgs e);
        public delegate void CoreSwapEventHandler(object sender, CoreSwapEventArgs e);

        public class CoreEventArgs : EventArgs
        {
            #region ** パラメータ

            private int _tabIndex = InvalidIndex;
            private Tab _tab = null;

            #endregion

            public CoreEventArgs(Tab tab, int tabIndex)
            {
                _tabIndex = tabIndex;
                _tab = tab;
            }

            #region ** プロパティ

            public int TabIndex
            {
                get { return _tabIndex; }
            }

            public Tab Tab
            {
                get { return _tab; }
            }

            #endregion
        }

        public class CoreCancelEventArgs : CancelEventArgs
        {
            #region ** パラメータ

            private int _tabIndex = InvalidIndex;
            private Tab _tab = null;

            #endregion

            public CoreCancelEventArgs(Tab tab, int tabIndex)
            {
                _tabIndex = tabIndex;
                _tab = tab;
            }

            #region ** プロパティ

            public int TabIndex
            {
                get { return _tabIndex; }
            }

            public Tab Tab
            {
                get { return _tab; }
            }

            #endregion
        }

        public class CoreSwapEventArgs : CoreEventArgs
        {
            #region ** パラメータ

            private int _tab2Index = InvalidIndex;
            private Tab _tab2 = null;

            #endregion

            public CoreSwapEventArgs(Tab tab1, int tab1Index, Tab tab2, int tab2Index)
                : base(tab1, tab1Index)
            {
                _tab2Index = tab2Index;
                _tab2 = tab2;
            }

            #region ** プロパティ

            public int Tab2Index
            {
                get { return _tab2Index; }
            }

            public Tab Tab2
            {
                get { return _tab2; }
            }

            #endregion
        }

        #endregion

        #region ** コレクション

        public class TabCollection : Collection<Tab>
        {
            #region ** パラメータ

            private Dictionary<ITabPage, Tab> _tabDictionary = new Dictionary<ITabPage, Tab>();

            #endregion

            #region ** イベント

            public event EventHandler ItemsCleared;
            public event CoreEventHandler ItemInserted;
            public event CoreCancelEventHandler ItemRemoving;
            public event CoreEventHandler ItemRemoved;
            public event CoreSwapEventHandler ItemSwapped;

            #endregion

            #region ** メソッド

            public bool Contains(ITabPage page)
            {
                return _tabDictionary.ContainsKey(page);
            }

            public int IndexOf(ITabPage page)
            {
                if (!Contains(page)) { return InvalidIndex; }
                return base.IndexOf(_tabDictionary[page]);
            }

            /// <summary>
            /// タブページを追加します。
            /// </summary>
            /// <param name="item">追加するタブページ。</param>
            public void AddPage(ITabPage page)
            {
                if (null == page) { throw new ArgumentNullException("item"); }
                Add(new Tab(page));
            }

            /// <summary>
            /// 指定したインデックスの位置にタブページを挿入します。
            /// </summary>
            /// <param name="index">タブページを挿入する位置の、0 から始まるインデックス。</param>
            /// <param name="item">挿入するタブページ。</param>
            public void InsertPage(int index, ITabPage page)
            {
                if (index < 0) { throw new ArgumentOutOfRangeException("index"); }
                if (Count < index) { throw new ArgumentOutOfRangeException("index"); }
                if (null == page) { throw new ArgumentNullException("item"); }

                Insert(index, new Tab(page));
            }

            //
            /// <summary>
            /// 最初に見つかった指定タブページを削除します。
            /// </summary>
            /// <param name="item">タブページ</param>
            /// <returns>タブページが正常に削除された場合は true。それ以外の場合は false。</returns>
            public bool RemovePage(ITabPage page)
            {
                if (!Contains(page)) { return false; }
                return Remove(this[page]);
            }

            public void SwapPage(int index1, int index2)
            {
                if (index1 < 0) { throw new ArgumentOutOfRangeException("index1"); }
                if (index2 < 0) { throw new ArgumentOutOfRangeException("index2"); }
                if (Count <= index1) { throw new ArgumentOutOfRangeException("index1"); }
                if (Count <= index2) { throw new ArgumentOutOfRangeException("index2"); }

                Tab tab1 = Items[index1];
                Tab tab2 = Items[index2];

                SetItem(index1, tab2);
                SetItem(index2, tab1);

                if (null != ItemSwapped)
                {
                    ItemSwapped(this, new CoreSwapEventArgs(tab1, index2, tab2, index1));
                }
            }

            public void MovePage(int from, int to)
            {
                if (from < 0) { throw new ArgumentOutOfRangeException("from"); }
                if (to < 0) { throw new ArgumentOutOfRangeException("to"); }
                if (Count <= from) { throw new ArgumentOutOfRangeException("from"); }
                if (Count <= to) { throw new ArgumentOutOfRangeException("to"); }

                int index = from;

                while (index != to)
                {

                    int next = (index < to) ? index + 1 : index - 1;

                    SwapPage(index, next);
                    index = next;

                }
            }

            #endregion

            #region ** メソッドのオーバーライド

            /// <summary>
            /// System.Collections.ObjectModel.Collection<T> からすべての要素を削除します。
            /// </summary>
            protected override void ClearItems()
            {
                while (Count > 0)
                {
                    RemoveItem(0);
                }

                base.ClearItems();

                if (null != ItemsCleared)
                {
                    ItemsCleared(this, new EventArgs());
                }
            }

            /// <summary>
            /// System.Collections.ObjectModel.Collection<T> 内の指定したインデックスの位置に要素を挿入します。
            /// </summary>
            /// <param name="index">item を挿入する位置の、0 から始まるインデックス番号。</param>
            /// <param name="item">挿入するオブジェクト。参照型の場合、null の値を使用できます。</param>
            protected override void InsertItem(int index, Tab item)
            {
                if (null == item) { throw new ArgumentNullException(); }

                base.InsertItem(index, item);
                _tabDictionary.Add(item.Page, item);

                item.Page.Hide();

                if (null != ItemInserted)
                {
                    ItemInserted(this, new CoreEventArgs(item, index));
                }
            }

            /// <summary>
            /// System.Collections.ObjectModel.Collection<T> の指定したインデックスにある要素を削除します。
            /// </summary>
            /// <param name="index">削除する要素の、0 から始まるインデックス番号。</param>
            protected override void RemoveItem(int index)
            {
                Tab item = Items[index];

                if (null != ItemRemoving)
                {

                    CoreCancelEventArgs e = new CoreCancelEventArgs(item, index);
                    ItemRemoving(this, e);

                    if (e.Cancel) { return; }

                }

                item.Page.Hide();

                _tabDictionary.Remove(item.Page);
                base.RemoveItem(index);

                if (null != ItemRemoved)
                {
                    ItemRemoved(this, new CoreEventArgs(item, index));
                }
            }

            #endregion

            #region ** インデクサ

            public Tab this[ITabPage page]
            {
                get
                {
                    if (!Contains(page)) { return null; }
                    return _tabDictionary[page];
                }
            }

            #endregion
        }

        #endregion

        #region ** Win32

        internal sealed class Win32API
        {
            /// <summary>
            /// インポート関数。
            /// </summary>
            [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
            public static extern IntPtr GetFocus();
        }

        #endregion
    }
}
