﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using NintendoWare.SoundFoundation.Core;

namespace NintendoWare.SoundFoundation.Windows.CommandBars
{
    /// <summary>
    /// コマンドバーボタンの表示スタイル。
    /// </summary>
    public enum CommandBarButtonStyle
    {
        /// <summary>
        /// テキストのみ。
        /// </summary>
        [Text("TextOnly")]
        TextOnly,

        /// <summary>
        /// イメージのみ。
        /// </summary>
        [Text("ImageOnly")]
        ImageOnly,

        /// <summary>
        /// テキストの後ろにイメージ。
        /// </summary>
        [Text("TextBeforeImage")]
        TextBeforeImage,

        /// <summary>
        /// テキストの前にイメージ。
        /// </summary>
        [Text("TextAfterImage")]
        TextAfterImage,
    }

    /// <summary>
    /// CommandBarButtonStyle に関連する拡張機能を提供します。
    /// </summary>
    public static class CommandBarButtonStyleEx
    {
        /// <summary>
        /// CommandBarButtonStyle の拡張メソッド。
        /// CommandBarButtonStyle フィールドのTextAttribute 属性から、フィールドに関連付けられたテキストを取得します。
        /// </summary>
        /// <param name="obj">CommandBarButtonStyle。</param>
        /// <returns>CommandBarButtonStyle フィールドに関連付けられたテキスト。</returns>
        public static string GetText(this CommandBarButtonStyle obj)
        {
            return EnumEx.GetText(obj);
        }

        /// <summary>
        /// 指定テキストに対応する CommandBarButtonStyle 値を取得します。
        /// 対応する CommandBarButtonStyle 値が見つからなかった場合は、指定デフォルト値を返します。
        /// </summary>
        /// <typeparam name="EnumType">Enum 型。</typeparam>
        /// <param name="text">テキスト。</param>
        /// <param name="defaultValue">デフォルト値。</param>
        /// <returns>対魚する CommandBarButtonStyle 値が見つかった場合は、その Enum値、見つからなかった場合は、指定デフォルト値。</returns>
        public static CommandBarButtonStyle ParseText(string text, CommandBarButtonStyle defaultValue)
        {
            try
            {
                return ParseText(text);
            }
            catch (ArgumentException)
            {
                return defaultValue;
            }
        }

        /// <summary>
        /// 指定テキストに対応する CommandBarButtonStyle 値を取得します。
        /// </summary>
        /// <typeparam name="EnumType">CommandBarButtonStyle 型。</typeparam>
        /// <param name="text">テキスト。</param>
        /// <returns>対魚する CommandBarButtonStyle 値が見つかった場合は、その値。</returns>
        public static CommandBarButtonStyle ParseText(string text)
        {
            return (CommandBarButtonStyle)EnumEx.ParseText<CommandBarButtonStyle>(text);
        }
    }
}
