﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using Core.Parameters;
    using ToolDevelopmentKit;

    /// <summary>
    /// パラメータ値を生成します。
    /// </summary>
    public static class ParameterValueFactory
    {
        public delegate IParameterValue CreateParameterValueHandler(object value, object min, object max);

        private static Dictionary<Type, CreateParameterValueHandler>
            handlerDictionary = new Dictionary<Type, CreateParameterValueHandler>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        static ParameterValueFactory()
        {
            handlerDictionary.Add(typeof(int), CreateIntParameterValue);
            handlerDictionary.Add(typeof(uint), CreateUIntParameterValue);
            handlerDictionary.Add(typeof(ulong), CreateULongParameterValue);
            handlerDictionary.Add(typeof(float), CreateFloatParameterValue);
            handlerDictionary.Add(typeof(double), CreateDoubleParameterValue);
            handlerDictionary.Add(typeof(bool), CreateBoolParameterValue);
            handlerDictionary.Add(typeof(string), CreateTextParameterValue);
            handlerDictionary.Add(typeof(DecayCurve3D), CreateDecayCurve3DParameterValue);
        }

        /// <summary>
        /// パラメータ値クラス生成ハンドラ辞書を取得します。
        /// </summary>
        public static IDictionary<Type, CreateParameterValueHandler> HandlerDictionary
        {
            get { return handlerDictionary; }
        }

        /// <summary>
        /// パラメータ値を生成します。
        /// </summary>
        /// <param name="value">パラメータの値を指定します。</param>
        /// <param name="min">
        /// パラメータの最小値を指定します。
        /// この引数は省略可能です。
        /// </param>
        /// <param name="max">
        /// パラメータの最大値を指定します。
        /// この引数は省略可能です。
        /// </param>
        /// <returns>IParameterValue を返します。</returns>
        public static IParameterValue Create(object value, object min, object max)
        {
            Ensure.Argument.NotNull(value);
            return Create(value.GetType(), value, min, max);
        }

        /// <summary>
        /// パラメータ値を生成します。
        /// </summary>
        /// <param name="valueType">パラメータ値の型を指定します。</param>
        /// <param name="value">パラメータの値を指定します。</param>
        /// <param name="min">
        /// パラメータの最小値を指定します。
        /// この引数は省略可能です。
        /// </param>
        /// <param name="max">
        /// パラメータの最大値を指定します。
        /// この引数は省略可能です。
        /// </param>
        /// <returns>IParameterValue を返します。</returns>
        public static IParameterValue Create(Type valueType, object value, object min, object max)
        {
            Ensure.Argument.NotNull(valueType);
            return handlerDictionary[valueType](value, min, max);
        }

        private static IParameterValue CreateIntParameterValue(object value, object min, object max)
        {
            if (value == null)
            {
                if (min == null || max == null)
                {
                    return new IntParameterValue();
                }

                return new IntParameterValue(0, (int)min, (int)max);
            }

            if (min == null || max == null)
            {
                return new IntParameterValue((int)value);
            }

            return new IntParameterValue((int)value, (int)min, (int)max);
        }

        private static IParameterValue CreateUIntParameterValue(object value, object min, object max)
        {
            if (value == null)
            {
                return new UIntParameterValue();
            }

            return new UIntParameterValue((uint)value);
        }

        private static IParameterValue CreateULongParameterValue(object value, object min, object max)
        {
            if (value == null)
            {
                if (min == null || max == null)
                {
                    return new ULongParameterValue();
                }

                return new ULongParameterValue(0, (ulong)min, (ulong)max);
            }

            if (min == null || max == null)
            {
                return new ULongParameterValue((ulong)value);
            }

            return new ULongParameterValue((ulong)value, (ulong)min, (ulong)max);
        }

        private static IParameterValue CreateFloatParameterValue(object value, object min, object max)
        {
            if (value == null)
            {
                if (min == null || max == null)
                {
                    return new FloatParameterValue();
                }

                return new FloatParameterValue(0.0f, (float)min, (float)max);
            }

            if (min == null || max == null)
            {
                return new FloatParameterValue((float)value);
            }

            return new FloatParameterValue((float)value, (float)min, (float)max);
        }

        private static IParameterValue CreateDoubleParameterValue(object value, object min, object max)
        {
            if (value == null)
            {
                if (min == null || max == null)
                {
                    return new DoubleParameterValue();
                }

                return new DoubleParameterValue(0, (double)min, (double)max);
            }

            if (min == null || max == null)
            {
                return new DoubleParameterValue((double)value);
            }

            return new DoubleParameterValue((double)value, (double)min, (double)max);
        }

        private static IParameterValue CreateBoolParameterValue(object value, object min, object max)
        {
            if (value == null)
            {
                return new BoolParameterValue();
            }

            return new BoolParameterValue((bool)value);
        }

        private static IParameterValue CreateTextParameterValue(object value, object min, object max)
        {
            if (value == null)
            {
                return new TextParameterValue();
            }

            return new TextParameterValue((string)value);
        }

        private static IParameterValue CreateDecayCurve3DParameterValue(object value, object min, object max)
        {
            if (value == null)
            {
                return new DecayCurve3DParameterValue();
            }

            return new DecayCurve3DParameterValue((DecayCurve3D)value);
        }
    }
}
