﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using Core;
    using Core.Parameters;
    using Resources;
    using ToolDevelopmentKit;

    /// <summary>
    /// DecayCurve3D 型のパラメータ値を格納します。
    /// </summary>
    public class DecayCurve3DParameterValue : ParameterValue<DecayCurve3D>
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public DecayCurve3DParameterValue()
            : this(DecayCurve3D.Log)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータ値を指定します。</param>
        public DecayCurve3DParameterValue(DecayCurve3D value)
            : base(value)
        {
        }

        /// <summary>
        /// パラメータの値を取得するキャストオペレータです。
        /// </summary>
        /// <param name="value">対象となるパラメータ値を指定します。</param>
        /// <returns>パラメータの値を返します。</returns>
        public static implicit operator DecayCurve3D(DecayCurve3DParameterValue value)
        {
            Ensure.Argument.NotNull(value);
            return value.Value;
        }

        /// <summary>
        /// パラメータの値からテキストを取得します。
        /// </summary>
        /// <returns>テキストを返します。</returns>
        public override string ToString()
        {
            switch (this.Value)
            {
                case DecayCurve3D.Log:
                    return MessageResource.Label_DecayCurve3D_Log;

                case DecayCurve3D.Linear:
                    return MessageResource.Label_DecayCurve3D_Linear;
            }

            Debug.Assert(false, "An unexpected value");
            return String.Empty;
        }

        /// <summary>
        /// テキストからパラメータ値を取得します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        protected override DecayCurve3D ParseInternal(string text)
        {
            Ensure.Argument.NotNull(text);

            if (text == MessageResource.Label_DecayCurve3D_Log)
            {
                return DecayCurve3D.Log;
            }
            else if (text == MessageResource.Label_DecayCurve3D_Linear)
            {
                return DecayCurve3D.Linear;
            }

            throw new ArgumentException("invalid text.");
        }

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        protected override ValidationResult ValidateInternal(DecayCurve3D value)
        {
            return ValidationResult.NoError;
        }

        /// <summary>
        /// パラメータ値のバイト列を取得します。
        /// </summary>
        /// <returns>バイト列の列挙子を返します。</returns>
        protected override IEnumerable<byte> GetBytes()
        {
            return BitConverter.GetBytes((int)this.Value);
        }
    }
}
