﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.ToolDevelopmentKit;
    using LegacyFormats = NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat;
    using MsgRsrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// RVL サウンドセットをインポートします。
    /// </summary>
    public class SoundSetRvlDocumentImporter : IDocumentImporter
    {
        private readonly DocumentService documentService = null;
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="documentService">ドキュメントサービスを指定します。</param>
        /// <param name="intermediateOutputTraits">中間出力特性を指定します。</param>
        public SoundSetRvlDocumentImporter(
            DocumentService documentService,
            SoundIntermediateOutputTraits intermediateOutputTraits)
        {
            Ensure.Argument.NotNull(documentService);
            Ensure.Argument.NotNull(intermediateOutputTraits);

            this.documentService = documentService;
            this.intermediateOutputTraits = intermediateOutputTraits;
        }

        /// <summary>
        /// インポート元のドキュメントタイプ名を取得します。
        /// </summary>
        public string SourceDocumentTypeName
        {
            get { return Platforms.Rvl.SoundSetDocument; }
        }

        /// <summary>
        /// インポート後のドキュメントタイプ名を取得します。
        /// </summary>
        public string DestinationDocumentTypeName
        {
            get { return this.intermediateOutputTraits.SoundSetDocumentTypeName; }
        }

        /// <summary>
        /// ドキュメントをインポートします。
        /// </summary>
        /// <param name="document">インポートするドキュメントを指定します。</param>
        /// <returns>インポートしたドキュメント参照の配列を返します。</returns>
        public DocumentReference[] Import(Document document)
        {
            Assertion.Argument.NotNull(document);
            Ensure.Argument.True(document is SoundSetRvlDocument);
            Ensure.Argument.True(document.Resource is FileResource);

            string typeName = this.intermediateOutputTraits.SoundSetDocumentTypeName;
            List<DocumentReference> results = new List<DocumentReference>();
            try
            {
                DocumentReference[] soundSetDocumentRefs = this.ImportSoundSetDocument(document);
                results.AddRange(soundSetDocumentRefs);
            }
            catch
            {
                foreach (DocumentReference soundSetDocumentRef in results)
                {
                    soundSetDocumentRef.Close();
                }
                throw;
            }

            return results.ToArray();
        }


        /// <summary>
        /// サウンドセットドキュメントをインポートします。
        /// </summary>
        /// <param name="filePaths">サウンドセットファイルパス。</param>
        /// <returns>インポートしたドキュメントへの参照配列を返します。</returns>
        public DocumentReference[] Import(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFileRead, filePath);
            List<DocumentReference> results = new List<DocumentReference>();
            try
            {
                using (DocumentReference documentRef =
                       this.documentService.OpenDocument(new FileResource(filePath)))
                {
                    if (documentRef == null)
                    {
                        throw new FileNotSupportedException(filePath);
                    }

                    if (!(documentRef.Document is SoundSetRvlDocument))
                    {
                        throw new FileNotSupportedException(filePath);
                    }

                    DocumentReference[] soundSetDocumentRefs =
                        this.ImportSoundSetDocument(documentRef.Document);
                    results.AddRange(soundSetDocumentRefs);
                }
            }
            catch (Exception e)
            {
                foreach (DocumentReference soundSetDocumentRef in results)
                {
                    soundSetDocumentRef.Close();
                }
                e.Data["ImportFilePath"] = filePath;
                throw e;
            }

            return results.ToArray();
        }

        private DocumentReference[] ImportSoundSetDocument(Document document)
        {
            List<DocumentReference> importedDocumentRefs = new List<DocumentReference>();

            try
            {
                string filePath = document.Resource.Key;

                ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFile, filePath);

                // サウンドセットドキュメント
                DocumentReference importedSoundSetDocumentRef =
                    this.documentService.ConvertDocument(document, this.intermediateOutputTraits.SoundSetDocumentTypeName);

                if (importedSoundSetDocumentRef == null)
                {
                    throw new FileNotSupportedException(filePath);
                }
                else if (!(importedSoundSetDocumentRef.Document is SoundSetDocument))
                {
                    importedSoundSetDocumentRef.Close();
                    throw new FileNotSupportedException(filePath);
                }

                importedDocumentRefs.Add(importedSoundSetDocumentRef);

                // バンクドキュメント
                foreach (DocumentReference importedBankDocumentRef in
                         this.ImportBankDocuments(document as SoundSetRvlDocument))
                {
                    importedDocumentRefs.Add(importedBankDocumentRef);
                }

                // テキストシーケンスサウンドドキュメント
                foreach (DocumentReference importedTextSequenceSoundDocumentRef in
                         this.ImportTextSequenceSoundDocuments(document as SoundSetRvlDocument))
                {
                    importedDocumentRefs.Add(importedTextSequenceSoundDocumentRef);
                }
            }
            catch
            {
                foreach (DocumentReference importedDocumentRef in importedDocumentRefs)
                {
                    importedDocumentRef.Close();
                }

                throw;
            }

            return importedDocumentRefs.ToArray();
        }


        /// <summary>
        /// バンクドキュメントをインポートします。
        /// </summary>
        /// <param name="legacySoundSetDocument">
        /// レガシーサウンドセットドキュメントを指定します。
        /// </param>
        /// <returns>インポートしたドキュメントへの参照配列を返します。</returns>
        private DocumentReference[] ImportBankDocuments(
            SoundSetRvlDocument legacySoundSetDocument)
        {
            Assertion.Argument.NotNull(legacySoundSetDocument);

            List<DocumentReference> importedDocumentRefs = new List<DocumentReference>();

            string soundSetDirectoryPath =
                Path.GetDirectoryName(legacySoundSetDocument.SoundSet.FilePath);

            string typeName = this.intermediateOutputTraits.BankDocumentTypeName;
            BankRvlDocumentImporter importer =
                new BankRvlDocumentImporter(
                                            this.documentService,
                                            this.intermediateOutputTraits
                                            );
            try
            {
                foreach (LegacyFormats.Nw4rSoundSetBank legacySoundSetBank in
                    legacySoundSetDocument.SoundSet.BankList)
                {
                    string path = Path.Combine(
                        soundSetDirectoryPath,
                        legacySoundSetBank.FilePath
                        ).GetFullPath();

                    DocumentReference[] docRef = importer.Import(path);
                    importedDocumentRefs.AddRange(docRef);
                }
            }
            catch
            {
                foreach (DocumentReference documentRef in importedDocumentRefs)
                {
                    documentRef.Close();
                }

                throw;
            }

            return importedDocumentRefs.ToArray();
        }

        /// <summary>
        /// テキストシーケンスサウンドドキュメントをインポートします。
        /// </summary>
        /// <param name="legacySoundSetDocument">
        /// レガシーサウンドセットドキュメントを指定します。
        /// </param>
        /// <returns>インポートしたドキュメントへの参照配列を返します。</returns>
        private DocumentReference[] ImportTextSequenceSoundDocuments(
            SoundSetRvlDocument legacySoundSetDocument)
        {
            Assertion.Argument.NotNull(legacySoundSetDocument);

            List<DocumentReference> importedDocumentRefs = new List<DocumentReference>();

            string soundSetDirectoryPath =
                Path.GetDirectoryName(legacySoundSetDocument.SoundSet.FilePath);
            HashSet<string> importedFilePaths = new HashSet<string>();

            try
            {
                // シーケンスサウンド
                foreach (LegacyFormats.Nw4rSoundSetSeq legacySequenceSound in
                    legacySoundSetDocument.SoundSet.SeqList)
                {
                    if (legacySequenceSound.FileType != LegacyFormats.Nw4rSoundSetSeqFileType.Text)
                    {
                        continue;
                    }

                    importedDocumentRefs.AddRange(
                        this.ImportTextSequenceSoundDocument(
                            Path.Combine(
                                soundSetDirectoryPath,
                                legacySequenceSound.FilePath
                                ).GetFullPath(),
                            importedFilePaths
                            ));
                }

                // シーケンスサウンドセット内のシーケンスサウンド
                foreach (LegacyFormats.Nw4rSoundSetSeqSoundSet legacySequenceSoundSet in
                    legacySoundSetDocument.SoundSet.SeqSoundSetList)
                {
                    foreach (LegacyFormats.Nw4rSoundSetSeq legacySequenceSound in
                             legacySequenceSoundSet)
                    {
                        if (legacySequenceSound.FileType !=
                            LegacyFormats.Nw4rSoundSetSeqFileType.Text)
                        {
                            continue;
                        }

                        importedDocumentRefs.AddRange(
                            this.ImportTextSequenceSoundDocument(
                                Path.Combine(
                                    soundSetDirectoryPath,
                                    legacySequenceSound.FilePath
                                    ).GetFullPath(),
                                importedFilePaths
                                ));
                    }
                }
            }
            catch
            {
                foreach (DocumentReference documentRef in importedDocumentRefs)
                {
                    documentRef.Close();
                }

                throw;
            }

            return importedDocumentRefs.ToArray();
        }

        /// <summary>
        /// テキストシーケンスサウンドドキュメントをインポートします。
        /// </summary>
        /// <param name="filePaths">テキストシーケンスサウンドファイルパスを指定します。</param>
        /// <param name="importedFilePaths">
        /// インポート済みのファイルパスコレクションを指定します。
        /// </param>
        /// <returns>インポートしたテキストシーケンスサウンドドキュメントへの参照を返します。</returns>
        private DocumentReference[] ImportTextSequenceSoundDocument(
                                                                string filePath,
                                                                HashSet<string> importedFilePaths
                                                                )
        {
            Ensure.Argument.StringNotEmpty(filePath);
            Ensure.Argument.NotNull(importedFilePaths);

            if (importedFilePaths.Contains(filePath) == true)
            {
                return new DocumentReference[0];
            }

            if (File.Exists(filePath) == false)
            {
                ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportWarning_FileNotFound, filePath);
                return new DocumentReference[0];
            }

            TextSequenceSoundDocument document = new TextSequenceSoundDocument()
            {
                Resource = new FileResource((filePath)),
            };

            ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFileRead, filePath);
            try
            {
                new TextSequenceSoundDocumentReader(Platforms.Any.TextSequenceSoundDocument).
                    Read(document.Resource, document);
            }
            catch (Exception e)
            {
                e.Data["ImportFilePath"] = filePath;
                throw e;
            }

            ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFile, filePath);
            document.Text =
                document.Text.Replace(
                                     ".rseq", "." +
                                     this.intermediateOutputTraits.TextSequenceSoundFileExtension
                                     );
            document.Text =
                document.Text.Replace(
                                      ".rinl", "." +
                                      this.intermediateOutputTraits.BankIncludeFileExtension
                                      );
            document.Resource = new FileResource(
                Path.ChangeExtension(
                                     filePath,
                                     this.intermediateOutputTraits.TextSequenceSoundFileExtension
                                     )
                );

            List<DocumentReference> importedDocumentRefs = new List<DocumentReference>();

            importedDocumentRefs.Add(
                new DocumentReference(null)
                {
                    Document = document,
                });
            importedFilePaths.Add(filePath);

            SequenceSoundTextLexer lexer = new SequenceSoundTextLexer(filePath)
            {
                SkipNotFoundFiles = true,
            };

            while (lexer.ReadToken() != SequenceSoundTextLexer.Token.End)
            {
                // 依存ファイル収集のため空回し
            }

            foreach (string dependentFilePath in lexer.DependFiles)
            {
                if (dependentFilePath == filePath)
                {
                    continue;
                }

                if (dependentFilePath.EndsWith(".rseq"))
                {
                    importedDocumentRefs.AddRange(
                        this.ImportTextSequenceSoundDocument(dependentFilePath, importedFilePaths)
                        );
                }
            }

            return importedDocumentRefs.ToArray();
        }
    }
}
