﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.ToolDevelopmentKit;
    using LegacyFormats = NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat;
    using MsgRsrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// RVL サウンドプロジェクトをインポートします。
    /// </summary>
    public class SoundRvlDocumentImporter : IDocumentImporter
    {
        private readonly DocumentService documentService = null;
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="documentService">ドキュメントサービスを指定します。</param>
        /// <param name="intermediateOutputTraits">中間出力特性を指定します。</param>
        public SoundRvlDocumentImporter(
            DocumentService documentService,
            SoundIntermediateOutputTraits intermediateOutputTraits)
        {
            Ensure.Argument.NotNull(documentService);
            Ensure.Argument.NotNull(intermediateOutputTraits);

            this.documentService = documentService;
            this.intermediateOutputTraits = intermediateOutputTraits;
        }

        /// <summary>
        /// インポート元のドキュメントタイプ名を取得します。
        /// </summary>
        public string SourceDocumentTypeName
        {
            get { return Platforms.Rvl.SoundProjectDocument; }
        }

        /// <summary>
        /// インポート後のドキュメントタイプ名を取得します。
        /// </summary>
        public string DestinationDocumentTypeName
        {
            get { return this.intermediateOutputTraits.SoundProjectDocumentTypeName; }
        }

        /// <summary>
        /// ドキュメントをインポートします。
        /// </summary>
        /// <param name="document">インポートするドキュメントを指定します。</param>
        /// <returns>インポートしたドキュメント参照の配列を返します。</returns>
        public DocumentReference[] Import(Document document)
        {
            Assertion.Argument.NotNull(document);
            Ensure.Argument.True(document is SoundProjectRvlDocument);
            Ensure.Argument.True(document.Resource is FileResource);

            string typeName = this.intermediateOutputTraits.SoundProjectDocumentTypeName;
            List<DocumentReference> results = new List<DocumentReference>();
            try
            {
                ImportLog.WriteLine(
                                    MsgRsrc.MessageResource.Message_ImportFile,
                                    document.Resource.Key
                                    );
                DocumentReference soundProjectDocumentRef =
                    this.documentService.ConvertDocument(document, typeName);
                results.Add(soundProjectDocumentRef);

                DocumentReference[] importedDocumentRefs =
                    this.ImportSoundSetDocuments(document as SoundProjectRvlDocument);
                results.AddRange(importedDocumentRefs);
            }
            catch
            {
                foreach (DocumentReference soundSetDocumentRef in results)
                {
                    soundSetDocumentRef.Close();
                }
                throw;
            }

            return results.ToArray();
        }

        /// <summary>
        /// サウンドセットドキュメントをインポートします。
        /// </summary>
        /// <param name="legacySoundProjectDocument">
        /// レガシーサウンドプロジェクトドキュメントを指定します。
        /// </param>
        /// <returns>インポートしたドキュメントへの参照配列を返します。</returns>
        private DocumentReference[] ImportSoundSetDocuments(
            SoundProjectRvlDocument legacySoundProjectDocument)
        {
            Assertion.Argument.NotNull(legacySoundProjectDocument);

            List<DocumentReference> results = new List<DocumentReference>();

            string soundProjectDirectoryPath =
                Path.GetDirectoryName(legacySoundProjectDocument.Project.FilePath);

            try
            {
                SoundSetRvlDocumentImporter importer =
                    new SoundSetRvlDocumentImporter(this.documentService,
                                                    this.intermediateOutputTraits);

                foreach (LegacyFormats.Nw4rSoundMakerProject.SoundSet legacySoundSet in
                    legacySoundProjectDocument.Project.SoundSets)
                {
                    DocumentReference[] importedDocumentRefs =
                        importer.Import(
                          Path.Combine(soundProjectDirectoryPath, legacySoundSet.FilePath).
                          GetFullPath()
                          );
                    results.AddRange(importedDocumentRefs);
                }
            }
            catch
            {
                foreach (DocumentReference importedDocumentRef in results)
                {
                    importedDocumentRef.Close();
                }

                throw;
            }

            return results.ToArray();
        }
    }
}
