﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using Core;
    using Core.Parameters;
    using NintendoWare.SoundFoundation.FileFormats.Wave;

    public class WaveSoundBase : Sound
    {
        public const string ParameterNameOfDataSize = "DataSize";

        /// <summary>
        ///
        /// </summary>
        public WaveSoundBase()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public WaveFile WaveFile
        {
            get;
            set;
        }

        /// <summary>
        /// チャンネル優先度
        /// </summary>
        public int ChannelPriority
        {
            get { return GetIntValue(ProjectParameterNames.Sound.ChannelPriority); }
            set { SetIntValue(ProjectParameterNames.Sound.ChannelPriority, value); }
        }

        /// <summary>
        /// パン
        /// </summary>
        public int Pan
        {
            get { return GetIntValue(ProjectParameterNames.Pan); }
            set { SetIntValue(ProjectParameterNames.Pan, value); }
        }

        /// <summary>
        /// サラウンドパン
        /// </summary>
        public int SurroundPan
        {
            get { return GetIntValue(ProjectParameterNames.SurroundPan); }
            set { SetIntValue(ProjectParameterNames.SurroundPan, value); }
        }

        /// <summary>
        /// ピッチ
        /// </summary>
        public float Pitch
        {
            get { return GetValue<float>(ProjectParameterNames.Pitch); }
            set { SetValue<float>(ProjectParameterNames.Pitch, value); }
        }

        /// <summary>
        /// リリースエンベローブ
        /// </summary>
        public int ReleaseEnvelope
        {
            get { return GetIntValue(ProjectParameterNames.WaveSound.EnvelopeRelease); }
            set { SetIntValue(ProjectParameterNames.WaveSound.EnvelopeRelease, value); }
        }

        /// <summary>
        /// センド設定を取得または設定します。
        /// </summary>
        public Sends Sends
        {
            get { return this.GetValue<Sends>(ProjectParameterNames.Sound.Sends); }
            set { this.SetValue<Sends>(ProjectParameterNames.Sound.Sends, value); }
        }

        /// <summary>
        /// リリース時優先度固定
        /// </summary>
        public bool ReleasePriorityFixed
        {
            get { return GetBoolValue(ProjectParameterNames.Sound.ReleasePriorityFixed); }
            set { SetBoolValue(ProjectParameterNames.Sound.ReleasePriorityFixed, value); }
        }

        /// <summary>
        /// エンコード
        /// </summary>
        public WaveEncoding Encoding
        {
            get { return GetValue<WaveEncoding>(ProjectParameterNames.WaveEncoding); }
            set { SetValue<WaveEncoding>(ProjectParameterNames.WaveEncoding, value); }
        }

        /// <summary>
        /// パンモード
        /// </summary>
        public PanMode PanMode
        {
            get { return GetPanModeValue(ProjectParameterNames.PanMode); }
            set { SetPanModeValue(ProjectParameterNames.PanMode, value); }
        }

        /// <summary>
        /// パンカーブ
        /// </summary>
        public PanCurve PanCurve
        {
            get { return GetPanCurveValue(ProjectParameterNames.PanCurve); }
            set { SetPanCurveValue(ProjectParameterNames.PanCurve, value); }
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public int DataSize
        {
            get { return GetIntValue(ParameterNameOfDataSize); }
            set { SetIntValue(ParameterNameOfDataSize, value); }
        }

        ///--------------------------------
        /// <summary>
        /// フロントバイパス
        /// </summary>
        public bool FrontBypass
        {
            get { return this.GetBoolValue(ProjectParameterNames.FrontBypass); }
            set { this.SetBoolValue(ProjectParameterNames.FrontBypass, value); }
        }

        ///--------------------------------
        /// <summary>
        /// LPF (Low-pass filter)
        /// </summary>
        public int LPF
        {
            get { return GetIntValue(ProjectParameterNames.LPF); }
            set { SetIntValue(ProjectParameterNames.LPF, value); }
        }

        ///--------------------------------
        /// <summary>
        /// Biquad フィルタの種類
        /// </summary>
        public BiquadType BiquadType
        {
            get { return GetValue<BiquadType>(ProjectParameterNames.BiquadType); }
            set { SetValue<BiquadType>(ProjectParameterNames.BiquadType, value); }
        }

        ///--------------------------------
        /// <summary>
        /// Biquad フィルタ値
        /// </summary>
        public int Biquad
        {
            get { return GetIntValue(ProjectParameterNames.Biquad); }
            set { SetIntValue(ProjectParameterNames.Biquad, value); }
        }

        /// <summary>
        /// リサンプルの設定を取得または設定します。
        /// </summary>
        public bool IsResampleEnabled
        {
            get { return this.GetBoolValue(ProjectParameterNames.IsResampleEnabled); }
            set { this.SetBoolValue(ProjectParameterNames.IsResampleEnabled, value); }
        }

        /// <summary>
        /// サンプルレートを取得または設定します。
        /// </summary>
        public int SampleRate
        {
            get { return this.GetIntValue(ProjectParameterNames.SampleRate); }
            set { this.SetValue(ProjectParameterNames.SampleRate, value); }
        }

        /// <summary>
        /// ダウンミックス設定を取得または設定します。
        /// </summary>
        public bool IsDownMixEnabled
        {
            get { return this.GetBoolValue(ProjectParameterNames.IsDownMixEnabled); }
            set { this.SetBoolValue(ProjectParameterNames.IsDownMixEnabled, value); }
        }

        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            SetInstanceBoolValue(ProjectParameterNames.SndEdit, false);
            SetInstanceBoolValue(ProjectParameterNames.SndEditEnabled, true);

            SetInstanceWaveEncodingValue(ProjectParameterNames.WaveEncoding, WaveEncoding.Adpcm);
            SetInstanceBoolValue(ProjectParameterNames.IsResampleEnabled, false);
            SetInstanceParameterValue(ProjectParameterNames.SampleRate, new SampleRateParameterValue());
            SetInstanceBoolValue(ProjectParameterNames.IsDownMixEnabled, false);

            SetInstanceIntValue(ProjectParameterNames.Sound.ChannelPriority, 64, 0, 127);
            SetInstanceIntValue(ProjectParameterNames.Pan, 64, 0, 127);
            SetInstanceIntValue(ProjectParameterNames.SurroundPan, 0, 0, 127);
            this.Parameters.AddValue(ProjectParameterNames.Pitch,
                                      new PitchParameterValue(1.0F, 0.00001F, float.MaxValue));
            SetInstanceIntValue(ProjectParameterNames.WaveSound.EnvelopeRelease, 127, 0, 127);
            SetInstanceBoolValue(ProjectParameterNames.Sound.ReleasePriorityFixed, false);
            SetInstancePanModeValue(ProjectParameterNames.PanMode, PanMode.Dual);
            SetInstancePanCurveValue(ProjectParameterNames.PanCurve, PanCurve.Sqrt);
            SetInstanceBoolValue(ProjectParameterNames.FrontBypass, false);
            SetInstanceIntValue(ParameterNameOfDataSize, -1);
            SetInstanceIntValue(ProjectParameterNames.LPF, 64, 0, 64);
            SetInstanceBiquadTypeValue(ProjectParameterNames.BiquadType, BiquadType.None);
            SetInstanceIntValue(ProjectParameterNames.Biquad, 0, 0, 127);

            // ウェーブサウンドバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.WaveEncoding, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.IsResampleEnabled, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.SampleRate, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.IsDownMixEnabled, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.FilePath, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.Pan, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.SurroundPan, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.Pitch, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.WaveSound.EnvelopeRelease, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.LPF, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.BiquadType, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.Biquad, ParameterAttributes.ComputeHash);

            // サウンドアーカイブバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.FrontBypass, ParameterAttributes.ComputeSoundArchiveHash);

            // sndedit 転送対象パラメータ（値が変化しても sndedit 用にコンバートされません）
            this.SetParameterAttributes(ProjectParameterNames.Sound.ChannelPriority, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.ReleasePriorityFixed, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.PanMode, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.PanCurve, ParameterAttributes.ComputeSndeditTransferHash);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceWaveEncodingValue(string name, WaveEncoding value)
        {
            SetInstanceParameterValue(name, new WaveEncodingParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceLoopTypeValue(string name, LoopType value)
        {
            SetInstanceParameterValue(name, new LoopTypeParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceBiquadTypeValue(string name, BiquadType value)
        {
            SetInstanceParameterValue(name, new BiquadTypeParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnParameterValueChanged(ParameterEventArgs e)
        {
            if (e.Key == ProjectParameterNames.FilePath)
            {
                WaveFile = null;
            }

            base.OnParameterValueChanged(e);
        }
    }
}
