﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.FileFormats.Wave;

    public class VelocityRegion : BankItem, ICloneable
    {
        public const string ParameterNameOfEnvelope = "Envelope";
        public const string ParameterNameOfDataSize = "DataSize";

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public VelocityRegion()
        {
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパス
        /// </summary>
        public string FilePath
        {
            get { return GetFilePathValue(ProjectParameterNames.FilePath); }
            set { SetFilePathValue(ProjectParameterNames.FilePath, value); }
        }

        ///--------------------------------
        /// <summary>
        /// エンコード
        /// </summary>
        public WaveEncoding Encoding
        {
            get { return GetValue<WaveEncoding>(ProjectParameterNames.WaveEncoding); }
            set { SetValue<WaveEncoding>(ProjectParameterNames.WaveEncoding, value); }
        }

        ///--------------------------------
        /// <summary>
        /// オリジナルキー
        /// </summary>
        public int OriginalKey
        {
            get { return GetIntValue(ProjectParameterNames.VelocityRegion.OriginalKey); }
            set { SetIntValue(ProjectParameterNames.VelocityRegion.OriginalKey, value); }
        }

        ///--------------------------------
        /// <summary>
        /// エンベローブ
        /// </summary>
        public Envelope Envelope
        {
            get { return this.GetValue<Envelope>(ProjectParameterNames.EnvelopeParams); }
            set { this.SetValue<Envelope>(ProjectParameterNames.EnvelopeParams, value); }
        }

        ///--------------------------------
        /// <summary>
        /// 開始位置
        /// </summary>
        public int VelocityMin
        {
            get { return GetIntValue(ProjectParameterNames.VelocityRegion.VelocityMin); }
            set { SetIntValue(ProjectParameterNames.VelocityRegion.VelocityMin, value); }
        }

        ///--------------------------------
        /// <summary>
        /// 終了位置
        /// </summary>
        public int VelocityMax
        {
            get { return GetIntValue(ProjectParameterNames.VelocityRegion.VelocityMax); }
            set { SetIntValue(ProjectParameterNames.VelocityRegion.VelocityMax, value); }
        }

        ///--------------------------------
        /// <summary>
        /// 音量
        /// </summary>
        public int Volume
        {
            get { return GetIntValue(ProjectParameterNames.Volume); }
            set { SetIntValue(ProjectParameterNames.Volume, value); }
        }

        ///--------------------------------
        /// <summary>
        /// パン
        /// </summary>
        public int Pan
        {
            get { return GetIntValue(ProjectParameterNames.Pan); }
            set { SetIntValue(ProjectParameterNames.Pan, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int PitchSemitones
        {
            get { return GetIntValue(ProjectParameterNames.VelocityRegion.PitchSemitones); }
            set { SetIntValue(ProjectParameterNames.VelocityRegion.PitchSemitones, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int PitchCents
        {
            get { return GetIntValue(ProjectParameterNames.VelocityRegion.PitchCents); }
            set { SetIntValue(ProjectParameterNames.VelocityRegion.PitchCents, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool PercussionMode
        {
            get { return GetBoolValue(ProjectParameterNames.VelocityRegion.PercussionMode); }
            set { SetBoolValue(ProjectParameterNames.VelocityRegion.PercussionMode, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int KeyGroup
        {
            get { return GetIntValue(ProjectParameterNames.VelocityRegion.KeyGroup); }
            set { SetIntValue(ProjectParameterNames.VelocityRegion.KeyGroup, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public InterpolationType InterpolationType
        {
            get { return GetValue<InterpolationType>(ProjectParameterNames.VelocityRegion.InterpolationType); }
            set { SetValue<InterpolationType>(ProjectParameterNames.VelocityRegion.InterpolationType, value); }
        }

        ///--------------------------------
        /// <summary>
        /// データサイズ
        /// </summary>
        public int DataSize
        {
            get { return GetIntValue(ParameterNameOfDataSize); }
            set { SetIntValue(ParameterNameOfDataSize, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public WaveFile WaveFile { get; set; }

        ///
        public VelocityRegion Clone()
        {
            VelocityRegion velRegion = new VelocityRegion();

            velRegion.FilePath = this.FilePath;
            velRegion.Encoding = this.Encoding;
            velRegion.OriginalKey = this.OriginalKey;

            velRegion.Envelope.Attack = this.Envelope.Attack;
            velRegion.Envelope.Hold = this.Envelope.Hold;
            velRegion.Envelope.Decay = this.Envelope.Decay;
            velRegion.Envelope.Sustain = this.Envelope.Sustain;
            velRegion.Envelope.Release = this.Envelope.Release;

            velRegion.VelocityMin = this.VelocityMin;
            velRegion.VelocityMax = this.VelocityMax;
            velRegion.Volume = this.Volume;
            velRegion.Pan = this.Pan;
            velRegion.PitchSemitones = this.PitchSemitones;
            velRegion.PitchCents = this.PitchCents;
            velRegion.KeyGroup = this.KeyGroup;
            velRegion.InterpolationType = this.InterpolationType;
            velRegion.PercussionMode = this.PercussionMode;

            return velRegion;
        }

        ///
        object ICloneable.Clone()
        {
            return this.Clone();
        }

        ///--------------------------------
        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            SetInstanceFilePathValue(ProjectParameterNames.FilePath, string.Empty);
            SetInstanceWaveEncodingValue(ProjectParameterNames.WaveEncoding,
                                            WaveEncoding.Adpcm);

            //this.SetInstanceLoopTypeValue(ProjectParameterNames.LoopType, LoopType.InFile);
            //this.SetInstanceIntValue(ProjectParameterNames.LoopStartFrame, 0);
            //this.SetInstanceIntValue(ProjectParameterNames.LoopEndFrame, 0);

            SetInstanceIntValue(ProjectParameterNames.VelocityRegion.OriginalKey, 60, 0, 127);

            EnvelopeParameterValue envelope = new EnvelopeParameterValue(
                new Envelope()
                {
                    Attack = 127,
                    Hold = 0,
                    Decay = 127,
                    Sustain = 127,
                    Release = 127,
                });
            Parameters.AddValue(ProjectParameterNames.EnvelopeParams, envelope);

            SetInstanceIntValue(ProjectParameterNames.VelocityRegion.VelocityMin, 0, 0, 127);
            SetInstanceIntValue(ProjectParameterNames.VelocityRegion.VelocityMax, 0, 0, 127);
            SetInstanceIntValue(ProjectParameterNames.Volume, 127, 0, 255);
            SetInstanceIntValue(ProjectParameterNames.Pan, 64, 0, 127);
            SetInstanceIntValue(ProjectParameterNames.VelocityRegion.PitchSemitones, 0, -50, 50);
            SetInstanceIntValue(ProjectParameterNames.VelocityRegion.PitchCents, 0, -100, 100);
            SetInstanceIntValue(ProjectParameterNames.VelocityRegion.KeyGroup, 0, 0, 15);
            SetInstanceParameterValue(ProjectParameterNames.VelocityRegion.InterpolationType,
                                      new InterpolationTypeParameterValue(InterpolationType.Polyphase));
            SetInstanceBoolValue(ProjectParameterNames.VelocityRegion.PercussionMode, false);
            SetInstanceIntValue(ParameterNameOfDataSize, -1);

            // バンクバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.FilePath, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.WaveEncoding, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.OriginalKey, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.EnvelopeParams, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.VelocityMin, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.VelocityMax, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.Volume, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.Pan, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.PitchSemitones, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.PitchCents, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.KeyGroup, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.InterpolationType, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.PercussionMode, ParameterAttributes.ComputeHash);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceWaveEncodingValue(string name, WaveEncoding value)
        {
            SetInstanceParameterValue(name, new WaveEncodingParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceLoopTypeValue(string name, LoopType value)
        {
            SetInstanceParameterValue(name, new LoopTypeParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnParameterValueChanged(ParameterEventArgs e)
        {
            base.OnParameterValueChanged(e);

            if (e.Key == ProjectParameterNames.FilePath)
            {
                WaveFile = null;
                this.Router.BubbleEvent(new VelocityRegionFilePathChangedEventArgs());
            }
        }
    }
}
