﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.ComponentModel;
    using System.Text;
    using Core;
    using Core.Parameters;
    using ToolDevelopmentKit.Collections;

    /// <summary>
    /// シーケンスサウンドのモデルクラスです。
    /// </summary>
    public class SequenceSoundBase : Sound
    {
        public const int BankReferenceCount = 4;

        private List<string> SavedSoundSetBankNames = new List<string>();
        private List<SoundSetBankBase> soundSetBanks = new List<SoundSetBankBase>();

        /// <summary>
        ///
        /// </summary>
        public SequenceSoundBase()
        {
        }

        /// <summary>
        /// ファイルタイプ
        /// </summary>
        public SequenceSoundFileType FileType
        {
            get
            {
                return GetSequenceSoundFileTypeValue(ProjectParameterNames.SequenceSound.FileType);
            }
            set
            {
                SetSequenceSoundFileTypeValue(ProjectParameterNames.SequenceSound.FileType, value);
            }
        }

        /// <summary>
        /// スタート位置
        /// </summary>
        public string StartPosition
        {
            get { return GetTextValue(ProjectParameterNames.SequenceSound.StartPosition); }
            set { SetTextValue(ProjectParameterNames.SequenceSound.StartPosition, value); }
        }

        /// <summary>
        /// バンク参照を取得または設定します。
        /// </summary>
        public ObservableList<ComponentReference> SoundSetBankReferences
        {
            get
            {
                string name = ProjectParameterNames.SequenceSound.SoundSetBankReferences;
                return GetValue<ObservableList<ComponentReference>>(name);
            }
            set
            {
                string name = ProjectParameterNames.SequenceSound.SoundSetBankReferences;
                SetValue<ObservableList<ComponentReference>>(name, value);
            }
        }

        /// <summary>
        /// チャンネル優先度
        /// </summary>
        public int ChannelPriority
        {
            get { return GetIntValue(ProjectParameterNames.Sound.ChannelPriority); }
            set { SetIntValue(ProjectParameterNames.Sound.ChannelPriority, value); }
        }

        /// <summary>
        /// リリース時優先度固定
        /// </summary>
        public bool ReleasePriorityFixed
        {
            get { return GetBoolValue(ProjectParameterNames.Sound.ReleasePriorityFixed); }
            set { SetBoolValue(ProjectParameterNames.Sound.ReleasePriorityFixed, value); }
        }

        /// <summary>
        /// フロントバイパス
        /// </summary>
        public bool FrontBypass
        {
            get { return this.GetBoolValue(ProjectParameterNames.FrontBypass); }
            set { this.SetBoolValue(ProjectParameterNames.FrontBypass, value); }
        }

        /// <summary>
        /// 最大発音数
        /// </summary>
        public uint? MaximumVoiceCount
        {
            get; set;
        }

        /// <summary>
        ///
        /// </summary>
        public IList<SoundSetBankBase> SoundSetBanks
        {
            get { return this.soundSetBanks; }
        }

        /// <summary>
        ///
        /// </summary>
        public override bool IsReference(Component component)
        {
            if (base.IsReference(component) != false)
            {
                return true;
            }

            if (!(component is SoundSetBankBase)) { return false; }
            return this.soundSetBanks.Contains(component as SoundSetBankBase);
        }

        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            SetInstanceBoolValue(ProjectParameterNames.SndEdit, false);
            SetInstanceBoolValue(ProjectParameterNames.SndEditEnabled, true);

            SetInstanceSequenceSoundFileTypeValue(ProjectParameterNames.SequenceSound.FileType,
                                                     SequenceSoundFileType.Smf);
            SetInstanceTextValue(ProjectParameterNames.SequenceSound.StartPosition, string.Empty);
            this.Parameters.AddValue
                (ProjectParameterNames.SequenceSound.SoundSetBankReferences,
                  new ComponentReferenceCollection());

            this.Parameters.RemoveValue(ProjectParameterNames.Volume);
            SetInstanceIntValue(ProjectParameterNames.Volume, 64, 0, 255);

            SetInstanceIntValue(ProjectParameterNames.Sound.ChannelPriority, 64, 0, 127);
            SetInstanceBoolValue(ProjectParameterNames.Sound.ReleasePriorityFixed, false);
            SetInstanceBoolValue(ProjectParameterNames.FrontBypass, false);

            // シーケンスサウンドバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.FilePath, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.SequenceSound.FileType, ParameterAttributes.ComputeHash);

            // サウンドアーカイブバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.SequenceSound.SoundSetBankReferences, ParameterAttributes.ComputeSoundArchiveHash);
            this.SetParameterAttributes(ProjectParameterNames.SequenceSound.StartPosition, ParameterAttributes.ComputeSoundArchiveHash);
            this.SetParameterAttributes(ProjectParameterNames.FrontBypass, ParameterAttributes.ComputeSoundArchiveHash);

            // sndedit 転送対象パラメータ
            this.SetParameterAttributes(ProjectParameterNames.Volume, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.ChannelPriority, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.ReleasePriorityFixed, ParameterAttributes.ComputeSndeditTransferHash);
        }

        /// <summary>
        ///
        /// </summary>
        protected SequenceSoundFileType GetSequenceSoundFileTypeValue(string name)
        {
            return (SequenceSoundFileTypeParameterValue)Parameters.GetValue(name);
        }

        /// <summary>
        ///
        /// </summary>
        protected void SetSequenceSoundFileTypeValue(string name, SequenceSoundFileType value)
        {
            Parameters[name].Value = value;
        }

        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceSequenceSoundFileTypeValue(
                                                             string name,
                                                             SequenceSoundFileType value
                                                             )
        {
            Parameters.AddValue(name, new SequenceSoundFileTypeParameterValue(value));
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnParameterValueChanged(ParameterEventArgs e)
        {
            if (SuspendParameterValueChangedEventBubbling != false)
            {
                return;
            }

            base.OnParameterValueChanged(e);

            if (e.Key == ProjectParameterNames.SequenceSound.SoundSetBankReferences)
            {
                this.soundSetBanks.Clear();
                Router.BubbleEvent(new RoutingEventArgs
                                    (new RequestUpdateSoundProjectItemBindingEvent(), this));
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnUpdateSoundProjectItemBinding(ComponentManagerEventArgs e)
        {
            base.OnUpdateSoundProjectItemBinding(e);

            this.soundSetBanks.Clear();

            foreach (ComponentReference bankReference in this.SoundSetBankReferences)
            {
                var soundSetBank = bankReference.TargetName != String.Empty ?
                    this.FindComponent<SoundSetBankBase>(e.ComponentManager, bankReference.TargetName) : null;

                // this.SoundSetBanks と this.SoundSetBankReferences のアイテム数を合わせるため、
                // soundSetBank が null であってもリストに追加します。
                this.soundSetBanks.Add(soundSetBank);
            }
        }

        /// <summary>
        /// アイテムの名前が変更された時に呼ばれます。
        /// </summary>
        protected override void OnUpdateRenamedSoundProjectItemBinding(ComponentManagerRenamedComponentEventArgs e)
        {
            base.OnUpdateRenamedSoundProjectItemBinding(e);

            this.soundSetBanks.Clear();

            foreach (ComponentReference bankReference in this.SoundSetBankReferences)
            {
                string targetName = bankReference.TargetName;

                if (targetName == e.NewName)
                {
                    targetName = e.OldName;
                }

                var soundSetBank = targetName != String.Empty ?
                    this.FindComponent<SoundSetBankBase>(e.ComponentManager, targetName) : null;

                if (soundSetBank != null)
                {
                    SuspendParameterValueChangedEventBubbling = true;
                    bankReference.TargetName = soundSetBank.Name;
                    SuspendParameterValueChangedEventBubbling = false;
                }

                // this.SoundSetBanks と this.SoundSetBankReferences のアイテム数を合わせるため、
                // soundSetBank が null であってもリストに追加します。
                this.soundSetBanks.Add(soundSetBank);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnQuerySoundProjectItemBinding(
            QuerySoundProjectItemBindingEventArgs e)
        {
            base.OnQuerySoundProjectItemBinding(e);

            if (e.BindingComponents.Contains(this)) { return; }

            if (e.Component is SoundSetBankBase &&
                this.SoundSetBanks.Contains(e.Component as SoundSetBankBase))
            {
                e.BindingComponents.Add(this);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnQueryComponentDirty(QueryComponentDirtyEventArgs e)
        {
            base.OnQueryComponentDirty(e);
            if (e.Result) { return; }

            if (this.SoundSetBankReferences.Count != this.SavedSoundSetBankNames.Count)
            {
                e.Result = true;
                return;
            }

            for (int i = 0; i < this.SavedSoundSetBankNames.Count; i++)
            {
                if (this.SoundSetBankReferences[i].TargetName != this.SavedSoundSetBankNames[i])
                {
                    e.Result = true;
                    return;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnResetComponentDirty(RoutingEventArgs e)
        {
            base.OnResetComponentDirty(e);

            this.SavedSoundSetBankNames.Clear();

            foreach (ComponentReference soundSetBankReference in SoundSetBankReferences)
            {
                this.SavedSoundSetBankNames.Add(soundSetBankReference.TargetName);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBankReferencePropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (SuspendParameterValueChangedEventBubbling != false)
            {
                return;
            }

            if (e.PropertyName == ProjectParameterNames.TargetName)
            {
                this.soundSetBanks.Clear();
                Router.BubbleEvent(new RoutingEventArgs
                                   (new RequestUpdateSoundProjectItemBindingEvent(), this));
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ComponentReferenceCollection : CollectionParameterValue<ObservableList<ComponentReference>, ComponentReference>
    {
        /// <summary>
        ///
        /// </summary>
        public ComponentReferenceCollection() : base(new ObservableList<ComponentReference>())
        {
            this.Value.CollectionChanged += OnCollectionChanged;
        }

        /// <summary>
        ///
        /// </summary>
        public override string ToString()
        {
            string text = string.Empty;

            foreach (ComponentReference bankRef in this.Value)
            {
                if (text != string.Empty)
                {
                    text += ",";
                }
                text += bankRef.TargetName;
            }

            return text;
        }

        /// <summary>
        ///
        /// </summary>
        protected override ObservableList<ComponentReference> ParseInternal(string text)
        {
            if (text != string.Empty)
            {
                this.Value.Clear();
                string[] banks = text.Split(new Char[] { ',' });
                foreach (string bank in banks)
                {
                    ComponentReference bankRef = new ComponentReference();
                    bankRef.TargetName = bank;
                    this.Value.Add(bankRef);
                }
            }

            return this.Value;
        }

        /// <summary>
        /// パラメータ値のバイト列を取得します。
        /// </summary>
        /// <returns>バイト列の列挙子を返します。</returns>
        protected override IEnumerable<byte> GetBytes()
        {
            return Encoding.Unicode.GetBytes(this.ToString());
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            OnValueChanged(new ParameterEventArgs(String.Empty, this));
        }

        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    if (e.NewItems != null)
                    {
                        foreach (ComponentReference cmpRef in e.NewItems)
                        {
                            cmpRef.PropertyChanged += OnPropertyChanged;
                        }
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                case NotifyCollectionChangedAction.Reset:
                    if (e.OldItems != null)
                    {
                        foreach (ComponentReference cmpRef in e.OldItems)
                        {
                            cmpRef.PropertyChanged -= OnPropertyChanged;
                        }
                    }
                    break;
            }
        }
    }
}
