﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using System.IO;

    public class Instrument : BankItem
    {
        public const string ParameterNameOfEnvelope = "Envelope";
        public const string ParameterNameOfDataSize = "DataSize";

        public const int ProgramNoMax = 32767;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Instrument()
        {
            RoutingEventBinding<ComponentManagerEventArgs> routingEventBinding =
                new RoutingEventBinding<ComponentManagerEventArgs>
                (
                 typeof(UpdateSoundProjectItemBindingEvent),
                 OnUpdateSoundProjectItemBinding,
                 null
                 );
            Router.Bindings.Add(routingEventBinding);

            RoutingEventBinding<VelocityRegionFilePathChangedEventArgs> routingEventBindingVel =
                new RoutingEventBinding<VelocityRegionFilePathChangedEventArgs>
                (
                 typeof(UpdateVelocityRegionFilePathChangedEvent),
                 OnUpdateSoundProjectItemBinding,
                 null
                 );

            Router.Bindings.Add(routingEventBindingVel);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual bool LinkedRegion { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public InstrumentEnvelopeMode EnvelopeMode
        {
            get { return GetEnvelopeModeValue(ProjectParameterNames.Instrument.EnvelopeMode); }
            set { SetEnvelopeModeValue(ProjectParameterNames.Instrument.EnvelopeMode, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool IsInstListEnvelope
        {
            get { return EnvelopeMode == InstrumentEnvelopeMode.Instrument ? true : false; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool IsVelocityRegionEnvelope
        {
            get { return EnvelopeMode == InstrumentEnvelopeMode.VelocityRegion ? true : false; }
        }

        ///--------------------------------
        /// <summary>
        /// プログラム番号
        /// </summary>
        public int ProgramNo
        {
            get { return GetIntValue(ProjectParameterNames.Instrument.ProgramNo); }
            set { SetIntValue(ProjectParameterNames.Instrument.ProgramNo, value); }
        }

        ///--------------------------------
        /// <summary>
        /// エンベローブ
        /// </summary>
        public Envelope Envelope
        {
            get { return this.GetValue<Envelope>(ProjectParameterNames.EnvelopeParams); }
            set { this.SetValue<Envelope>(ProjectParameterNames.EnvelopeParams, value); }
        }

        ///--------------------------------
        /// <summary>
        /// 音量
        /// </summary>
        public int Volume
        {
            get { return GetIntValue(ProjectParameterNames.Volume); }
            set { SetIntValue(ProjectParameterNames.Volume, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int PitchSemitones
        {
            get { return GetIntValue(ProjectParameterNames.VelocityRegion.PitchSemitones); }
            set { SetIntValue(ProjectParameterNames.VelocityRegion.PitchSemitones, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int PitchCents
        {
            get { return GetIntValue(ProjectParameterNames.VelocityRegion.PitchCents); }
            set { SetIntValue(ProjectParameterNames.VelocityRegion.PitchCents, value); }
        }

        ///--------------------------------
        /// <summary>
        /// データサイズ
        /// </summary>
        public int DataSize
        {
            get { return GetIntValue(ParameterNameOfDataSize); }
            set { SetIntValue(ParameterNameOfDataSize, value); }
        }

        /// <summary>
        /// リサンプルの設定を取得または設定します。
        /// </summary>
        public bool IsResampleEnabled
        {
            get { return this.GetBoolValue(ProjectParameterNames.IsResampleEnabled); }
            set { this.SetBoolValue(ProjectParameterNames.IsResampleEnabled, value); }
        }

        /// <summary>
        /// サンプルレートを取得または設定します。
        /// </summary>
        public int SampleRate
        {
            get { return this.GetIntValue(ProjectParameterNames.SampleRate); }
            set { this.SetValue(ProjectParameterNames.SampleRate, value); }
        }

        /// <summary>
        /// ダウンミックス設定を取得または設定します。
        /// </summary>
        public bool IsDownMixEnabled
        {
            get { return this.GetBoolValue(ProjectParameterNames.IsDownMixEnabled); }
            set { this.SetBoolValue(ProjectParameterNames.IsDownMixEnabled, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateLinkedRegion()
        {
            this.LinkedRegion = true;
            foreach (KeyRegion kRegion in this.Children)
            {
                foreach (VelocityRegion vRegion in kRegion.Children)
                {
                    if (vRegion.FilePath == string.Empty || !File.Exists(vRegion.FilePath))
                    {
                        LinkedRegion = false;
                        return;
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            SetInstanceIntValue(ProjectParameterNames.Instrument.ProgramNo, 0, 0, int.MaxValue);
            SetInstanceBoolValue(ProjectParameterNames.IsResampleEnabled, false);
            SetInstanceParameterValue(ProjectParameterNames.SampleRate, new SampleRateParameterValue());
            SetInstanceBoolValue(ProjectParameterNames.IsDownMixEnabled, false);
            SetInstanceIntValue(ProjectParameterNames.Volume, 127, 0, 255);
            SetInstanceIntValue(ProjectParameterNames.VelocityRegion.PitchSemitones, 0, -50, 50);
            SetInstanceIntValue(ProjectParameterNames.VelocityRegion.PitchCents, 0, -100, 100);

            EnvelopeParameterValue envelope = new EnvelopeParameterValue(
                new Envelope()
                {
                    Attack = 127,
                    Hold = 0,
                    Decay = 127,
                    Sustain = 127,
                    Release = 127,
                });
            Parameters.AddValue(ProjectParameterNames.EnvelopeParams, envelope);

            SetInstanceIntValue(ParameterNameOfDataSize, -1);
            SetInstanceEnvelopeModeValue(
                ProjectParameterNames.Instrument.EnvelopeMode,
                InstrumentEnvelopeMode.Instrument);

            // バンクバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.Instrument.ProgramNo, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.Volume, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.IsResampleEnabled, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.SampleRate, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.IsDownMixEnabled, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.EnvelopeParams, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.Instrument.EnvelopeMode, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.PitchSemitones, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.VelocityRegion.PitchCents, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.IsEnabled, ParameterAttributes.ComputeHash);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected InstrumentEnvelopeMode GetEnvelopeModeValue(string name)
        {
            return (InstrumentEnvelopeModeParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetEnvelopeModeValue(string name, InstrumentEnvelopeMode value)
        {
            Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceEnvelopeModeValue(string name, InstrumentEnvelopeMode value)
        {
            Parameters.AddValue(name, new InstrumentEnvelopeModeParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnUpdateSoundProjectItemBinding(object sender, ComponentManagerEventArgs e)
        {
            UpdateLinkedRegion();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnUpdateSoundProjectItemBinding(
                                                     object sender,
                                                     VelocityRegionFilePathChangedEventArgs e
                                                     )
        {
            UpdateLinkedRegion();
        }
    }

    public class UpdateVelocityRegionFilePathChangedEvent : RoutingEvent
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Type EventArgsType
        {
            get
            {
                return typeof(VelocityRegionFilePathChangedEventArgs);
            }
        }
    }

    public class VelocityRegionFilePathChangedEventArgs : RoutingEventArgs
    {
        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public VelocityRegionFilePathChangedEventArgs()
            : base(new UpdateVelocityRegionFilePathChangedEvent())
        {
        }
    }
}
