﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Diagnostics;
    using System.Linq;
    using System.Security.Cryptography;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// NintendoWare の基本コンポーネントです。
    /// </summary>
    public class Component : INamedTreeObject, IParameterProvider, IEventRoutable
    {
        // パラメータ
        private ParameterDictionary _Parameters;

        // イベント
        private IEventRoutable _eventTreeParent;
        private EventRouter _EventRouter;

        private Component _Parent = null;
        private ComponentList _Children = null;

        ///
        private string _OldName = String.Empty;

        public event NameChangedEventHandler NameChanged;
        public event ParameterEventHandler ParameterValueChanged;

        //-----------------------------------------------------------------

        public Component()
        {
            _Children = new ComponentList(this);
            _Parameters = new ParameterDictionary();
            _EventRouter = new EventRouter(this);

            Parent = null;

            InitializeParameters();

            _Children.CollectionChanged += OnChildrenChanged;
            _Parameters.ParameterValueChanged += OnParameterValueChanged;
        }

        //-----------------------------------------------------------------

        public Component Parent
        {
            get { return _Parent; }
            set
            {
                if (value == _Parent) { return; }
                _Parent = value;
                OnParentChanged(new EventArgs());
            }
        }

        /// <summary>
        /// イベントツリー上の親を取得または設定します。
        /// </summary>
        public IEventRoutable EventTreeParent
        {
            get { return _eventTreeParent; }
            set { _eventTreeParent = value; }
        }

        public ComponentList Children
        {
            get { return _Children; }
        }

        public ItemID ID
        {
            get { return GetValue<ItemID>(ProjectParameterNames.ID); }
            set { SetValue(ProjectParameterNames.ID, value); }
        }

        public virtual string Name
        {
            get { return GetTextValue(ProjectParameterNames.Name); }
            set { SetTextValue(ProjectParameterNames.Name, value); }
        }

        public virtual string Comment
        {
            get { return GetTextValue(ProjectParameterNames.Comment); }
            set { SetTextValue(ProjectParameterNames.Comment, value); }
        }

        public virtual string Comment1
        {
            get { return GetTextValue(ProjectParameterNames.Comment1); }
            set { SetTextValue(ProjectParameterNames.Comment1, value); }
        }

        public virtual string Comment2
        {
            get { return GetTextValue(ProjectParameterNames.Comment2); }
            set { SetTextValue(ProjectParameterNames.Comment2, value); }
        }

        public virtual string Comment3
        {
            get { return GetTextValue(ProjectParameterNames.Comment3); }
            set { SetTextValue(ProjectParameterNames.Comment3, value); }
        }

        public virtual string Comment4
        {
            get { return GetTextValue(ProjectParameterNames.Comment4); }
            set { SetTextValue(ProjectParameterNames.Comment4, value); }
        }

        public virtual string Comment5
        {
            get { return GetTextValue(ProjectParameterNames.Comment5); }
            set { SetTextValue(ProjectParameterNames.Comment5, value); }
        }

        public virtual string Comment6
        {
            get { return GetTextValue(ProjectParameterNames.Comment6); }
            set { SetTextValue(ProjectParameterNames.Comment6, value); }
        }

        public virtual string Comment7
        {
            get { return GetTextValue(ProjectParameterNames.Comment7); }
            set { SetTextValue(ProjectParameterNames.Comment7, value); }
        }

        public virtual string Comment8
        {
            get { return GetTextValue(ProjectParameterNames.Comment8); }
            set { SetTextValue(ProjectParameterNames.Comment8, value); }
        }

        public virtual string Comment9
        {
            get { return GetTextValue(ProjectParameterNames.Comment9); }
            set { SetTextValue(ProjectParameterNames.Comment9, value); }
        }

        /// <summary>
        /// 有効/無効
        /// </summary>
        public virtual bool IsEnabled
        {
            get { return GetBoolValue(ProjectParameterNames.IsEnabled); }
            set { SetBoolValue(ProjectParameterNames.IsEnabled, value); }
        }

        /// <summary>
        /// 階層が有効かどうかを調べます。
        /// </summary>
        public bool IsHierarchyEnabled
        {
            get
            {
                Component component = this;

                while (component != null)
                {
                    if (!component.IsEnabled)
                    {
                        return false;
                    }

                    component = component.Parent;
                }

                return true;
            }
        }

        /// <summary>
        /// カラーのインデックス
        /// </summary>
        public virtual int ColorIndex
        {
            get { return GetIntValue(ProjectParameterNames.ColorIndex); }
            set { SetIntValue(ProjectParameterNames.ColorIndex, value); }
        }

        /// <summary>
        /// ITreeObject
        /// </summary>
        ITreeObject ITreeObject.Parent
        {
            get { return this.Parent; }
            set { this.Parent = value as Component; }
        }

        ///
        ITreeObjectCollection ITreeObject.Children
        {
            get { return this.Children; }
        }

        public IParameterDictionary Parameters
        {
            get { return _Parameters; }
        }

        // --------------------------------
        // IEventRoutable

        /// <summary>
        /// イベントツリー上の親を取得または設定します。
        /// </summary>
        IEventRoutable IEventRoutable.Parent
        {
            get
            {
                if (null != _eventTreeParent) { return _eventTreeParent; }
                return Parent;
            }
        }

        /// <summary>
        /// イベントツリー上の子コレクションを取得します。
        /// </summary>
        IReadOnlyEventRoutableCollection IEventRoutable.Children
        {
            get { return Children; }
        }

        /// <summary>
        /// このオブジェクトに関連付けられたルータを取得します。
        /// </summary>
        EventRouter IEventRoutable.Router
        {
            get { return _EventRouter; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// パラメータキーを含めたパラメータのハッシュコードを取得します。
        /// </summary>
        /// <param name="algorithm">ハッシュアルゴリズムを指定します。</param>
        /// <param name="filter">対象パラメータのフィルタを指定します。</param>
        /// <returns>ハッシュコードを返します。</returns>
        public virtual HashCode GetComponentHashCode(
            HashAlgorithm algorithm,
            Func<IParameterValue, bool> optionalFilter)
        {
            HashCode parametersHash =
                this.Parameters.GetParameterDictionaryHashCode(
                algorithm,
                string.Empty,
                parameterValue => this.IsParameterHashTarget(
                    parameterValue,
                    ParameterAttributes.ComputeHash,
                    ParameterAttributes.OptionalHashTarget,
                    optionalFilter)
                );

            if (this.Children.Count == 0)
            {
                return parametersHash;
            }

            // パラメータハッシュと子コンポーネントハッシュを連結してハッシュ計算します。
            var bytes = parametersHash.Value.Concat(
                this.Children.
                Where<Component>(component => component.IsEnabled).
                Select<Component, HashCode>(component => component.GetComponentHashCode(algorithm, optionalFilter)).
                SelectMany<HashCode, byte>(hashCode => hashCode.Value));
            return new HashCode(algorithm.ComputeHash(bytes.ToArray()));
        }

        /// <summary>
        /// パラメータキーを含めたパラメータのサウンドアーカイブ用ハッシュコードを取得します。
        /// </summary>
        /// <param name="algorithm">ハッシュアルゴリズムを指定します。</param>
        /// <param name="filter">対象パラメータのフィルタを指定します。</param>
        /// <returns>サウンドアーカイブ用ハッシュコードを返します。</returns>
        public virtual HashCode GetComponentSoundArchiveHashCode(
            HashAlgorithm algorithm,
            Func<IParameterValue, bool> optionalFilter)
        {
            HashCode parametersHash =
                this.Parameters.GetParameterDictionaryHashCode(
                algorithm,
                string.Empty,
                parameterValue => this.IsParameterHashTarget(
                    parameterValue,
                    ParameterAttributes.ComputeSoundArchiveHash,
                    ParameterAttributes.OptionalHashTarget,
                    optionalFilter)
                );

            if (this.Children.Count == 0)
            {
                return parametersHash;
            }

            // パラメータハッシュと子コンポーネントハッシュを連結してハッシュ計算します。
            var bytes = parametersHash.Value.Concat(
                this.Children.
                Where<Component>(component => component.IsEnabled).
                Select<Component, HashCode>(component => component.GetComponentSoundArchiveHashCode(algorithm, optionalFilter)).
                SelectMany<HashCode, byte>(hashCode => hashCode.Value));
            return new HashCode(algorithm.ComputeHash(bytes.ToArray()));
        }

        /// <summary>
        /// パラメータキーを含めたパラメータの sndedit 再転送用ハッシュコードを取得します。
        /// </summary>
        /// <param name="algorithm">ハッシュアルゴリズムを指定します。</param>
        /// <param name="filter">対象パラメータのフィルタを指定します。</param>
        /// <returns>サウンドアーカイブ用ハッシュコードを返します。</returns>
        public virtual HashCode GetComponentSndeditTransferHashCode(
            HashAlgorithm algorithm,
            Func<IParameterValue, bool> optionalFilter)
        {
            HashCode parametersHash =
                this.Parameters.GetParameterDictionaryHashCode(
                algorithm,
                string.Empty,
                parameterValue => this.IsParameterHashTarget(
                    parameterValue,
                    ParameterAttributes.ComputeSndeditTransferHash,
                    ParameterAttributes.OptionalHashTarget,
                    optionalFilter)
                );

            if (this.Children.Count == 0)
            {
                return parametersHash;
            }

            // パラメータハッシュと子コンポーネントハッシュを連結してハッシュ計算します。
            var bytes = parametersHash.Value.Concat(
                this.Children.
                Where<Component>(component => component.IsEnabled).
                Select<Component, HashCode>(component => component.GetComponentSndeditTransferHashCode(algorithm, optionalFilter)).
                SelectMany<HashCode, byte>(hashCode => hashCode.Value));
            return new HashCode(algorithm.ComputeHash(bytes.ToArray()));
        }

        protected TComponent FindComponent<TComponent>(ComponentManager componentManager, IEnumerable<string> itemNames)
            where TComponent : Component
        {
            Debug.Assert(null != componentManager, "invalid argument");
            Debug.Assert(null != itemNames, "invalid argument");

            foreach (string itemName in itemNames)
            {
                TComponent item = FindComponent<TComponent>(componentManager, itemName);
                if (null != item) { return item; }
            }

            return null;
        }

        protected TComponent FindComponent<TComponent>(ComponentManager componentManager, string itemName)
            where TComponent : Component
        {
            Debug.Assert(null != componentManager, "invalid argument");
            Debug.Assert(null != itemName, "invalid argument");

            if (!componentManager.ComponentDictionary.Contains(itemName)) { return null; }

            return (from component in componentManager.ComponentDictionary[itemName]
                    where component is TComponent
                    select component as TComponent).FirstOrDefault();
        }

        protected void OnNameChanged(NameChangedEventArgs e)
        {
            if (NameChanged != null)
            {
                NameChanged(this, e);
            }

            _EventRouter.BubbleEvent(new ComponentNameChangedEventArgs(e.OldName, e.NewName));
            _OldName = e.NewName;
        }

        protected virtual void OnParameterValueChanged(ParameterEventArgs e)
        {
            if (e.Key == ProjectParameterNames.Name)
            {
                OnNameChanged(new NameChangedEventArgs(_OldName, Name));
            }

            if (ParameterValueChanged != null)
            {
                ParameterValueChanged(this, e);
            }
        }

        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected virtual void InitializeParameters()
        {
            SetInstanceParameterValue(ProjectParameterNames.ID, new ObjectParameterValue<ItemID>());
            SetInstanceTextValue(ProjectParameterNames.Name, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment1, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment2, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment3, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment4, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment5, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment6, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment7, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment8, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.Comment9, String.Empty);
            SetInstanceIntValue(ProjectParameterNames.ColorIndex, 0, 0, 8);
            SetInstanceBoolValue(ProjectParameterNames.IsEnabled, true);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment1,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment2,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment3,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment4,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment5,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment6,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment7,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment8,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);

            this.SetParameterAttributes(
                ProjectParameterNames.Comment9,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.OptionalHashTarget);
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void ValidateParent(Component parent)
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void ValidateChild(Component child)
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void OnParentChanged(EventArgs e)
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void OnChildrenChanged(NotifyCollectionChangedEventArgs e)
        {
            ComponentChildrenChangedEventArgs routingEvent;

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (Component component in e.NewItems)
                    {
                        component.Parent = this;
                    }

                    routingEvent = new ComponentChildrenChangedEventArgs
                        (ComponentChildrenChangedAction.Add, e.NewItems.Cast<Component>());
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (Component component in e.OldItems)
                    {
                        component.Parent = null;
                    }

                    routingEvent = new ComponentChildrenChangedEventArgs
                            (ComponentChildrenChangedAction.Remove, e.OldItems.Cast<Component>());
                    break;

                case NotifyCollectionChangedAction.Move:
                    routingEvent = new ComponentChildrenChangedEventArgs
                        (ComponentChildrenChangedAction.Move,
                          e.NewItems.Cast<Component>());
                    break;

                default:
                    routingEvent = new ComponentChildrenChangedEventArgs
                        (ComponentChildrenChangedAction.Reset);
                    break;
            }

            _EventRouter.BubbleEvent(routingEvent);
        }

        /// <summary>
        ///
        /// </summary>
        protected EventRouter Router
        {
            get { return _EventRouter; }
        }

        /// <summary>
        /// 指定したパラメータ値を取得します。
        /// </summary>
        /// <typeparam name="TValue">パラメータ値の型を指定します。</typeparam>
        /// <param name="name">パラメータ名を指定します。</param>
        /// <returns>パラメータ値を返します。</returns>
        protected TValue GetValue<TValue>(string name)
        {
            Assertion.Argument.NotNull(name);
            return (TValue)this.Parameters[name].Value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected string GetTextValue(string name)
        {
            return (TextParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected string GetFilePathValue(string name)
        {
            return (FilePathParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int GetIntValue(string name)
        {
            return (IntParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected uint GetUIntValue(string name)
        {
            return (UIntParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ulong GetULongValue(string name)
        {
            return (ULongParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected float GetFloatValue(string name)
        {
            return (FloatParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool GetBoolValue(string name)
        {
            return (BoolParameterValue)Parameters.GetValue(name);
        }

        /// <summary>
        /// 指定したパラメータに値を設定します。
        /// </summary>
        /// <typeparam name="TValue">パラメータ値の型を指定します。</typeparam>
        /// <param name="name">パラメータ名を指定します。</param>
        /// <param name="value">設定する値を指定します。</param>
        protected void SetValue<TValue>(string name, TValue value)
        {
            Assertion.Argument.NotNull(name);
            Assertion.Argument.True(this.Parameters.ContainsKey(name));
            this.Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetTextValue(string name, string value)
        {
            _Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetFilePathValue(string name, string value)
        {
            _Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetIntValue(string name, int value)
        {
            _Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetULongValue(string name, ulong value)
        {
            _Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetFloatValue(string name, float value)
        {
            _Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetBoolValue(string name, bool value)
        {
            _Parameters[name].Value = value;
        }

        /// <summary>
        /// 指定パラメータに属性を設定します。
        /// </summary>
        /// <param name="parameterName">対象のパラメータ名を指定します。</param>
        /// <param name="attributes">属性を指定します。</param>
        protected void SetParameterAttributes(string parameterName, params string[] attributes)
        {
            Assertion.Argument.StringNotEmpty(parameterName);
            Assertion.Argument.True(this.Parameters.ContainsKey(parameterName));
            Assertion.Argument.NotNull(attributes);

            this.SetParameterAttributes(this.Parameters[parameterName], attributes);
        }

        /// <summary>
        /// 指定パラメータに属性を設定します。
        /// </summary>
        /// <param name="parameter">対象のパラメータを指定します。</param>
        /// <param name="attributes">属性を指定します。</param>
        protected void SetParameterAttributes(IParameterValue parameter, params string[] attributes)
        {
            Assertion.Argument.NotNull(parameter);
            Assertion.Argument.NotNull(attributes);

            foreach (string attribute in attributes)
            {
                parameter.Attributes.Add(attribute);
            }
        }

        ///@@@
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceTextValue(string name, string value)
        {
            _Parameters.AddValue(name, new TextParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceFilePathValue(string name, string value)
        {
            _Parameters.AddValue(name, new FilePathParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceIntValue(string name, int value)
        {
            _Parameters.AddValue(name, new IntParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceIntValue(string name, int value, int minValue, int maxValue)
        {
            _Parameters.AddValue(name, new IntParameterValue(value, minValue, maxValue));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceUIntValue(string name, uint value)
        {
            _Parameters.AddValue(name, new UIntParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceULongValue(string name, ulong value)
        {
            _Parameters.AddValue(name, new ULongParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceULongValue(string name, ulong value, ulong minValue, ulong maxValue)
        {
            _Parameters.AddValue(name, new ULongParameterValue(value, minValue, maxValue));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceFloatValue(string name, float value)
        {
            _Parameters.AddValue(name, new FloatParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceFloatValue(string name, float value, float minValue, float maxValue)
        {
            _Parameters.AddValue(name, new FloatParameterValue(value, minValue, maxValue));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceDoubleValue(string name, double value)
        {
            _Parameters.AddValue(name, new DoubleParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceDoubleValue(string name, double value, double minValue, double maxValue)
        {
            _Parameters.AddValue(name, new DoubleParameterValue(value, minValue, maxValue));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceBoolValue(string name, bool value)
        {
            _Parameters.AddValue(name, new BoolParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceParameterValue(string name, IParameterValue value)
        {
            if (null == name) { throw new ArgumentNullException("name"); }
            if (null == value) { throw new ArgumentNullException("value"); }
            _Parameters.AddValue(name, value);
        }

        private bool IsParameterHashTarget(
            IParameterValue parameterValue,
            string targetAttribute,
            string optionalTargetAttribute,
            Func<IParameterValue, bool> optionalFilter)
        {
            if (!parameterValue.Attributes.Contains(targetAttribute))
            {
                // 非ハッシュ計算対象
                return false;
            }

            if (string.IsNullOrEmpty(optionalTargetAttribute) ||
                !parameterValue.Attributes.Contains(optionalTargetAttribute))
            {
                // 必須のハッシュ計算対象
                return true;
            }

            return optionalFilter != null && optionalFilter(parameterValue);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnParameterValueChanged(object sender, ParameterEventArgs e)
        {
            OnParameterValueChanged(e);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnChildrenChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            OnChildrenChanged(e);
        }
    }
}
