﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// サウンドプロジェクトに含まれるコンポーネントを管理します。
    /// </summary>
    public class SoundProjectService : ComponentService
    {
        private const string CockpitExtension = "cockpit";  // HACK : 本当は Traits から注入すべき内容

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="documentService">ドキュメントサービス。</param>
        /// <param name="intermediateOutputTraits">サウンド中間出力の特性を指定します。</param>
        public SoundProjectService(
            DocumentService documentService,
            SoundIntermediateOutputTraits intermediateOutputTraits)
            : base(documentService, new SoundProjectComponentManager())
        {
            Ensure.Argument.NotNull(intermediateOutputTraits);
            this.IntermediateOutputTraits = intermediateOutputTraits;
        }

        /// <summary>
        /// プロジェクト、サウンドセットドキュメントの再読み込み前に発生します。
        /// </summary>
        public event ProjectReloadEventHandler ProjectReloading;

        /// <summary>
        /// プロジェクト、サウンドセットドキュメントの再読み込みすると発生します。
        /// </summary>
        public event ProjectReloadEventHandler ProjectReloaded;

        /// <summary>
        /// プロジェクトドキュメントを開くと発生します。
        /// </summary>
        public event SoundDocumentResourceEventHandler ProjectDocumentOpened;

        /// <summary>
        /// プロジェクトドキュメントを閉じると発生します。
        /// </summary>
        public event SoundDocumentResourceEventHandler ProjectDocumentClosed;

        /// <summary>
        /// サウンドセットドキュメントを開くと発生します。
        /// </summary>
        public event SoundDocumentResourceEventHandler SoundSetDocumentOpened;

        /// <summary>
        /// サウンドセットドキュメントを閉じると発生します。
        /// </summary>
        public event SoundDocumentResourceEventHandler SoundSetDocumentClosed;

        /// <summary>
        /// サウンドセットが追加されると発生します。
        /// </summary>
        public event SoundDocumentEventHandler SoundSetAdded;

        /// <summary>
        /// サウンドセットが削除されると発生します。
        /// </summary>
        public event SoundDocumentEventHandler SoundSetRemoved;

        /// <summary>
        /// 開かれているプロジェクトドキュメントを取得します。
        /// </summary>
        public SoundProjectDocument ProjectDocument
        {
            get { return ComponentManager.ProjectDocument; }
        }

        /// <summary>
        /// 開かれているサウンドセットドキュメントを取得します。
        /// </summary>
        public SoundSetDocument[] SoundSetDocuments
        {
            get { return ComponentManager.SoundSetDocuments.ToArray(); }
        }

        /// <summary>
        /// プロジェクトを取得します。
        /// </summary>
        public SoundProject Project
        {
            get
            {
                if (null == ProjectDocument) { return null; }
                return ProjectDocument.Project;
            }
        }

        /// <summary>
        /// プロジェクトに含まれるサウンドセットの一覧を取得します。
        /// </summary>
        public SoundSet[] SoundSets
        {
            get { return ComponentManager.SoundSets.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれるサウンドセットアイテムの一覧を取得します。
        /// </summary>
        public IEnumerable<SoundSetItem> SoundSetItems
        {
            get { return ComponentManager.SoundSetItems; }
        }

        /// <summary>
        /// プロジェクトに含まれるサウンドの一覧を取得します。
        /// </summary>
        public Sound[] Sounds
        {
            get { return ComponentManager.Sounds.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれる StreamSoundBase の一覧を取得します。
        /// </summary>
        public StreamSoundBase[] StreamSounds
        {
            get { return ComponentManager.StreamSounds.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれる WaveSoundSetBase の一覧を取得します。
        /// </summary>
        public WaveSoundSetBase[] WaveSoundSets
        {
            get { return ComponentManager.WaveSoundSets.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれる WaveSoundBase の一覧を取得します。
        /// </summary>
        public WaveSoundBase[] WaveSounds
        {
            get { return ComponentManager.WaveSounds.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれる SequenceSoundSetBase の一覧を取得します。
        /// </summary>
        public SequenceSoundSetBase[] SequenceSoundSets
        {
            get { return ComponentManager.SequenceSoundSets.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれる SequenceSoundBase の一覧を取得します。
        /// </summary>
        public SequenceSoundBase[] SequenceSounds
        {
            get { return ComponentManager.SequenceSounds.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれるバンクの一覧を取得します。
        /// </summary>
        public SoundSetBankBase[] SoundSetBanks
        {
            get { return ComponentManager.SoundSetBanks.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれる WaveArchiveBase の一覧を取得します。
        /// </summary>
        public WaveArchiveBase[] WaveArchives
        {
            get { return ComponentManager.WaveArchives.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれるプレイヤーの一覧を取得します。
        /// </summary>
        public PlayerBase[] Players
        {
            get { return ComponentManager.Players.ToArray(); }
        }

        /// <summary>
        /// プロジェクトに含まれるグループの一覧を取得します。
        /// </summary>
        public GroupBase[] Groups
        {
            get { return ComponentManager.Groups.ToArray(); }
        }

        /// <summary>
        /// プロジェクトの出力先パスを取得します。
        /// </summary>
        public string ProjectOutputPath
        {
            get
            {
                if (this.ProjectFilePath.Length == 0)
                {
                    return string.Empty;
                }

                return Path.Combine(
                    Path.GetDirectoryName(this.ProjectFilePath),
                    this.Project.OutputDirectoryPath);
            }
        }

        /// <summary>
        /// 外部ファイルの出力先パスを取得します。
        /// </summary>
        public string ProjectExternalFilesOutputPath
        {
            get
            {
                if (this.ProjectOutputPath.Length == 0)
                {
                    return string.Empty;
                }

                return Path.Combine(
                    this.ProjectOutputPath,
                    this.Project.ValidExternalFileOutputDirectoryPath
                ).GetFullPath();
            }
        }

        /// <summary>
        /// プロジェクトの中間出力先パスを取得します。
        /// </summary>
        public string ProjectIntermediateOutputPath
        {
            get
            {
                if (this.ProjectFilePath.Length == 0)
                {
                    return string.Empty;
                }

                return Path.Combine(
                    Path.GetDirectoryName(this.ProjectFilePath),
                    this.Project.IntermediateOutputPath);
            }
        }

        /// <summary>
        /// プロジェクトのインゲーム編集キャッシュの出力先パスを取得します。
        /// </summary>
        public string ProjectInGameEditCacheOutputPath
        {
            get
            {
                if (this.ProjectFilePath.Length == 0)
                {
                    return string.Empty;
                }

                return Path.Combine(
                    Path.GetDirectoryName(this.ProjectFilePath),
                    this.Project.InGameEditCacheOutputPath);
            }
        }

        /// <summary>
        /// プロジェクトファイルパスを取得します。
        /// </summary>
        public string ProjectFilePath
        {
            get
            {
                if (this.ProjectDocument == null)
                {
                    return string.Empty;
                }

                return this.ProjectDocument.Resource.Key.GetFullPath();
            }
        }

        ///
        public void SendUpdateSoundProjectItemBindingEvent()
        {
            ComponentManager.SendUpdateSoundProjectItemBindingEvent();
        }

        public Component[] QueryComponentBindings(Component component)
        {
            return ComponentManager.QueryComponentBindings(component);
        }


        /// <summary>
        /// コンポーネントマネージャを取得します。
        /// </summary>
        protected new SoundProjectComponentManager ComponentManager
        {
            get { return base.ComponentManager as SoundProjectComponentManager; }
        }

        /// <summary>
        /// 開かれているサウンドセットのファイルパスを取得します。
        /// </summary>
        private IEnumerable<string> SoundSetFilePaths
        {
            get
            {
                return this.SoundSetDocuments.Select(d => d.Resource.Key);
            }
        }

        /// <summary>
        /// サウンド中間出力の特性を取得します。
        /// </summary>
        private SoundIntermediateOutputTraits IntermediateOutputTraits { get; set; }

        /// <summary>
        /// 新しいサウンドプロジェクトを作成します。
        /// </summary>
        /// <param name="filePath">サウンドプロジェクトファイルパス。</param>
        public void Create(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            Create(new FileResource(filePath));
        }

        /// <summary>
        /// サウンドプロジェクトファイルを開きます。
        /// </summary>
        /// <param name="filePath">サウンドプロジェクトファイルパス。</param>
        public void Open(string filePath, bool readOnly = false)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            Open(new FileResource(filePath), readOnly);
        }

        public string GetPreprocessedTag()
        {
            if (string.IsNullOrEmpty(this.ProjectFilePath))
            {
                return string.Empty;
            }

            string cockpitFilePath = Path.ChangeExtension(this.ProjectFilePath, CockpitExtension);

            if (!File.Exists(cockpitFilePath))
            {
                return string.Empty;
            }

            using (var fileStream = File.Open(cockpitFilePath, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                using (var reader = new StreamReader(fileStream))
                {
                    return reader.ReadLine();
                }
            }
        }

        /// <summary>
        /// サウンドプロジェクトファイルを再読み込みします。
        /// </summary>
        /// <param name="func">true を返すと再読み込みします</param>
        public void ReloadProject(Func<DateTime, string, bool> func)
        {
            bool isReloaded = false;
            FileResource resource = this.ProjectDocument.Resource as FileResource;

            this.OnProjectReloading(new ProjectReloadEventArgs());

            if (func(resource.LastWriteTime, resource.Key) == true)
            {
                // プロジェクトファイルを再読み込みします。
                this.ReloadProjectFile(resource.Key);
                isReloaded = true;
            }

            // サウンドセットファイルを再読み込みします。
            this.SoundSetDocuments
                .Where(d => func((d.Resource as FileResource).LastWriteTime, d.Resource.Key) == true)
                .ForEach(d => { this.ReloadSoundSetFile(d.Resource.Key); isReloaded = true; });

            this.OnProjectReloaded(new ProjectReloadEventArgs(isReloaded));
        }

        /// <summary>
        /// サウンドプロジェクトファイルを再読み込みします。
        /// </summary>
        /// <param name="filePath">サウンドプロジェクトファイルパス。</param>
        private void ReloadProjectFile(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            using (DocumentReference documentRef =
                DocumentService.OpenDocument(new FileResource(filePath)))
            {
                if (null == documentRef)
                {
                    throw new FileNotSupportedException(filePath);
                }

                if (documentRef == null ||
                    !(documentRef.Document is SoundProjectDocument))
                {
                    throw new FileNotSupportedException(filePath);
                }

                // リロードできるか GetDocumentReader で試す。
                IDocumentReader reader = DocumentService.Traits.GetDocumentReader(new FileResource(filePath));
                if (reader == null)
                {
                    throw new FileNotSupportedException(filePath);
                }
                foreach (string targetFilePath in this.SoundSetFilePaths)
                {
                    reader = DocumentService.Traits.GetDocumentReader(new FileResource(targetFilePath));
                    if (reader == null)
                    {
                        throw new FileNotSupportedException(filePath);
                    }
                }

                List<DocumentReference> docRefs = new List<DocumentReference>();

                try
                {
                    this.BeginDelayComponentsRemoveEvent();

                    // SoundSet 削除

                    foreach (string targetFilePath in this.SoundSetFilePaths.ToArray())
                    {
                        using (DocumentReference docRef =
                               DocumentService.OpenDocument(new FileResource(targetFilePath)))
                        {
                            docRefs.Add(docRef.Clone());
                        }
                        ComponentManager.RemoveDocument(targetFilePath);
                    }

                    this.EndDelayComponentsRemoveEvent();
                }
                catch
                {
                    this.CancelDelayComponentsRemoveEvent();
                    throw;
                }

                // SoundProject 削除、リロード、追加

                ComponentManager.RemoveDocument(filePath);

                DocumentService.ReloadDocument(documentRef.Document);

                try
                {
                    // SoundProject
                    this.BeginDelayComponentsAddEvent();
                    ComponentManager.AddDocument(documentRef);

                    // SoundSets
                    OpenSoundSets(documentRef.Document as SoundProjectDocument);
                    this.EndDelayComponentsAddEvent();

                    ComponentManager.SendUpdateSoundProjectItemBindingEvent();
                }
                catch (Exception)
                {
                    this.CancelDelayComponentsAddEvent();
                    ComponentManager.ClearDocuments();
                    throw;
                }
                finally
                {
                    foreach (DocumentReference docRef in docRefs)
                    {
                        this.OnSoundSetDocumentClosed(new SoundDocumentResourceEventArgs(docRef.Document.Resource));
                        docRef.Close();
                    }
                }
            }

            this.SendUpdateSoundProjectItemBindingEvent();
        }

        /// <summary>
        /// サウンドセットファイルを再読み込みします。
        /// </summary>
        /// <param name="filePath">サウンドセットファイルパス。</param>
        private void ReloadSoundSetFile(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            if (!ContainsSoundSetDocument(filePath))
            {
                throw new ArgumentException("document not found in this project.");
            }

            // リロードできるか GetDocumentReader で試す。
            IDocumentReader reader = DocumentService.Traits.GetDocumentReader(new FileResource(filePath));
            if (reader == null)
            {
                throw new FileNotSupportedException(filePath);
            }
            //

            using (DocumentReference documentRef =
                        DocumentService.OpenDocument(new FileResource(filePath)))
            {

                if (!(documentRef.Document is SoundSetDocument))
                {
                    throw new ApplicationException(
                        string.Format(Resources.MessageResource.Message_UnsupportedFileFormat, filePath));
                }

                SoundSetDocument soundSetDocument = documentRef.Document as SoundSetDocument;
                string soundSetName = soundSetDocument.SoundSet.Name;
                Component parent = soundSetDocument.SoundSet.Parent;
                int index = parent.Children.IndexOf(soundSetDocument.SoundSet);

                try
                {
                    this.BeginDelayComponentsRemoveEvent();
                    ComponentManager.RemoveDocument(filePath);
                    this.EndDelayComponentsRemoveEvent();
                    this.OnSoundSetDocumentClosed(new SoundDocumentResourceEventArgs(soundSetDocument.Resource));
                }
                catch
                {
                    this.CancelDelayComponentsRemoveEvent();
                    throw;
                }

                DocumentService.ReloadDocument(soundSetDocument);
                soundSetDocument.SoundSet.Name = soundSetName;
                var projectSoundSetComponent = new ProjectSoundSetComponent(soundSetName, filePath);
                parent.Children.Insert(index, projectSoundSetComponent);

                try
                {
                    this.BeginDelayComponentsAddEvent();
                    ComponentManager.AddDocument(documentRef);
                    this.EndDelayComponentsAddEvent();
                    this.OnSoundSetDocumentOpened(new SoundDocumentResourceEventArgs(soundSetDocument.Resource));
                }
                catch
                {
                    this.CancelDelayComponentsAddEvent();
                    throw;
                }
            }

            this.SendUpdateSoundProjectItemBindingEvent();
        }

        /// <summary>
        /// 新しいサウンドセットを作成してプロジェクトに追加ます。
        /// </summary>
        /// <param name="filePath">サウンドセットファイルパス。</param>
        public void AddNewSoundSet(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            using (DocumentReference documentRef = DocumentService.CreateDocument(
                this.IntermediateOutputTraits.SoundSetDocumentTypeName))
            {

                if (null == documentRef ||
                    !(documentRef.Document is SoundSetDocument))
                {
                    throw new ArgumentException("documentTypeName is invalid.");
                }

                documentRef.Document.Resource = new FileResource(filePath);

                (documentRef.Document as SoundSetDocument).SoundSet = NewSoundSet(filePath);

                DocumentService.SaveDocument(documentRef.Document);
                AddSoundSet(documentRef);

                OnSoundSetAdded(new SoundDocumentEventArgs(documentRef.Document as SoundDocument));
                this.OnSoundSetDocumentOpened(new SoundDocumentResourceEventArgs(documentRef.Document.Resource));
            }
        }

        /// <summary>
        /// サウンドセットをプロジェクトに追加します。
        /// </summary>
        /// <param name="filePath">サウンドセットファイルパス。</param>
        /// <param name="parentCompoent">追加するサウンドセットの親Componentの指定です。</param>
        public void AddSoundSet(string filePath, Component parentComponent)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            using (DocumentReference documentRef =
                        DocumentService.OpenDocument(new FileResource(filePath)))
            {

                if (null == documentRef) { throw new FileNotSupportedException(filePath); }

                InsertSoundSet(documentRef, -1, parentComponent);

                SoundSet soundSet = ((SoundSetDocument)documentRef.Document).SoundSet;
                soundSet.Name = Path.GetFileNameWithoutExtension(filePath);
            }
        }

        /// <summary>
        /// サウンドセットをプロジェクトに追加します。
        /// </summary>
        /// <param name="documentRef">サウンドセットドキュメントへの参照。</param>
        private void AddSoundSet(DocumentReference documentRef)
        {
            InsertSoundSet(documentRef, -1, this.Project);
        }

        /// <summary>
        /// サウンドセットをプロジェクトに追加します。
        /// </summary>
        /// <param name="filePath">サウンドセットファイルパス。</param>
        /// <param name="insertIndex">サウンドセットの挿入先インデックス</param>
        /// <param name="parentCompoent">追加するサウンドセットの親Componentの指定です。</param>
        public void InsertSoundSet(string filePath, int insertIndex, Component parentComponent)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            using (DocumentReference documentRef =
                        DocumentService.OpenDocument(new FileResource(filePath)))
            {

                if (null == documentRef) { throw new FileNotSupportedException(filePath); }

                InsertSoundSet(documentRef, insertIndex, parentComponent);

                SoundSet soundSet = ((SoundSetDocument)documentRef.Document).SoundSet;
                soundSet.Name = Path.GetFileNameWithoutExtension(filePath);
            }
        }

        /// <summary>
        /// サウンドセットをプロジェクトに追加します。
        /// </summary>
        /// <param name="documentRef">サウンドセットドキュメントへの参照。</param>
        /// <param name="insertIndex">挿入先インデックス</param>
        /// <param name="parentCompoent">追加するサウンドセットの親Componentの指定です。</param>
        private void InsertSoundSet(DocumentReference documentRef, int insertIndex, Component parentComponent)
        {
            if (null == documentRef) { throw new ArgumentNullException("documentRef"); }
            if (null == documentRef.Document)
            {
                throw new ArgumentNullException("documentRef.Document");
            }
            if (-1 > insertIndex) { throw new ArgumentException("insertIndex must not be less than -1."); }
            if (Project.Children.Count < insertIndex)
            {
                throw new ArgumentException("insertIndex must be less than SoundSet Count.");
            }

            SoundSetDocument document = documentRef.Document as SoundSetDocument;
            if (null == document)
            {
                throw new ArgumentException("documentRef.Document.SoundSet must be SoundSetDocument.");
            }
            if (null == document.SoundSet)
            {
                throw new ArgumentException("documentRef.Document.SoundSet must not be null.");
            }
            if (null == document.Resource)
            {
                throw new ArgumentException("documentRef.Document.Resource must not be null.");
            }
            if (0 == document.Resource.Key.Length)
            {
                throw new ArgumentException("documentRef.Document.Resource.Key.Length must not be 0.");
            }

            ProjectSoundSetComponent component = new ProjectSoundSetComponent(document.SoundSet.Name, document.Resource.Key);

            if (-1 == insertIndex)
            {
                parentComponent.Children.Add(component);
            }
            else
            {
                parentComponent.Children.Insert(insertIndex, component);
            }

            try
            {
                this.BeginDelayComponentsAddEvent();
                ComponentManager.AddDocument(documentRef);
                this.EndDelayComponentsAddEvent();
            }
            catch
            {
                this.CancelDelayComponentsAddEvent();
                throw;
            }

            // 以下は OnSoundSetDocumentOpened, OnSoundSetAdded の順で呼ばれるのが正しいと思うが、
            // 修正するにも検証が必要です。現時点ではこのまま。
            OnSoundSetAdded(new SoundDocumentEventArgs(document));
            this.OnSoundSetDocumentOpened(new SoundDocumentResourceEventArgs(document.Resource));
        }

        /// <summary>
        /// サウンドセットをプロジェクトから削除します。
        /// </summary>
        /// <param name="filePath">サウンドセットファイルパス。</param>
        public void RemoveSoundSet(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            if (!ComponentManager.Documents.ContainsKey(filePath)) { return; }

            SoundDocument target = ComponentManager.Documents[filePath];

            try
            {
                this.BeginDelayComponentsRemoveEvent();
                ComponentManager.RemoveDocument(filePath);
                this.EndDelayComponentsRemoveEvent();
            }
            catch
            {
                this.CancelDelayComponentsRemoveEvent();
                throw;
            }

            OnSoundSetRemoved(new SoundDocumentEventArgs(target));
            this.OnSoundSetDocumentClosed(new SoundDocumentResourceEventArgs(new FileResource(filePath)));
        }

        public void AddSoundArchiveOutputType(string filePath, SoundArchiveOutputTypes outputType)
        {
            var soundSet = this.SoundSetDocuments
                .Where(d => d.Resource.Key == filePath)
                .First()
                .SoundSet;

            this.AddSoundArchiveOutputType(soundSet, outputType);
        }

        public void AddSoundArchiveOutputType(SoundSet soundSet, SoundArchiveOutputTypes outputType)
        {
            this.ProjectDocument.Project.SoundArchiveOutputTypes.Add(soundSet, outputType);
        }

        public bool RemoveSoundArchiveOutputType(string filePath, ref SoundArchiveOutputTypes outputType)
        {
            var project = this.ProjectDocument.Project;
            var soundSet = this.SoundSetDocuments
                .Where(d => d.Resource.Key == filePath)
                .First()
                .SoundSet;

            if (project.SoundArchiveOutputTypes.ContainsKey(soundSet) == false)
            {
                return false;
            }

            outputType = project.SoundArchiveOutputTypes[soundSet];
            project.SoundArchiveOutputTypes.Remove(soundSet);
            return true;
        }

        public SoundArchiveOutputTypes GetSoundArchiveOutputType(SoundSet soundSet)
        {
            if (Project.SoundArchiveOutputTypes.ContainsKey(soundSet) == true)
            {
                return Project.SoundArchiveOutputTypes[soundSet];
            }
            else
            {
                return SoundArchiveOutputTypes.SoundArchive;
            }
        }

        public void SetSoundArchiveOutputType(SoundSet soundSet, SoundArchiveOutputTypes outputType)
        {
            if (Project.SoundArchiveOutputTypes.ContainsKey(soundSet) == true)
            {
                Project.SoundArchiveOutputTypes[soundSet] = outputType;
            }
            else
            {
                AddSoundArchiveOutputType(soundSet, outputType);
            }
        }

        /// <summary>
        /// サウンドセットファイルがプロジェクトサービスに含まれるかどうか調べます。
        /// </summary>
        /// <param name="filePath">サウンドセットファイルのファイルパス。</param>
        /// <returns>含まれる場合は true、含まれない場合は false。</returns>
        public bool ContainsSoundSetDocument(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            return this.SoundSetFilePaths.Any(p => p == filePath);
        }

        /// <summary>
        /// サウンドプロジェクトリソースを作成します。
        /// </summary>
        /// <param name="filePath">サウンドプロジェクトファイルリソース。</param>
        protected override void CreateInternal(IStreamResource resource)
        {
            if (!(resource is FileResource)) { throw new ArgumentException("resource must be FileResource."); }
            if (null == resource.Key) { throw new ArgumentException("resource.Key must not be null."); }

            if (null != ProjectDocument)
            {
                throw new Exception("sound project is already opened.");
            }

            using (DocumentReference documentRef = DocumentService.CreateDocument(
                this.IntermediateOutputTraits.SoundProjectDocumentTypeName))
            {

                if (null == documentRef ||
                    !(documentRef.Document is SoundProjectDocument))
                {
                    throw new ArgumentException("documentTypeName is invalid.");
                }

                documentRef.Document.Resource = resource;

                (documentRef.Document as SoundProjectDocument).Project =
                    new SoundProject()
                    {
                        Name = Path.GetFileNameWithoutExtension(resource.Key),
                    };

                DocumentService.SaveDocument(documentRef.Document);
                ComponentManager.AddDocument(documentRef);
                ComponentManager.SendResetComponentDirtyEvent();
            }
        }

        /// <summary>
        /// サウンドプロジェクトリソースを開きます。
        /// </summary>
        /// <param name="filePath">サウンドプロジェクトファイルリソース。</param>
        protected override bool OpenInternal(IStreamResource resource, bool readOnly = false)
        {
            if (!(resource is FileResource)) { throw new ArgumentException("resource must be FileResource."); }
            if (null == resource.Key) { throw new ArgumentException("resource.Key must not be null."); }

            if (null != ProjectDocument)
            {
                throw new Exception("sound project is already opened.");
            }

            using (DocumentReference documentRef =
                DocumentService.OpenDocument(new FileResource(resource.Key)))
            {
                if (documentRef == null ||
                    documentRef.Document is SoundProjectDocument == false ||
                    documentRef.Document.TypeName != this.IntermediateOutputTraits.SoundProjectDocumentTypeName)
                {
                    throw new FileNotSupportedException(resource.Key);
                }

                (documentRef.Document as SoundProjectDocument).IsReadOnly = readOnly;
                ComponentManager.AddDocument(documentRef);

                try
                {
                    OpenSoundSets(documentRef.Document as SoundProjectDocument);
                    ComponentManager.SendUpdateSoundProjectItemBindingEvent();
                    ComponentManager.SendResetComponentDirtyEvent();
                }
                catch (Exception)
                {
                    ComponentManager.ClearDocuments();
                    throw;
                }

                return true;
            }
        }

        /// <summary>
        /// サウンドプロジェクトリソースに保存します。
        /// </summary>
        protected override void SaveInternal()
        {
            if (null == ProjectDocument)
            {
                throw new Exception("souind project is not opened.");
            }

            foreach (SoundDocument document in Documents.Values)
            {
                if (!document.IsDirty) { continue; }
                DocumentService.SaveDocument(document);
            }

            ComponentManager.SendResetComponentDirtyEvent();
        }

        /// <summary>
        /// サウンドプロジェクトを閉じます。
        /// </summary>
        protected override void CloseInternal()
        {
        }

        /// <summary>
        /// リソースが開かれると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnOpened(EventArgs e)
        {
            base.OnOpened(e);

            this.OnProjectDocumentOpened(new SoundDocumentResourceEventArgs(this.ProjectDocument.Resource));
        }

        /// <summary>
        /// リソースが閉じられると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnClosed(EventArgs e)
        {
            IStreamResource resource = this.ProjectDocument.Resource;

            this.BeginDelayComponentsRemoveEvent();
            ComponentManager.ClearDocuments();
            this.EndDelayComponentsRemoveEvent();

            base.OnClosed(e);

            this.OnProjectDocumentClosed(new SoundDocumentResourceEventArgs(resource));
        }

        /// <summary>
        /// プロジェクト、サウンドセットドキュメントの再読み込み前に発生します。
        /// </summary>
        /// <param name="e">ドキュメントイベントデータ。</param>
        protected virtual void OnProjectReloading(ProjectReloadEventArgs e)
        {
            if (this.ProjectReloading != null)
            {
                this.ProjectReloading(this, e);
            }
        }

        /// <summary>
        /// プロジェクト、サウンドセットドキュメントの再読み込みが完了すると発生します。
        /// </summary>
        /// <param name="e">ドキュメントイベントデータ。</param>
        protected virtual void OnProjectReloaded(ProjectReloadEventArgs e)
        {
            if (ProjectReloaded != null)
            {
                ProjectReloaded(this, e);
            }
        }

        /// <summary>
        /// プロジェクトドキュメントを開くと発生します。
        /// </summary>
        /// <param name="e">ドキュメントイベントデータ。</param>
        protected virtual void OnProjectDocumentOpened(SoundDocumentResourceEventArgs e)
        {
            if (ProjectDocumentOpened != null)
            {
                ProjectDocumentOpened(this, e);
            }
        }

        /// <summary>
        /// プロジェクトドキュメントを閉じると発生します。
        /// </summary>
        /// <param name="e">ドキュメントイベントデータ。</param>
        protected virtual void OnProjectDocumentClosed(SoundDocumentResourceEventArgs e)
        {
            if (ProjectDocumentClosed != null)
            {
                ProjectDocumentClosed(this, e);
            }
        }

        /// <summary>
        /// サウンドセットドキュメントを開くと発生します。
        /// </summary>
        /// <param name="e">ドキュメントイベントデータ。</param>
        protected virtual void OnSoundSetDocumentOpened(SoundDocumentResourceEventArgs e)
        {
            if (SoundSetDocumentOpened != null)
            {
                SoundSetDocumentOpened(this, e);
            }
        }

        /// <summary>
        /// サウンドセットドキュメントを閉じると発生します。
        /// </summary>
        /// <param name="e">ドキュメントイベントデータ。</param>
        protected virtual void OnSoundSetDocumentClosed(SoundDocumentResourceEventArgs e)
        {
            if (SoundSetDocumentClosed != null)
            {
                SoundSetDocumentClosed(this, e);
            }
        }

        /// <summary>
        /// サウンドセットが追加されると発生します。
        /// </summary>
        /// <param name="e">ドキュメントイベントデータ。</param>
        protected virtual void OnSoundSetAdded(SoundDocumentEventArgs e)
        {
            if (null != SoundSetAdded)
            {
                SoundSetAdded(this, e);
            }
        }

        /// <summary>
        /// サウンドセットが削除されると発生します。
        /// </summary>
        /// <param name="e">ドキュメントイベントデータ。</param>
        protected virtual void OnSoundSetRemoved(SoundDocumentEventArgs e)
        {
            if (null != SoundSetRemoved)
            {
                SoundSetRemoved(this, e);
            }
        }

        /// <summary>
        /// サウンドセットを開きます。
        /// </summary>
        /// <param name="document">プロジェクトドキュメント。</param>
        private void OpenSoundSets(SoundProjectDocument document)
        {
            if (null == document) { throw new ArgumentNullException("document"); }

            foreach (string targetFilePath in document.Project.GetProjectSoundSetComponentFilePaths())
            {
                OpenSoundSet(targetFilePath);
            }
        }

        /// <summary>
        /// サウンドセットを開きます。
        /// </summary>
        /// <param name="filePaths">サウンドセットファイルパス。</param>
        private void OpenSoundSet(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            using (DocumentReference documentRef =
                        DocumentService.OpenDocument(new FileResource(filePath)))
            {
                if (documentRef == null ||
                    documentRef.Document is SoundSetDocument == false ||
                    documentRef.Document.TypeName != this.IntermediateOutputTraits.SoundSetDocumentTypeName)
                {
                    throw new FileNotSupportedException(filePath);
                }

                ComponentManager.AddDocument(documentRef);
                this.OnSoundSetDocumentOpened(new SoundDocumentResourceEventArgs(documentRef.Document.Resource));
            }
        }

        /// <summary>
        /// 新しいサウンドセットを作成します。
        /// </summary>
        private SoundSet NewSoundSet(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            SoundSet soundSet = new SoundSet()
            {
                Name = Path.GetFileNameWithoutExtension(filePath),
            };

            soundSet.Children.Add(new StreamSoundPack());
            soundSet.Children.Add(new WaveSoundSetPack());
            soundSet.Children.Add(new SequenceSoundPack());
            soundSet.Children.Add(new SequenceSoundSetPack());
            soundSet.Children.Add(new SoundSetBankPack());
            soundSet.Children.Add(new WaveArchivePack());
            soundSet.Children.Add(new GroupPack());
            soundSet.Children.Add(new PlayerPack());

            return soundSet;
        }
    }
}
