﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// バンクサービスを管理します。
    /// </summary>
    public class BankServiceManager
    {
        private DocumentService documentService;
        private Dictionary<string, BankService> bankServices = new Dictionary<string, BankService>();
        private Dictionary<BankService, BankServiceItem> items = new Dictionary<BankService, BankServiceItem>();

        public BankServiceManager(
            DocumentService documentService,
            SoundIntermediateOutputTraits intermediateOutputTraits)
        {
            Ensure.Argument.NotNull(documentService);
            Ensure.Argument.NotNull(intermediateOutputTraits);

            this.documentService = documentService;
            this.IntermediateOutputTraits = intermediateOutputTraits;
        }

        /// <summary>
        /// バンクサービスが追加されると発生します。
        /// </summary>
        public event BankServiceEventHandler Added;

        /// <summary>
        /// バンクサービスが削除されると発生します。
        /// </summary>
        public event BankServiceEventHandler Removed;

        /// <summary>
        /// バンクサービスの一覧を取得します。
        /// </summary>
        public IEnumerable<BankService> Values
        {
            get { return this.bankServices.Values; }
        }

        /// <summary>
        /// バンクファイルパスからバンクサービスを取得します。
        /// </summary>
        /// <param name="bankFilePath">バンクファイルパス。</param>
        /// <returns>バンクサービス。</returns>
        public BankService this[string bankFilePath]
        {
            get
            {
                Ensure.Argument.NotNull(bankFilePath);
                return this.bankServices[bankFilePath];
            }
        }

        /// <summary>
        /// サウンド中間出力の特性を取得します。
        /// </summary>
        private SoundIntermediateOutputTraits IntermediateOutputTraits { get; set; }

        /// <summary>
        /// バンクファイルパスに関連するバンクサービスがディクショナリに存在するかどうかを調べます。
        /// </summary>
        /// <param name="bankFilePath">バンクファイルパス。</param>
        /// <returns>存在する場合は true、存在しない場合は false。</returns>
        public bool Contains(string bankFilePath)
        {
            Ensure.Argument.NotNull(bankFilePath);
            return this.bankServices.ContainsKey(bankFilePath);
        }

        /// <summary>
        /// バンクサービスを開きます。
        /// </summary>
        /// <param name="filePath">バンクファイルパスを指定します。</param>
        /// <returns>コンポーネントサービスへの参照を返します。</returns>
        public BankServiceReference OpenItem(string filePath)
        {
            Ensure.Argument.NotNull(filePath);
            Ensure.Argument.StringNotEmpty(filePath);

            bool bankServiceCreated = false;
            BankService bankService = null;
            this.bankServices.TryGetValue(filePath, out bankService);

            if (bankService == null)
            {
                bankService = new BankService(
                    this.documentService, this.IntermediateOutputTraits);
                bankService.Open(new FileResource(filePath));

                this.bankServices.Add(bankService.BankDocument.Resource.Key, bankService);
                this.items.Add(bankService, new BankServiceItem(bankService));

                bankService.Closing += this.OnBankServiceClosing;
                bankServiceCreated = true;
            }

            BankServiceReference reference = new BankServiceReference(bankService);

            try
            {
                this.items[bankService].AddReference(reference);

                // 2013/12/6 aoyagi
                // BankServiceが構築された時にしかイベントを発生しないようにしました。
                // BankServiceの構築を外部から知るようにする為です。
                if (bankServiceCreated == true)
                {
                    OnAdded(new BankServiceEventArgs(bankService));
                }
            }
            catch
            {
                reference.Close();
                throw;
            }

            return reference;
        }

        /// <summary>
        /// 指定されたバンクファイルパスに関連するバンクサービスを閉じます。
        /// バンクサービスへの参照があったとしても全て閉じられます。
        /// </summary>
        /// <param name="bankFilePath">バンクファイルパス。</param>
        public void CloseItem(string bankFilePath)
        {
            if (null == bankFilePath) { throw new ArgumentNullException("bankService"); }

            if (!bankServices.ContainsKey(bankFilePath)) { return; }

            BankService bankService = this.bankServices[bankFilePath];

            bankService.Close();
        }

        /// <summary>
        /// 全てのバンクサービスを削除します。
        /// </summary>
        public void Clear()
        {
            while (0 < bankServices.Count)
            {
                this.CloseItem(bankServices.First().Key);
            }
        }

        protected virtual void OnAdded(BankServiceEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != Added)
            {
                Added(this, e);
            }
        }

        protected virtual void OnRemoved(BankServiceEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            e.BankService.Closing -= this.OnBankServiceClosing;

            if (null != Removed)
            {
                Removed(this, e);
            }
        }

        private void OnBankServiceClosing(object sender, EventArgs e)
        {
            Assertion.Argument.NotNull(sender);
            Assertion.Argument.NotNull(e);

            BankService bankService = sender as BankService;

            this.bankServices.Remove(bankService.BankDocument.Resource.Key);
            this.items.Remove(bankService);

            this.OnRemoved(new BankServiceEventArgs(bankService));
        }

        private class BankServiceItem
        {
            private HashSet<BankServiceReference> references = new HashSet<BankServiceReference>();

            public BankServiceItem(BankService target)
            {
                Ensure.Argument.NotNull(target);
                this.Target = target;
                target.Closed += this.OnBankServiceClosed;
            }

            public BankService Target { get; private set; }

            public void AddReference(BankServiceReference reference)
            {
                Ensure.Argument.NotNull(reference);

                this.references.Add(reference);
                reference.Closed += this.OnReferenceClosed;
            }

            public void RemoveReference(BankServiceReference reference)
            {
                Ensure.Argument.NotNull(reference);

                this.references.Remove(reference);
                reference.Closed -= this.OnReferenceClosed;
            }

            private void CloseAllReferences()
            {
                // 列挙ループ中に閉じる（コレクションを操作する）ので、配列化しておきます。
                foreach (BankServiceReference reference in this.references.ToArray())
                {
                    reference.Close();
                }

                this.references.Clear();
            }

            private void OnBankServiceClosed(object sender, EventArgs e)
            {
                Assertion.Argument.NotNull(sender);
                Assertion.Argument.NotNull(e);

                this.CloseAllReferences();
                this.Target.Closed -= this.OnBankServiceClosed;
            }

            /// <summary>
            /// コンポーネントサービスへの参照が閉じられると発生します。
            /// </summary>
            /// <param name="sender">イベントの送信元を指定します。</param>
            /// <param name="e">イベントデータを指定します。</param>
            private void OnReferenceClosed(object sender, EventArgs e)
            {
                this.RemoveReference(sender as BankServiceReference);

                if (this.Target.IsOpened && this.references.Count == 0)
                {
                    this.Target.Close();
                }
            }
        }
    }
}
