﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Collections;
using NintendoWare.SoundFoundation.Documents;

namespace NintendoWare.SoundFoundation.Projects
{
    public delegate void ComponentNameChangedEventHandler(object sender, ComponentNameChangedEventArgs e);

    public abstract partial class ComponentManager : IEventRoutable
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private DocumentReferenceDictionary<SoundDocument> _Documents =
                                                new DocumentReferenceDictionary<SoundDocument>();

        private ComponentDictionaryImpl _ComponentDictionary =
                                                new ComponentDictionaryImpl();

        private Component _rootComponent;
        private ComponentList _eventChildren = new ComponentList(null);
        private EventRouter _eventRouter;

        public ComponentManager()
        {
            _eventRouter = new EventRouter(this);

            BindRoutingEvent<ComponentChildrenChangedEventArgs>(typeof(ComponentChildrenChangedEvent),
                                                          OnBubbleChildrenChanged, null);
            BindRoutingEvent<ComponentNameChangedEventArgs>(typeof(ComponentNameChangedEvent),
                                                      OnBubbleNameChanged, null);
            BindRoutingEvent<RoutingEventArgs>
                (typeof(RequestUpdateSoundProjectItemBindingEvent),
                  OnBubbleRequestUpdateSoundProjectItemBindingEvent, null);
        }

        public event ComponentEventHandler ComponentsAdded;
        public event ComponentEventHandler ComponentsRemoved;
        public event ComponentEventHandler ComponentsResetted;
        public event ComponentNameChangedEventHandler ComponentNameChanged;

        public IReadOnlyDictionary<string, SoundDocument> Documents
        {
            get { return _Documents; }
        }

        public ComponentDictionary ComponentDictionary
        {
            get { return _ComponentDictionary; }
        }

        /// <summary>
        /// イベントツリー上の親を取得します。
        /// </summary>
        IEventRoutable IEventRoutable.Parent
        {
            get { return null; }
        }

        /// <summary>
        /// イベントツリー上の子コレクションを取得します。
        /// </summary>
        IReadOnlyEventRoutableCollection IEventRoutable.Children
        {
            get { return _eventChildren; }
        }

        /// <summary>
        /// このオブジェクトに関連付けられたルータを取得します。
        /// </summary>
        EventRouter IEventRoutable.Router
        {
            get { return _eventRouter; }
        }

        /// <summary>
        /// ルートコンポーネントを取得または設定します。
        /// </summary>
        protected Component RootComponent
        {
            get { return _rootComponent; }
            private set
            {
                if (value == _rootComponent) { return; }

                if (null != _rootComponent)
                {
                    _rootComponent.EventTreeParent = null;
                    _eventChildren.Remove(_rootComponent);
                }

                _rootComponent = value;

                if (null != value)
                {
                    _rootComponent.EventTreeParent = this;
                    _eventChildren.Add(_rootComponent);
                }
            }
        }

        /// <summary>
        /// 指定リソースがコンポーネントマネージャに含まれているかどうか調べます。
        /// </summary>
        /// <param name="resourceKey"></param>
        /// <returns>含まれている場合 true、含まれていない場合 false。</returns>
        public bool ContainsDocument(string resourceKey)
        {
            return _Documents.ContainsKey(resourceKey);
        }

        /// <summary>
        /// 指定ドキュメントと、そのドキュメントに含まれるコンポーネントを管理します。
        /// </summary>
        public void AddDocument(DocumentReference docRef)
        {
            DocumentReference clone = docRef.Clone();
            SoundDocument document = (SoundDocument)(clone.Document);

            _Documents.Add(clone);

            if (null == RootComponent && 0 < document.TopComponents.Length)
            {

                foreach (Component component in document.TopComponents)
                {
                    if (IsComponentTreeRoot(component) == true)
                    {
                        RootComponent = component;
                        AddComponents(document.TopComponents);
                        break;
                    }
                }
            }

            UpdateComponentBindingByAddDocument(document);
            OnAddDocumentReference(new DocumentReferenceEventArgs(clone));
        }

        /// <summary>
        ///指定ドキュメントと、そのドキュメントに含まれるコンポーネントを除外します。
        /// </summary>
        public void RemoveDocument(string resourceKey)
        {
            if (null == resourceKey) { throw new ArgumentNullException("resourceKey"); }
            if (!_Documents.ContainsKey(resourceKey)) { return; }

            DocumentReference target = _Documents[resourceKey];
            SoundDocument targetDocument = target.Document as SoundDocument;

            if (null != targetDocument)
            {
                foreach (Component topComponent in targetDocument.TopComponents)
                {
                    if (RootComponent == topComponent)
                    {
                        RootComponent = null;
                    }
                }

                if (RootComponent == null)
                {
                    RemoveComponents(targetDocument.TopComponents);
                }
                else
                {
                    UpdateComponentBindingByRemoveDocument(targetDocument);
                }
            }

            _Documents.Remove(resourceKey);

            OnRemoveDocumentReference(new DocumentReferenceEventArgs(target));
            target.Close();
        }

        /// <summary>
        /// 全てのドキュメントを除外します。
        /// </summary>
        public void ClearDocuments()
        {
            string[] keys = _Documents.Keys.ToArray();

            foreach (string key in keys)
            {
                RemoveDocument(key);
            }
        }

        /// <summary>
        /// 指定コンポーネントがルートコンポーネントかどうか調べます。
        /// </summary>
        /// <param name="component">対象コンポーネント。</param>
        /// <returns>ルートコンポーネントの場合は true、それ以外の場合は false。</returns>
        protected abstract bool IsComponentTreeRoot(Component component);

        /// <summary>
        /// ドキュメント参照が追加されると発生します。
        /// </summary>
        /// <param name="e">DocumentReference イベントデータ。</param>
        protected virtual void OnAddDocumentReference(DocumentReferenceEventArgs e)
        {
        }

        /// <summary>
        /// ドキュメント参照が削除されると発生します。
        /// </summary>
        /// <param name="e">DocumentReference イベントデータ。</param>
        protected virtual void OnRemoveDocumentReference(DocumentReferenceEventArgs e)
        {
        }

        /// <summary>
        /// コンポーネントが追加されると発生します。
        /// </summary>
        /// <param name="e">コンポーネントイベントデータ。</param>
        protected virtual void OnComponentsAdded(ComponentEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ComponentsAdded)
            {
                ComponentsAdded(this, e);
            }
        }

        /// <summary>
        /// コンポーネントが削除されると発生します。
        /// </summary>
        /// <param name="e">コンポーネントイベントデータ。</param>
        protected virtual void OnComponentsRemoved(ComponentEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ComponentsRemoved)
            {
                ComponentsRemoved(this, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void OnComponentsResetted(ComponentEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ComponentsResetted)
            {
                ComponentsResetted(this, e);
            }
        }

        /// <summary>
        /// コンポーネント名が変更されると発生します。
        /// </summary>
        /// <param name="e">コンポーネントイベントデータ。</param>
        protected virtual void OnComponentNameChanged(ComponentNameChangedEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ComponentNameChanged)
            {
                ComponentNameChanged(this, e);
            }
        }

        ///
        private void UpdateComponentBindingByAddDocument(SoundDocument document)
        {
            _eventRouter.DispatchEvent(
                new SoundDocumentBindingEventArgs(
                            document, SoundDocumentChangedAction.Added));
        }

        ///
        private void UpdateComponentBindingByRemoveDocument(SoundDocument document)
        {
            _eventRouter.DispatchEvent(
                new SoundDocumentBindingEventArgs(
                            document, SoundDocumentChangedAction.Removed));
        }

        private void UpdateComponentsDirty()
        {
            foreach (DocumentReference documentRef in _Documents.Values)
            {
                (documentRef.Document as SoundDocument).UpdateComponentsDirty();
            }
        }


        /// <summary>
        /// 指定コンポーネントとその子供をディクショナリに追加します。
        /// </summary>
        /// <param name="component">コンポーネント。</param>
        private void AddComponents(ICollection<Component> components)
        {
            List<Component> list = AddComponentsRecursive(components);

            if (0 < list.Count)
            {
                OnComponentsAdded(new ComponentEventArgs(list));
            }
        }

        private List<Component> AddComponentsRecursive(ICollection<Component> components)
        {
            List<Component> list = new List<Component>();

            if (components != null)
            {
                foreach (Component component in components)
                {
                    list.AddRange(AddComponent(component));
                }
            }

            return list;
        }

        private List<Component> AddComponent(Component component)
        {
            List<Component> list = new List<Component>();

            if (component != null)
            {
                _ComponentDictionary.Add(component.Name, component);
                list.AddRange(AddComponentsRecursive(component.Children));
                list.Add(component);
            }

            return list;
        }


        /// <summary>
        /// 指定コンポーネントとその子供をディクショナリから削除します。
        /// </summary>
        /// <param name="component">コンポーネント。</param>
        private void RemoveComponents(ICollection<Component> components)
        {
            List<Component> list = RemoveComponentsRecursive(components);

            if (0 < list.Count)
            {
                OnComponentsRemoved(new ComponentEventArgs(list));
            }
        }

        private List<Component> RemoveComponentsRecursive(ICollection<Component> components)
        {
            List<Component> list = new List<Component>();

            if (components != null)
            {
                foreach (Component component in components)
                {
                    list.AddRange(RemoveComponent(component));
                }
            }

            return list;
        }

        private List<Component> RemoveComponent(Component component)
        {
            List<Component> list = new List<Component>();

            if (component != null && _ComponentDictionary.Contains(component.Name) == true)
            {
                _ComponentDictionary.Remove(component);
                list.AddRange(RemoveComponentsRecursive(component.Children));
                list.Add(component);
            }

            return list;
        }

        /// <summary>
        /// 指定コンポーネントのルーティングイベントにバインドします。
        /// </summary>
        /// <typeparam name="TEventArgs">イベントデータの型。</typeparam>
        /// <param name="component">コンポーネント。</param>
        /// <param name="eventType">イベントの種類。</param>
        /// <param name="bubblingEventHandler">バブリングイベントハンドラ。</param>
        /// <param name="eventHandler">イベントハンドラ。</param>
        private void BindRoutingEvent<TEventArgs>(Type eventType,
                                   RoutingEventBinding<TEventArgs>.SpecificEventHandler bubblingEventHandler,
                                   RoutingEventBinding<TEventArgs>.SpecificEventHandler eventHandler)
            where TEventArgs : RoutingEventArgs
        {
            if (null == eventType) { throw new ArgumentNullException("eventType"); }

            _eventRouter.Bindings.Add(
                new RoutingEventBinding<TEventArgs>(eventType, bubblingEventHandler, eventHandler));
        }

        /// <summary>
        /// ComponentChildrenChanged バブリングイベントが発生すると実行されます。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">ComponentChildrenChanged イベントデータ。</param>
        private void OnBubbleChildrenChanged(object sender, ComponentChildrenChangedEventArgs e)
        {
            if (e.Components != null &&
                e.Components.Length > 0)
            {
                switch (e.Action)
                {
                    case ComponentChildrenChangedAction.Add:
                        AddComponents(e.Components);
                        break;

                    case ComponentChildrenChangedAction.Remove:
                        RemoveComponents(e.Components);
                        break;

                    case ComponentChildrenChangedAction.Move:
                        break;
                }
            }
            else
            {
                switch (e.Action)
                {
                    case ComponentChildrenChangedAction.Reset:
                        OnComponentsResetted(new ComponentEventArgs());
                        break;
                }
            }

            _eventRouter.DispatchEvent
                (new ComponentManagerEventArgs(new UpdateSoundProjectItemBindingEvent(), this));

            UpdateComponentsDirty();
        }

        ///--------------------------------
        /// <summary>
        /// ComponentNameChanged バブリングイベントが発生すると実行されます。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">ComponentNameChanged イベントデータ。</param>
        private void OnBubbleNameChanged(object sender, ComponentNameChangedEventArgs e)
        {
            _eventRouter.DispatchEvent
                (new ComponentManagerRenamedComponentEventArgs(new UpdateRenamedSoundProjectItemBindingEvent(), this, e.OldName, e.NewName));

            //
            if (_ComponentDictionary.Contains(e.OldName) == true)
            {
                _ComponentDictionary.Rename(e.OldName, e.NewName);
            }

            UpdateComponentsDirty();

            OnComponentNameChanged(e);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnBubbleRequestUpdateSoundProjectItemBindingEvent(object sender, RoutingEventArgs e)
        {
            _eventRouter.DispatchEvent
                (new ComponentManagerEventArgs(new UpdateSoundProjectItemBindingEvent(), this));
        }
    }
}
