﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;

using NintendoWare.SoundFoundation.Documents;

namespace NintendoWare.SoundFoundation.Projects
{
    public class BankComponentManager : ComponentManager
    {
        /// <summary>
        /// 開かれているバンクドキュメントへの参照。
        /// </summary>
        private DocumentReference _bankDocumentReference = null;

        public BankDocument BankDocument
        {
            get
            {
                if (null == _bankDocumentReference) { return null; }
                Debug.Assert(Documents.Count() == 1, "BankDocument : Only one is permitted.");
                return _bankDocumentReference.Document as BankDocument;
            }
        }

        public Bank Bank
        {
            get
            {
                return BankDocument.Bank;
            }
        }

        /// <summary>
        /// バンクに含まれるインストルメントを列挙します。
        /// </summary>
        public IEnumerable<Instrument> Instruments
        {
            get
            {
                return from Instrument instrument in Bank.Children
                       select instrument;
            }
        }

        /// <summary>
        /// バンクに含まれる VelocityRegion を列挙します。
        /// </summary>
        public IEnumerable<VelocityRegion> VelocityRegions
        {
            get
            {
                return from Instrument instrument in Bank.Children
                       from KeyRegion keyRegion in instrument.Children
                       from VelocityRegion velocityRegion in keyRegion.Children
                       select velocityRegion;
            }
        }

        public event EventHandler BankDocumentChanged;

        /// <summary>
        /// 指定コンポーネントがルートコンポーネントかどうか調べます。
        /// </summary>
        /// <param name="component">対象コンポーネント。</param>
        /// <returns>ルートコンポーネントの場合は true、それ以外の場合は false。</returns>
        protected override bool IsComponentTreeRoot(Component component)
        {
            return (component is Bank);
        }

        protected override void OnAddDocumentReference(DocumentReferenceEventArgs e)
        {
            base.OnAddDocumentReference(e);

            if (null == e.DocumentReference) { return; }
            if (!(e.DocumentReference.Document is BankDocument)) { return; }

            if (null != _bankDocumentReference)
            {
                throw new Exception("bank document is already existing.");  // ★仮の例外クラス
            }

            _bankDocumentReference = e.DocumentReference;

            if (null != Bank)
            {
                OnBankDocumentChanged(new EventArgs());
            }
        }

        protected override void OnRemoveDocumentReference(DocumentReferenceEventArgs e)
        {
            base.OnRemoveDocumentReference(e);

            if (null == e.DocumentReference) { return; }
            if (e.DocumentReference != _bankDocumentReference) { return; }

            _bankDocumentReference = null;

            OnBankDocumentChanged(new EventArgs());
        }

        protected virtual void OnBankDocumentChanged(EventArgs e)
        {
            if (null != BankDocumentChanged)
            {
                BankDocumentChanged(this, e);
            }
        }
    }
}
