﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Projects;

namespace NintendoWare.SoundFoundation.Parameters
{
    /// <summary>
    /// パラメータコレクションを削除するオペレーションです。
    /// </summary>
    public class RemoveParameterOperation : OperationImpl
    {
        // 削除するパラメータ
        private IList<ComponentReference> removeParameterList = new List<ComponentReference>();

        // 削除先のパラメータコレクション
        private IList<ComponentReference> target = null;

        // 削除パラメータのインデックスディクショナリ
        private Dictionary<ComponentReference, int> indexes =
            new Dictionary<ComponentReference, int>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="list">パラメータコレクション。</param>
        /// <param name="provider">削除するパラメータ。</param>
        public RemoveParameterOperation(
                                        IList<ComponentReference> list,
                                        ComponentReference provider
                                        ) : this(list, new ComponentReference[] { provider })
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="list">パラメータコレクション。</param>
        /// <param name="providers">削除するパラメータの IEnumerable。</param>
        public RemoveParameterOperation(
                                        IList<ComponentReference> list,
                                        IEnumerable<ComponentReference> providers
                                        )
        {
            if (null == list) { throw new ArgumentNullException("list"); }
            if (null == providers) { throw new ArgumentNullException("providers"); }

            this.target = list;

            foreach (ComponentReference provider in providers)
            {
                if (null == provider) { throw new ArgumentException("invalid argument"); }
                if (this.indexes.ContainsKey(provider)) { continue; }
                this.removeParameterList.Add(provider);
                this.indexes.Add(provider, -1);
            }
        }

        /// <summary>
        /// 削除するパラメータの IEnumerable を取得します。
        /// </summary>
        public IEnumerable<ComponentReference> RemoveParameterList
        {
            get { return this.removeParameterList; }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool ExecuteInternal()
        {
            try
            {
                foreach (ComponentReference provider in this.removeParameterList)
                {
                    int index = this.target.IndexOf(provider);
                    if (index < 0) { throw new Exception("expected error"); }

                    this.indexes[provider] = index;
                    this.target.RemoveAt(index);
                }
            }
            catch
            {
                RollbackInternal();
            }

            return true;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool RollbackInternal()
        {
            for (int i = this.removeParameterList.Count - 1; i >= 0; i--)
            {

                ComponentReference provider = this.removeParameterList[i];
                if (!this.indexes.ContainsKey(provider)) { continue; }

                int index = this.indexes[provider];
                if (index < 0 || this.target.Count < index) { continue; }

                this.target.Insert(this.indexes[provider], provider);

            }

            this.indexes.Clear();

            return true;
        }
    }
}
