﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace NintendoWare.SoundFoundation.Operations
{
    /// <summary>
    /// トランザクションのスタックを実装します。
    /// </summary>
    public class TransactionStack
    {
        private LinkedList<Transaction> _undoStack = new LinkedList<Transaction>();   // Undoスタック
        private LinkedList<Transaction> _redoStack = new LinkedList<Transaction>();   // Redoスタック
        private Transaction _savedItem = null;                            // 保存位置のトランザクション

        /// <summary>
        /// トランザクションスタックの IEnumerator を返します。
        /// </summary>
        public IEnumerable<Transaction> AllItems
        {
            get
            {
                foreach (Transaction transaction in _undoStack)
                {
                    yield return transaction;
                }

                foreach (Transaction transaction in _redoStack)
                {
                    yield return transaction;
                }
            }
        }

        /// <summary>
        /// Undo 対象トランザクションの IEnumerator を返します。
        /// </summary>
        public IEnumerable<Transaction> UndoItems
        {
            get { return _undoStack; }
        }

        /// <summary>
        /// Redo 対象トランザクションの IEnumerator を返します。
        /// </summary>
        public IEnumerable<Transaction> RedoItems
        {
            get { return _redoStack; }
        }

        /// <summary>
        /// ドキュメントが編集されているかどうかを調べます。
        /// </summary>
        public bool IsDirty
        {
            get
            {
                if (0 == _undoStack.Count) { return (null != _savedItem); }
                return (_undoStack.First.Value != _savedItem);
            }
        }

        /// <summary>
        /// 現在位置のトランザクションを取得します。
        /// </summary>
        public Transaction CurrentItem
        {
            get { return UndoItem; }
        }

        /// <summary>
        /// Undo対象のトランザクションを取得します。
        /// </summary>
        public Transaction UndoItem
        {
            get { return (0 == _undoStack.Count) ? null : _undoStack.First.Value; }
        }

        /// <summary>
        /// Redo対象のトランザクションを取得します。
        /// </summary>
        public Transaction RedoItem
        {
            get { return (0 == _redoStack.Count) ? null : _redoStack.Last.Value; }
        }

        /// <summary>
        /// 保存位置のトランザクションを取得します。
        /// </summary>
        public Transaction SavedItem
        {
            get { return _savedItem; }
        }

        /// <summary>
        /// トランザクションスタックが変更されたときに発生します。
        /// </summary>
        public event EventHandler ItemsChanged;

        /// <summary>
        /// Dirty 状態が変更されたときに発生します。
        /// </summary>
        public event EventHandler DirtyChanged;

        /// <summary>
        /// 現在位置のトランザクションが変更されたときに発生します。
        /// </summary>
        public event EventHandler CurrentItemChanged;

        /// <summary>
        /// 保存位置のトランザクションが変更されたときに発生します。
        /// </summary>
        public event EventHandler SavedItemChanged;

        /// <summary>
        /// スタックにトランザクションを追加します。
        /// </summary>
        /// <param name="transaction">追加するトランザクション。</param>
        public void Push(Transaction transaction)
        {
            if (null == transaction) { throw new ArgumentNullException("transaction"); }

            using (new DirtyChecker(this))
            {

                _redoStack.Clear();
                _undoStack.AddFirst(transaction);

                try
                {

                    transaction.Stacks.Add(this);

                }
                catch (Exception exception)
                {

                    Debug.Assert(_undoStack.First.Value == transaction, "Internal Error : Failed to push transaction?");
                    _undoStack.RemoveFirst();

                    throw exception;

                }

                //OnItemsChanged( EventArgs.Empty );
                OnItemsChanged(new TransactionStackEventArgs(transaction));
            }
        }

        /// <summary>
        /// スタックをクリアします。
        /// </summary>
        public void Clear()
        {
            using (new DirtyChecker(this))
            {

                if (0 < _undoStack.Count || 0 < _redoStack.Count)
                {

                    ClearTransactionStack(_undoStack);
                    ClearTransactionStack(_redoStack);

                    //OnItemsChanged( EventArgs.Empty );
                    //OnCurrentItemChanged( EventArgs.Empty );
                    OnItemsChanged(new TransactionStackEventArgs(null));
                    OnCurrentItemChanged(new TransactionStackEventArgs(null));
                }

                if (null != _savedItem)
                {
                    _savedItem = null;
                    OnSavedItemChanged(EventArgs.Empty);
                }

                //OnItemsChanged( EventArgs.Empty );
                OnItemsChanged(new TransactionStackEventArgs(null));
            }
        }

        /// <summary>
        /// 保存位置を現在位置にリセットします。
        /// </summary>
        public void ResetSavedItem()
        {
            if (_savedItem == UndoItem) { return; }

            using (new DirtyChecker(this))
            {

                _savedItem = UndoItem;
                OnSavedItemChanged(EventArgs.Empty);

            }
        }

        public bool Undo()
        {
            if (null == UndoItem) { return false; }

            if (!UndoItem.Rollback()) { return false; }

            using (new DirtyChecker(this))
            {

                _redoStack.AddLast(_undoStack.First.Value);
                _undoStack.RemoveFirst();

                //OnCurrentItemChanged( EventArgs.Empty );
                OnCurrentItemChanged(new TransactionStackEventArgs(this._redoStack.Last.Value));
            }

            return true;
        }

        public bool Redo()
        {
            if (null == RedoItem) { return false; }

            if (!RedoItem.Execute()) { return false; }

            using (new DirtyChecker(this))
            {

                _undoStack.AddFirst(_redoStack.Last.Value);
                _redoStack.RemoveLast();

                //OnCurrentItemChanged( EventArgs.Empty );
                OnCurrentItemChanged(new TransactionStackEventArgs(this._undoStack.First.Value));
            }

            return true;
        }

        public bool CanUndo()
        {
            if (null == UndoItem) { return false; }

            return UndoItem.CanRollback();
        }

        public bool CanRedo()
        {
            if (null == RedoItem) { return false; }

            return RedoItem.CanExecute();
        }

        /// <summary>
        /// トランザクションスタックが変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnItemsChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ItemsChanged)
            {
                ItemsChanged(this, e);
            }
        }

        /// <summary>
        /// Dirty 状態が変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnDirtyChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != DirtyChanged)
            {
                DirtyChanged(this, e);
            }
        }

        /// <summary>
        /// 現在位置のトランザクションが変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnCurrentItemChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != CurrentItemChanged)
            {
                CurrentItemChanged(this, e);
            }
        }

        /// <summary>
        /// 保存位置のトランザクションが変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnSavedItemChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != SavedItemChanged)
            {
                SavedItemChanged(this, e);
            }
        }

        private void ClearTransactionStack(LinkedList<Transaction> stack)
        {
            Debug.Assert(null != stack, "invalid argument.");

            try
            {
                foreach (Transaction transation in stack)
                {
                    transation.Dispose();
                }
            }
            finally
            {
                stack.Clear();
            }
        }

        private class DirtyChecker : IDisposable
        {
            private TransactionStack _owner;
            private bool _isDirtyOld = false;
            private bool _disposed = false;

            public DirtyChecker(TransactionStack owner)
            {
                Debug.Assert(null != owner, "unexpected error");

                _owner = owner;
                _isDirtyOld = owner.IsDirty;
            }

            ~DirtyChecker()
            {
                Dispose();
            }

            void IDisposable.Dispose()
            {
                Dispose();
            }

            private void Dispose()
            {
                if (_disposed) { return; }

                if (_isDirtyOld != _owner.IsDirty)
                {
                    _owner.OnDirtyChanged(EventArgs.Empty);
                }

                _disposed = true;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class TransactionStackEventArgs : EventArgs
    {
        private Transaction transaction = null;

        /// <summary>
        ///
        /// </summary>
        public TransactionStackEventArgs(Transaction transaction)
        {
            // transaction == nullも許容します。
            this.transaction = transaction;
        }

        /// <summary>
        ///
        /// </summary>
        public Transaction Transaction
        {
            get
            {
                return this.transaction;
            }
        }
    }
}
