﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.Wave
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Resources;

    public class WaveFileAiffReader : WaveFileReader
    {
        private FileStream strm;
        private string filePath;
        private BinaryReader reader;
        private long dataBegin;
        private long dataEnd;
        private long curSample;
        private UInt32 numSampleFrames;
        private Int16 numChannels;
        private Int16 sampleSize;

        public override void Close()
        {
            strm.Close();
            filePath = null;
        }

        public override byte[] Read()
        {
            long readSamples = numSampleFrames * numChannels - curSample;
            int sampleByte = (sampleSize + 7) / 8;
            long readSize = readSamples * sampleByte;

            byte[] buffer = reader.ReadBytes(Convert.ToInt32(readSize));

            curSample += readSamples;

            return buffer;
        }

        protected override void DisposeInternal()
        {
            if (strm != null)
            {
                strm.Close();
            }
            filePath = null;
        }

        protected override WaveFile OpenInternal(string filePath)
        {
            try
            {
                return OpenAiff(filePath);
            }
            catch (Exception e)
            {
                if (e is WaveFileReaderException)
                {
                    throw e;
                }
                else
                {
                    this.filePath = null;

                    string message =
                        MessageResource.Message_InvalidWaveFile + " \"" + filePath + "\"";

                    throw new WaveFileReaderException(message);
                }
            }
        }

        protected override WaveDataStream CreateDataStream()
        {
            if (this.strm == null)
            {
                throw new InvalidOperationException();
            }

            var fileStream = File.OpenRead(this.filePath);
            fileStream.Position = this.dataBegin;

            return new WaveDataStream(fileStream, this.dataEnd - this.dataBegin);
        }

        private WaveFileAiff OpenAiff(string filePath)
        {
            WaveFileAiff waveFile = new WaveFileAiff();

            strm = File.OpenRead(filePath);
            reader = BigEndianBinaryReader.Create(strm, new System.Text.ASCIIEncoding());

            string fileChunk = new string(reader.ReadChars(4));
            if (fileChunk != "FORM")
            {
                throw new WaveFileReaderException("Invalid format");
            }

            Int32 fileSize = reader.ReadInt32();
            string format = new string(reader.ReadChars(4));
            if (format != "AIFF")
            {
                throw new WaveFileReaderException("Invalid format");
            }

            while (strm.Position < strm.Length)
            {
                string chunkType = new string(reader.ReadChars(4));
                UInt32 chunkSize = reader.ReadUInt32();
                long chunkEnd = strm.Position + chunkSize;
                long nextChunkPos = ((chunkEnd + 0x01) & ~0x01); // 2バイトアライン

                if (chunkType == "COMM")
                {
                    WaveFileAiff.AiffCommonChunk commonChunk = waveFile.CommonChunk;
                    commonChunk.Read(reader);

                    numSampleFrames = commonChunk.NumSampleFrames;
                    numChannels = commonChunk.NumChannels;
                    sampleSize = commonChunk.SampleSize;
                }
                else if (chunkType == "SSND")
                {
                    waveFile.SoundDataChunk.Read(reader);

                    dataBegin = strm.Position;
                    dataEnd = chunkEnd;
                }
                else if (chunkType == "INST")
                {
                    waveFile.InstChunk.Read(reader);
                }
                else if (chunkType == "MARK")
                {
                    waveFile.MarkerChunk.Read(reader);
                }
                else
                {
                    // unknown chunk
                }

                strm.Position = nextChunkPos;
            }

            curSample = 0;
            strm.Position = dataBegin;

            this.filePath = filePath;
            return waveFile;
        }
    }
}
