﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate
{
    using System;
    using System.Xml;
    using System.Xml.Serialization;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.ToolDevelopmentKit;

    [Serializable]
    [XmlRoot(
        "NintendoWareFile",
        Namespace = XmlNamespaces.Base,
        IsNullable = false)]
    public abstract class XmlCommonFileRoot<TBody>
        where TBody : new()
    {
        private const string VersionAttribute = "Version";
        private const string PlatformAttribute = "Platform";
        private const string HeadAttribute = "Head";
        private const string BodyAttribute = "Body";

        protected XmlCommonFileRoot()
        {
            this.Body = new TBody();
        }

        [XmlAttribute(VersionAttribute)]
        public string Version { get; set; }

        [XmlAttribute(PlatformAttribute)]
        public string Platform { get; set; }

        [XmlElement(HeadAttribute)]
        public XmlCommonFileHeader Header { get; set; }

        [XmlElement(BodyAttribute)]
        public TBody Body { get; set; }

        /// <summary>
        /// XMLデータを検証します。
        /// </summary>
        /// <param name="reader">対象となる XmlReader を指定します。</param>
        /// <param name="rootElementName">ルート要素名を指定します。</param>
        /// <param name="platform">プラットフォーム名を指定します。</param>
        /// <param name="rootComponentElementName">ルートコンポーネント要素名を指定します。</param>
        public static void Validate(
            XmlReader reader,
            string rootElementName,
            string platform,
            string version,
            string rootComponentElementName)
        {
            Ensure.Argument.NotNull(reader);
            Ensure.Argument.StringNotEmpty(rootElementName);
            Ensure.Argument.NotNull(platform);
            Ensure.Argument.StringNotEmpty(rootComponentElementName);

            while (reader.NodeType != XmlNodeType.Element)
            {
                if (!reader.Read())
                {
                    throw new NotSupportedException();
                }
            }

            XmlReader bankReader = reader.ReadSubtree();
            bankReader.Read();

            if (bankReader.LocalName != rootElementName)
            {
                throw new NotSupportedException();
            }

            if (platform.Length > 0 &&
                bankReader.GetAttribute(PlatformAttribute) != platform)
            {
                throw new NotSupportedException();
            }

            Version v1 = new Version(version);
            Version v2 = new Version(bankReader.GetAttribute(VersionAttribute));
            if (v1 == null || v2 == null || v1 < v2)
            {
                throw new FileNotSupportedVersionException(version);
            }

            if (!bankReader.ReadToDescendant(HeadAttribute))
            {
                throw new NotSupportedException();
            }

            if (!bankReader.ReadToNextSibling(BodyAttribute))
            {
                throw new NotSupportedException();
            }

            XmlReader bodyReader = reader.ReadSubtree();
            bodyReader.Read();

            if (!bankReader.ReadToDescendant(rootComponentElementName))
            {
                throw new NotSupportedException();
            }
        }

        public static string GetFileVersion(XmlReader reader)
        {
            Ensure.Argument.NotNull(reader);

            while (reader.NodeType != XmlNodeType.Element)
            {
                if (!reader.Read())
                {
                    throw new NotSupportedException();
                }
            }

            XmlReader bankReader = reader.ReadSubtree();
            bankReader.Read();

            return bankReader.GetAttribute(VersionAttribute);
        }
    }
}
