﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate
{
    using System;
    using System.Collections.Generic;
    using Core.Parameters;
    using Projects;
    using ToolDevelopmentKit;

    public class ItemParamTranslator
    {
        public delegate void Xml2ModelHandler(XmlParameter paramXml, IParameterProvider dest);
        public delegate void Model2XmlHandler(string paramName, IParameterValue value, Type xmlParamType, IXmlParameterProvider dest);

        private Dictionary<string, ConversionHandler> handlerDictionary = new Dictionary<string, ConversionHandler>();

        public void AddHandler(string paramName, Type xmlParamType)
        {
            this.AddHandler(paramName, xmlParamType, null, null);
        }

        public void AddHandler(
            string paramName, Type xmlParamType, Xml2ModelHandler model2Xml, Model2XmlHandler xml2Model)
        {
            Ensure.Argument.NotNull(paramName);
            Ensure.Argument.NotNull(xmlParamType);

            this.handlerDictionary.Add(
                paramName,
                new ConversionHandler()
                {
                    XmlParameterType = xmlParamType,
                    Xml2Model = model2Xml,
                    Model2Xml = xml2Model,
                });
        }

        public void Xml2Model(XmlParameter paramXml, IParameterProvider dest)
        {
            Ensure.Argument.NotNull(paramXml);
            Ensure.Argument.NotNull(dest);

            ConversionHandler handler = null;
            this.handlerDictionary.TryGetValue(paramXml.Name, out handler);

            if (handler == null) { return; }

            if (handler.Xml2Model != null)
            {
                handler.Xml2Model(paramXml, dest);
            }
            else
            {
                this.Xml2ModelInternal(paramXml, dest);
            }
        }

        public void Model2Xml(string paramName, IParameterValue value, IXmlParameterProvider dest)
        {
            Ensure.Argument.NotNull(paramName);
            Ensure.Argument.NotNull(value);
            Ensure.Argument.NotNull(dest);

            ConversionHandler handler = null;
            this.handlerDictionary.TryGetValue(paramName, out handler);

            if (handler == null) { return; }

            if (dest.Parameters == null)
            {
                dest.Parameters = new List<XmlParameter>();
            }

            if (handler.Model2Xml != null)
            {
                handler.Model2Xml(paramName, value, handler.XmlParameterType, dest);
            }
            else
            {
                this.Model2XmlInternal(paramName, value, handler.XmlParameterType, dest);
            }
        }

        protected void SetParameterValue(IParameterProvider parameterTarget, string paramName, object paramValue)
        {
            Assertion.Argument.NotNull(parameterTarget);
            Assertion.Argument.NotNull(paramName);

            if (parameterTarget.Parameters.ContainsKey(paramName))
            {
                parameterTarget.Parameters[paramName].Value = paramValue;
            }
            else
            {
                parameterTarget.Parameters.AddValue(
                    paramName, new ObjectParameterValue<object>(paramValue));
            }
        }

        private void Xml2ModelInternal(XmlParameter paramXml, IParameterProvider dest)
        {
            Assertion.Argument.NotNull(paramXml);
            Assertion.Argument.NotNull(dest);

            Ensure.Argument.True(!(paramXml.XmlValue is IXmlParameterProvider));

            this.SetParameterValue(dest, paramXml.Name, paramXml.XmlValue);
        }

        private void Model2XmlInternal(string paramName, IParameterValue value, Type xmlParamType, IXmlParameterProvider dest)
        {
            Assertion.Argument.NotNull(paramName);
            Assertion.Argument.NotNull(value);
            Assertion.Argument.NotNull(xmlParamType);
            Assertion.Argument.NotNull(dest);

            Ensure.Argument.True(!(value.Value is IParameterProvider));

            XmlParameter xmlParam = Activator.CreateInstance(xmlParamType) as XmlParameter;
            xmlParam.Name = paramName;
            xmlParam.XmlValue = value.ToString();

            if (dest.Parameters == null)
            {
                dest.Parameters = new List<XmlParameter>();
            }

            dest.Parameters.Add(xmlParam);
        }

        public class ConversionHandler
        {
            public Type XmlParameterType { get; set; }

            public Xml2ModelHandler Xml2Model { get; set; }
            public Model2XmlHandler Model2Xml { get; set; }
        }
    }
}
