﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Documents
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// ドキュメントを管理します。
    /// </summary>
    public class DocumentService : IDocumentService
    {
        private DocumentManager _DocumentManager = null;
        private IDocumentServiceTraits _DocumentServiceTraits = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DocumentService(IDocumentServiceTraits traits)
        {
            if (null == traits) { throw new ArgumentNullException("traits"); }

            _DocumentManager = new DocumentManager();
            _DocumentManager.DocumentAdded += OnDocumentAdded;
            _DocumentManager.DocumentRemoved += OnDocumentRemoved;
            _DocumentServiceTraits = traits;
        }

        /// <summary>
        /// ドキュメントを開こうとするときに発生します。
        /// </summary>
        public event EventHandler<DocumentOpeningEventArgs> DocumentOpening;

        /// <summary>
        /// ドキュメントが開かれると発生します。
        /// </summary>
        public event EventHandler<DocumentEventArgs> DocumentOpened;

        /// <summary>
        /// ドキュメントが DocumentManager に追加されると発生します。
        /// </summary>
        public event EventHandler DocumentAdded;

        /// <summary>
        /// ドキュメントが DocumentManager から削除されると発生します。
        /// </summary>
        public event EventHandler DocumentRemoved;

        /// <summary>
        /// ドキュメントを保存する前に発生します。
        /// </summary>
        public event DocumentEventHandler DocumentSaving;

        /// <summary>
        /// ドキュメントが保存されると発生します。
        /// </summary>
        public event DocumentEventHandler DocumentSaved;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IDocumentServiceTraits Traits
        {
            get { return _DocumentServiceTraits; }
        }

        /// <summary>
        /// 開かれているドキュメントの一覧を取得します。
        /// </summary>
        public Document[] Documents
        {
            get { return _DocumentManager.Documents; }
        }

        /// <summary>
        /// 指定した種類のドキュメントを作成します。
        /// </summary>
        /// <param name="typeName">作成するドキュメントタイプを指定します。</param>
        /// <returns>作成したドキュメントへの参照を返します。</returns>
        public DocumentReference CreateDocument(string typeName)
        {
            DocumentReference docRef = null;

            Document document = CreateDocumentByTypeName(typeName);
            if (document != null)
            {
                document.Resource = new NullResource();
                docRef = _DocumentManager.Add(document);
            }

            return docRef;
        }

        /// <summary>
        /// 指定したリソースのドキュメントを開きます。
        /// </summary>
        /// <param name="resouce">リソースを指定します。</param>
        /// <returns>開いたドキュメントへの参照を返します。</returns>
        public DocumentReference OpenDocument(IStreamResource resource)
        {
            Ensure.Argument.NotNull(resource);

            IDocumentReader reader = null;
            Document document = FindDocumentByResource(resource);

            if (document == null)
            {
                this.OnDocumentOpening(new DocumentOpeningEventArgs(resource));

                // HACK : ★本当は、FileResource による IStreamResource.Open() の実装でやるべき
                if (resource is FileResource)
                {
                    if (!File.Exists(resource.Key))
                    {
                        throw new FileNotFoundException(string.Empty, resource.Key);
                    }
                }

                if ((reader = _DocumentServiceTraits.GetDocumentReader(resource)) == null)
                {
                    return null;
                }

                document = CreateDocumentByTypeName(reader.DocumentTypeName);
                document.Resource = resource;

                reader.Read(resource, document);
                document.ResetDirty();

                // HACK : ★IStreamResource.ResetLastUpdateDateTime() を作るべき？
                if (resource is FileResource)
                {
                    (resource as FileResource).ResetLastUpdateDateTime();
                }
            }

            return _DocumentManager.Add(document);
        }

        public void ReloadDocument(Document document)
        {
            if (null == document) { throw new ArgumentNullException("document"); }

            IDocumentReader reader = _DocumentServiceTraits.GetDocumentReader(document.Resource);
            if (null == reader) { throw new FileFormatException("unsupported file format"); }

            reader.Read(document.Resource, document);
            document.ResetDirty();

            if (document.Resource is FileResource)
            {
                (document.Resource as FileResource).ResetLastUpdateDateTime();
            }
        }

        /// <summary>
        /// 指定したドキュメントを保存します。
        /// </summary>
        /// <param name="document">ドキュメントを指定します。</param>
        public void SaveDocument(Document document)
        {
            if (document != null)
            {
                IDocumentWriter writer;
                writer = _DocumentServiceTraits.GetDocumentWriter(document.TypeName);

                if (writer != null)
                {
                    this.OnDocumentSaving(this, new DocumentEventArgs(document));

                    writer.Write(document.Resource, document, this.Documents);
                    document.ResetDirty();

                    if (document.Resource is FileResource)
                    {
                        (document.Resource as FileResource).ResetLastUpdateDateTime();
                    }

                    this.OnDocumentSaved(this, new DocumentEventArgs(document));
                }
            }
        }

        /// <summary>
        /// 指定したドキュメントのクローンを作成します。
        /// </summary>
        /// <param name="document">ドキュメントを指定します。</param>
        /// <returns>ドキュメントのクローンへの参照を返します。</returns>
        public DocumentReference CloneDocument(Document document)
        {
            //未実装
            return null;
        }

        /// <summary>
        /// ドキュメントを指定した種類のドキュメントに変換します。
        /// </summary>
        /// <param name="inputDocument">変換元のドキュメントを指定します。</param>
        /// <param name="outputDocumentTypeName">変換後のドキュメントタイプを指定します。</param>
        /// <returns>変換されたドキュメントへの参照を返します。</returns>
        public DocumentReference ConvertDocument(Document inputDocument,
                                                 string outputDocumentTypeName)
        {
            IDocumentServiceTraits traits = _DocumentServiceTraits;
            IDocumentFactory factory;
            IDocumentConverter docConv;

            factory = traits.GetDocumentFactory(outputDocumentTypeName);
            docConv = traits.GetDocumentConverter(inputDocument.TypeName, outputDocumentTypeName);

            Document outputDocument = factory.CreateDocument();
            docConv.Convert(inputDocument, outputDocument);

            return _DocumentManager.Add(outputDocument);
        }

        /// <summary>
        /// 指定したリソースのドキュメントタイプを調べます。
        /// </summary>
        /// <param name="resource">リソースを指定します。</param>
        /// <returns>ドキュメントタイプを返します。</returns>
        public string GetDocumentTypeName(IStreamResource resource)
        {
            Document document = this.FindDocumentByResource(resource);
            if (document != null)
            {
                return document.TypeName;
            }

            IDocumentReader reader = this._DocumentServiceTraits.GetDocumentReader(resource);
            if (reader == null)
            {
                return null;
            }

            return reader.DocumentTypeName;
        }

        /// <summary>
        /// 指定したドキュメントタイプから変換可能なドキュメントタイプを調べます。
        /// </summary>
        /// <param name="typeName">ドキュメントタイプを指定します。</param>
        /// <returns>変換可能なドキュメントタイプの配列を返します。</returns>
        public string[] FindConvertibleDocumentTypeNames(string typeName)
        {
            List<string> typeNameList = new List<string>();
            IDocumentConverter[] converterArray;

            converterArray = _DocumentServiceTraits.FindDocumentConverter(typeName);
            foreach (IDocumentConverter converter in converterArray)
            {
                typeNameList.Add(converter.OutputDocumentTypeName);
            }

            return typeNameList.ToArray();
        }

        /// <summary>
        /// 指定したドキュメントタイプ間の変換が可能かどうかを調べます。
        /// </summary>
        /// <param name="inputDocumentTypeName">変換元のドキュメントタイプを指定します。</param>
        /// <param name="outputDocumentTypeName">変換後のドキュメントタイプを指定します。</param>
        /// <returns>変換可能な場合は true、不可能な場合は false を返します。</returns>
        public bool CanConvertDocument(string inputDocumentTypeName, string outputDocumentTypeName)
        {
            IDocumentConverter converter = null;

            converter = _DocumentServiceTraits.GetDocumentConverter(inputDocumentTypeName,
                                                                    outputDocumentTypeName);

            return (converter != null);
        }

        /// <summary>
        /// リソースキーをもとにドキュメントが存在するか調べます。
        /// </summary>
        /// <param name="resourceKey">リソースキーを指定します。</param>
        public bool IsDocumentExists(string resourceKey)
        {
            return this._DocumentManager.IsDocumentExists(resourceKey);
        }


        /// <summary>
        /// ドキュメントを開こうとするときに呼び出されます。
        /// </summary>
        /// <param name="e">イベントデータを指定します。</param>
        protected virtual void OnDocumentOpening(DocumentOpeningEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            this.DocumentOpening?.Invoke(this, e);
        }

        /// <summary>
        /// ドキュメントが開かれると呼び出されます。
        /// </summary>
        /// <param name="e">イベントデータを指定します。</param>
        protected virtual void OnDocumentOpened(DocumentEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.DocumentOpened != null)
            {
                this.DocumentOpened(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnDocumentAdded(object sender, EventArgs e)
        {
            if (DocumentAdded != null)
            {
                DocumentAdded(sender, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnDocumentRemoved(object sender, EventArgs e)
        {
            if (DocumentRemoved != null)
            {
                DocumentRemoved(sender, e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// ドキュメントを保存する前に <see cref="DocumentSaving" /> イベントを発生します。
        /// </summary>
        private void OnDocumentSaving(object sender, DocumentEventArgs e)
        {
            DocumentSaving?.Invoke(sender, e);
        }

        ///--------------------------------
        /// <summary>
        /// ドキュメントが保存されると <see cref="DocumentSaved" /> イベントを発生します。
        /// </summary>
        private void OnDocumentSaved(object sender, DocumentEventArgs e)
        {
            DocumentSaved?.Invoke(sender, e);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Document CreateDocumentByTypeName(string typeName)
        {
            Document document = null;
            IDocumentFactory factory = null;

            factory = _DocumentServiceTraits.GetDocumentFactory(typeName);
            if (factory != null)
            {
                document = factory.CreateDocument();
            }

            return document;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Document FindDocumentByResource(IStreamResource resource)
        {
            Document document = null;

            DocumentReference[] docRefArray = _DocumentManager.Find(resource.Key);
            if (docRefArray.Length > 0)
            {
                document = docRefArray[0].Document;
            }

            return document;
        }
    }
}
