﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Core
{
    using System;
    using System.IO;
    using System.Linq;
    using System.Text.RegularExpressions;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// ファイルパスの検証を行います。
    /// </summary>
    public static class FilePathValidator
    {
        /// <summary>
        /// ファイルパスを検証します。
        /// </summary>
        /// <param name="filePath">ファイルパス。</param>
        /// <returns>検証結果。</returns>
        public static ValidationResult ValidateFilePath(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            if (filePath.IndexOfAny(Path.GetInvalidPathChars()) >= 0)
            {
                return new ValidationResult(false, MessageResource.Message_IncludeInvalidFilePathChars);
            }

            ValidationResult result = ValidateFileName(Path.GetFileName(filePath));
            if (!result.IsValid) { return result; }


            if (filePath.ToCharArray().Where(c => c == ':').Count() > 1)
            {
                return new ValidationResult(false, MessageResource.Message_InvalidFilePath);
            }

            if (new Regex("^[a-zA-Z]:\\\\.*").IsMatch(filePath) == false)
            {
                return new ValidationResult(false, MessageResource.Message_InvalidFilePath);
            }

            return ValidationResult.NoError;
        }

        /// <summary>
        /// ファイル名を検証します。
        /// </summary>
        /// <param name="fileName">ファイル名。</param>
        /// <returns>検証結果。</returns>
        public static ValidationResult ValidateFileName(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }

            if (fileName.IndexOfAny(Path.GetInvalidFileNameChars()) >= 0)
            {
                return new ValidationResult(false, MessageResource.Message_IncludeInvalidFileNameChars);
            }

            return ValidationResult.NoError;
        }

        /// <summary>
        /// 指定相対パスがカレントなのかを検証します。
        /// </summary>
        /// <param name="filePath">パス名。</param>
        /// <returns>検証結果。</returns>
        public static bool ValidateRelativePathIsCurrent(string filePath)
        {
            // テスト用パス("C:\\test")を使いフルパスを作成して、それが元のテスト用パスと
            // 同じになるかによって、指定パスがカレントを意味しているのかを判断しています。
            // (例) "poi/poi/../../" = カレント
            //      "poi/poi/.." カレントではない

            string testFilePath = Path.GetFullPath("C:\\test");
            filePath = Path.Combine(Path.Combine(testFilePath, filePath), ".\\");
            filePath = Path.GetDirectoryName(filePath.GetFullPath());
            return testFilePath == filePath ? true : false;
        }
    }
}
