﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;

namespace NintendoWare.SoundFoundation.Core
{
    /// <summary>
    /// ツリーオブジェクトコレクションの基本機能を実装します。
    /// </summary>
    public abstract class TreeObjectCollectionBase<TItem> : ITreeObjectCollection
        where TItem : class, ITreeObject
    {
        private bool _eventLock = false;
#if false
        private bool _eventDirty = false;
#endif
        private TItem _Owner = null;
        private CollectionChangedEventCompressor _collectionChangedCompressor = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="owner">コレクションの所有者。</param>
        public TreeObjectCollectionBase(TItem owner)
        {
            _Owner = owner;
        }

        /// <summary>
        /// コレクションが変化すると発生します。
        /// </summary>
        public event NotifyCollectionChangedEventHandler CollectionChanged;

        /// <summary>
        /// コレクションの所有者を取得します。
        /// </summary>
        public TItem Owner
        {
            get { return _Owner; }
        }

        /// <summary>
        /// 読み取り専用かどうかを調べます。
        /// </summary>
        public bool IsReadOnly
        {
            get { return false; }
        }

        /// <summary>
        /// アイテムの数を取得します。
        /// </summary>
        public int Count
        {
            get { return InnerCollection.Count; }
        }

        /// <summary>
        /// 内部コレクションを取得します。
        /// </summary>
        protected abstract IList<TItem> InnerCollection { get; }

        /// <summary>
        /// 指定インデックスのアイテムを取得します。
        /// </summary>
        /// <param name="index">アイテムのインデックス。</param>
        /// <returns>アイテム。</returns>
        public TItem this[int index]
        {
            get { return InnerCollection[index]; }
        }

        /// <summary>
        /// 指定インデックスのアイテムを取得します。
        /// </summary>
        /// <param name="index">アイテムのインデックス。</param>
        /// <returns>アイテム。</returns>
        ITreeObject ITreeObjectCollection.this[int index]
        {
            get { return this[index]; }
        }

        /// <summary>
        /// 指定アイテムがコレクションに含まれているかどうかを調べます。
        /// </summary>
        /// <param name="item">アイテム。</param>
        /// <returns>含まれている場合は true、含まれていない場合は false。</returns>
        public bool Contains(TItem item)
        {
            return InnerCollection.Contains(item);
        }

        /// <summary>
        /// 指定アイテムがコレクションに含まれているかどうかを調べます。
        /// </summary>
        /// <param name="item">アイテム。</param>
        /// <returns>含まれている場合は true、含まれていない場合は false。</returns>
        bool ICollection<ITreeObject>.Contains(ITreeObject item)
        {
            return Contains(item as TItem);
        }

        /// <summary>
        /// 指定アイテムのインデックスを取得します。
        /// </summary>
        /// <param name="item">アイテム。</param>
        /// <returns>アイテムのインデックス。</returns>
        public int IndexOf(TItem item)
        {
            return InnerCollection.IndexOf(item);
        }

        /// <summary>
        /// 指定アイテムのインデックスを取得します。
        /// </summary>
        /// <param name="item">アイテム。</param>
        /// <returns>アイテムのインデックス。</returns>
        int ITreeObjectCollection.IndexOf(ITreeObject item)
        {
            return IndexOf(item as TItem);
        }

        /// <summary>
        /// アイテムを追加します。
        /// </summary>
        /// <param name="item">アイテム。</param>
        public void Add(TItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            int index = InnerCollection.Count;
            InnerCollection.Add(item);

            OnCollectionChanged(new NotifyCollectionChangedEventArgs
                                 (NotifyCollectionChangedAction.Add, item, index));
        }

        /// <summary>
        /// アイテムを追加します。
        /// </summary>
        /// <param name="item">アイテム。</param>
        void ICollection<ITreeObject>.Add(ITreeObject item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            Add(item as TItem);
        }

        /// <summary>
        /// アイテムを挿入します。
        /// </summary>
        /// <param name="index">挿入先のインデックス。</param>
        /// <param name="item">挿入するアイテム。</param>
        public void Insert(int index, TItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            InnerCollection.Insert(index, item);

            OnCollectionChanged(new NotifyCollectionChangedEventArgs
                                 (NotifyCollectionChangedAction.Add, item, index));
        }

        /// <summary>
        /// アイテムを挿入します。
        /// </summary>
        /// <param name="index">挿入先のインデックス。</param>
        /// <param name="item">挿入するアイテム。</param>
        void ITreeObjectCollection.Insert(int index, ITreeObject item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            Insert(index, item as TItem);
        }

        /// <summary>
        /// アイテムを挿入します。
        /// </summary>
        /// <param name="index">挿入先のアイテム。</param>
        /// <param name="item">挿入するアイテム。</param>
        public void Insert(TItem targetItem, TItem item)
        {
            //if ( null == targetItem ) { throw new ArgumentNullException( "targetItem" ); }
            if (null == item) { throw new ArgumentNullException("item"); }

            if (targetItem == null)
            {
                Add(item);
                return;
            }

            if (this != targetItem.Parent.Children)
            {
                throw new ArgumentException("targetItem must not have a other parent.");
            }

            Insert(InnerCollection.IndexOf(targetItem), item);
        }

        /// <summary>
        /// アイテムを挿入します。
        /// </summary>
        /// <param name="index">挿入先のアイテム。</param>
        /// <param name="item">挿入するアイテム。</param>
        public void Insert(ITreeObject targetItem, ITreeObject item)
        {
            if (null == targetItem) { throw new ArgumentNullException("targetItem"); }
            if (null == item) { throw new ArgumentNullException("item"); }
            Insert(targetItem as TItem, item as TItem);
        }

        /// <summary>
        /// アイテムを削除します。
        /// </summary>
        /// <param name="item">アイテム。</param>
        public bool Remove(TItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            int index = InnerCollection.IndexOf(item);
            if (-1 == index) { return false; }

            RemoveAt(index);
            return true;
        }

        /// <summary>
        /// アイテムを削除します。
        /// </summary>
        /// <param name="item">アイテム。</param>
        bool ICollection<ITreeObject>.Remove(ITreeObject item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            return Remove(item as TItem);
        }

        /// <summary>
        /// 指定インデックスのアイテムを削除します。
        /// </summary>
        /// <param name="index">削除するアイテムのインデックス。</param>
        public void RemoveAt(int index)
        {
            ITreeObject item = this[index];

            InnerCollection.RemoveAt(index);

            OnCollectionChanged(new NotifyCollectionChangedEventArgs
                                 (NotifyCollectionChangedAction.Remove, item, index));
        }

        /// <summary>
        /// アイテムをクリアします。
        /// </summary>
        public void Clear()
        {
            InnerCollection.Clear();
            OnCollectionChanged(new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Reset));
        }

        /// <summary>
        /// 指定インデックスからアイテムをコピーします。
        /// </summary>
        /// <param name="array"></param>
        /// <param name="arrayIndex"></param>
        public void CopyTo(TItem[] array, int arrayIndex)
        {
            InnerCollection.CopyTo(array, arrayIndex);
        }

        /// <summary>
        /// 指定インデックスからアイテムをコピーします。
        /// </summary>
        /// <param name="array"></param>
        /// <param name="arrayIndex"></param>
        void ICollection<ITreeObject>.CopyTo(ITreeObject[] array, int arrayIndex)
        {
            for (int i = 0; i + arrayIndex < Count && i < array.Length; i++)
            {
                array[i] = this[i + arrayIndex];
            }
        }

        /// <summary>
        /// アイテムの列挙子を取得します。
        /// </summary>
        /// <returns>アイテムの列挙子</returns>
        public IEnumerator<TItem> GetEnumerator()
        {
            return InnerCollection.GetEnumerator();
        }

        /// <summary>
        /// アイテムの列挙子を取得します。
        /// </summary>
        /// <returns>アイテムの列挙子</returns>
        IEnumerator<ITreeObject> IEnumerable<ITreeObject>.GetEnumerator()
        {
            return InnerCollection.Cast<ITreeObject>().GetEnumerator();
        }

        /// <summary>
        /// アイテムの列挙子を取得します。
        /// </summary>
        /// <returns>アイテムの列挙子</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

#if false
        /// 2013.4.26 aoyagi
        /// Componentの移動を Remove、Addによって実現していたものを
        /// Moveだけで行なうようにした時にこれらが不要になりました。
        /// SuspendEvent()して、ResumeEvent()すると CollectionChangedの
        /// 引数が、Action.Resetでは使う用途がありませんでした。

        /// <summary>
        /// イベントの発行を抑止します。
        /// </summary>
        public void SuspendEvent()
        {
            _eventLock = true;
        }

        /// <summary>
        /// イベントの発行を再開します。
        /// </summary>
        public void ResumeEvent()
        {
            ResumeEvent( true );
        }

        /// <summary>
        /// イベントの発行を再開します。
        /// </summary>
        /// <param name="raiseEvent">true の場合、イベント抑止中に発行要求があった場合は、Resetイベントを発行します。</param>
        public void ResumeEvent( bool raiseEvent )
        {
            if ( !_eventLock ) { return; }

            _eventLock = false;

            if ( raiseEvent && _eventDirty ) {
                OnCollectionChanged(
                    new NotifyCollectionChangedEventArgs( NotifyCollectionChangedAction.Reset ) );
            }

            _eventDirty = false;
        }
#endif

        public void BeginCompressEvent()
        {
            if (_collectionChangedCompressor == null)
            {
                _collectionChangedCompressor = new CollectionChangedEventCompressor();
                _collectionChangedCompressor.Begin();
            }
        }

        public void EndCompressEvent()
        {
            if (_collectionChangedCompressor != null)
            {
                var args = _collectionChangedCompressor.End();
                if (args != null)
                {
                    CollectionChanged?.Invoke(this, args);
                }

                _collectionChangedCompressor = null;
            }
        }

        /// <summary>
        /// コレクションが変更されると発生します。
        /// </summary>
        /// <param name="e">コレクション変更イベントデータ。</param>
        protected void OnCollectionChanged(NotifyCollectionChangedEventArgs e)
        {
            if (_eventLock)
            {
#if false
                _eventDirty = true;
#endif
                return;
            }

            if (_collectionChangedCompressor != null)
            {
                foreach (var args in _collectionChangedCompressor.Push(e))
                {
                    CollectionChanged?.Invoke(this, args);
                }
            }
            else
            {
                CollectionChanged?.Invoke(this, e);
            }
        }
    }
}
