﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;

namespace NintendoWare.SoundFoundation.Core
{
    /// <summary>
    /// ルーティングイベントとイベントハンドラを関連付けます。
    /// </summary>
    public class RoutingEventBinding
    {
        private Type _eventType;
        private RoutingEventHandler _bubblingEventHandler;
        private RoutingEventHandler _eventHandler;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="eventType">ルーティングイベントの種類。</param>
        /// <param name="bubblingEventHandler">バブリングイベントハンドラ。</param>
        /// <param name="eventHandler">イベントハンドラ。</param>
        public RoutingEventBinding(Type eventType, RoutingEventHandler bubblingEventHandler,
                                    RoutingEventHandler eventHandler)
        {
            if (null == eventType) { throw new ArgumentNullException("eventType"); }

            if (!(eventType.IsSubclassOf(typeof(RoutingEvent))))
            {
                throw new ArgumentException("invalid routing event type");
            }

            _eventType = eventType;
            _bubblingEventHandler = bubblingEventHandler;
            _eventHandler = eventHandler;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="eventType">ルーティングイベントの種類。</param>
        protected RoutingEventBinding(Type eventType)
        {
            if (null == eventType) { throw new ArgumentNullException("eventType"); }
            _eventType = eventType;
        }

        /// <summary>
        /// ルーティングイベントの種類を取得します。
        /// </summary>
        public Type EventType
        {
            get { return _eventType; }
        }

        /// <summary>
        /// バブリングキャストイベントを受け取ると実行されます。
        /// </summary>
        /// <param name="sender">イベントの発行元。</param>
        /// <param name="e">ルーティングイベントデータ。</param>
        public virtual void InvokeBubblingEvent(object sender, RoutingEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }
            if (null != _bubblingEventHandler)
            {
                _bubblingEventHandler(sender, e);
            }
        }

        /// <summary>
        /// イベントを受け取ると実行されます。
        /// </summary>
        /// <param name="sender">イベントの発行元。</param>
        /// <param name="e">ルーティングイベントデータ。</param>
        public virtual void InvokeEvent(object sender, RoutingEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }
            if (null != _eventHandler)
            {
                _eventHandler(sender, e);
            }
        }

        /// <summary>
        /// 指定 RoutingEventBinding とこの RoutingEventBinding が等しいかどうかを調べます。
        /// </summary>
        /// <param name="obj">比較対象の RoutingEventBinding。</param>
        /// <returns>等しい場合は true、等しくない場合は false。</returns>
        public override bool Equals(object obj)
        {
            RoutingEventBinding target = obj as RoutingEventBinding;
            if (null == target) { return false; }

            if (_eventType != target._eventType) { return false; }
            if (_bubblingEventHandler != target._bubblingEventHandler) { return false; }
            if (_eventHandler != target._eventHandler) { return false; }

            return true;
        }

        /// <summary>
        /// ハッシュコードを取得します。
        /// </summary>
        /// <returns>ハッシュコード。</returns>
        public override int GetHashCode()
        {
            return _eventType.GetHashCode() ^
                   _bubblingEventHandler.GetHashCode() ^
                   _eventHandler.GetHashCode();
        }
    }

    /// <summary>
    /// ルーティングイベントとイベントハンドラを関連付けます。
    /// </summary>
    /// <typeparam name="TEventArgs">イベントデータの型。</typeparam>
    public class RoutingEventBinding<TEventArgs> : RoutingEventBinding
        where TEventArgs : RoutingEventArgs
    {
        private SpecificEventHandler _bubblingEventHandler;
        private SpecificEventHandler _eventHandler;

        /// <summary>
        /// ルーティングイベントを受け取ると実行されます。
        /// </summary>
        /// <param name="sender">イベントの発行元。</param>
        /// <param name="e">ルーティングイベントデータ。</param>
        public delegate void SpecificEventHandler(object sender, TEventArgs e);

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="eventType">ルーティングイベントの型。</param>
        /// <param name="bubblingEventHandler">バブリングイベントハンドラ。</param>
        /// <param name="eventHandler">イベントハンドラ。</param>
        public RoutingEventBinding(Type eventType, SpecificEventHandler bubblingEventHandler,
                                    SpecificEventHandler eventHandler)
            : base(eventType)
        {
            _bubblingEventHandler = bubblingEventHandler;
            _eventHandler = eventHandler;
        }

        /// <summary>
        /// バブリングキャストイベントを受け取ると実行されます。
        /// </summary>
        /// <param name="sender">イベントの発行元。</param>
        /// <param name="e">ルーティングイベントデータ。</param>
        public override void InvokeBubblingEvent(object sender, RoutingEventArgs e)
        {
            if (!(e is TEventArgs)) { throw new ArgumentException("invalid event args type."); }
            if (null != _bubblingEventHandler)
            {
                _bubblingEventHandler(sender, e as TEventArgs);
            }
        }

        /// <summary>
        /// イベントを受け取ると実行されます。
        /// </summary>
        /// <param name="sender">イベントの発行元。</param>
        /// <param name="e">ルーティングイベントデータ。</param>
        public override void InvokeEvent(object sender, RoutingEventArgs e)
        {
            if (!(e is TEventArgs)) { throw new ArgumentException("invalid event args type."); }
            if (null != _eventHandler)
            {
                _eventHandler(sender, e as TEventArgs);
            }
        }
    }
}
