﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;

namespace NintendoWare.SoundFoundation.Core
{
    /// <summary>
    /// ルーティングイベントの種類を示す基本クラス。
    /// </summary>
    public abstract class RoutingEvent
    {
        /// <summary>
        /// イベントデータの種類を取得します。
        /// </summary>
        public abstract Type EventArgsType { get; }
    }

    /// <summary>
    /// ルーティングイベントを受け取ると実行されます。
    /// </summary>
    /// <param name="sender">イベントの発行元。</param>
    /// <param name="e">ルーティングイベントデータ。</param>
    public delegate void RoutingEventHandler(object sender, RoutingEventArgs e);

    /// <summary>
    /// ルーティングイベントのデータを格納します。
    /// </summary>
    public class RoutingEventArgs : CancelEventArgs
    {
        private RoutingEvent _event;
        private IEventRoutable _destination;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="e">ルーティングイベントの型。</param>
        public RoutingEventArgs(RoutingEvent e) : this(e, null) { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="e">ルーティングイベントの型。</param>
        /// <param name="destination">ルーティングイベントの宛先。</param>
        public RoutingEventArgs(RoutingEvent e, IEventRoutable destination)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (e.EventArgsType != GetType())
            {
                throw new ArgumentException("invalid event args type.");
            }

            _event = e;
            _destination = destination;
        }

        /// <summary>
        /// ルーティングイベントの型を取得します。
        /// </summary>
        public RoutingEvent Event
        {
            get { return _event; }
        }

        /// <summary>
        /// ルーティングイベントの宛先を取得します。
        /// </summary>
        public IEventRoutable Destination
        {
            get { return _destination; }
        }
    }
}
