﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Reflection;

namespace NintendoWare.SoundFoundation.Core
{
    /// <summary>
    /// Enum に関連する拡張機能を提供します。
    /// </summary>
    public static class EnumEx
    {
        /// <summary>
        /// Enum の拡張メソッド。
        /// Enum フィールドのTextAttribute 属性から、フィールドに関連付けられたテキストを取得します。
        /// </summary>
        /// <param name="obj">Enum。</param>
        /// <returns>Enum フィールドに関連付けられたテキスト。</returns>
        public static string GetText(this Enum obj)
        {
            return GetText(obj.GetType().GetField(Enum.GetName(obj.GetType(), obj)));
        }

        /// <summary>
        /// 指定テキストに対応する Enum 値を取得します。
        /// 対応する Enum 値が見つからなかった場合は、指定デフォルト値を返します。
        /// </summary>
        /// <typeparam name="EnumType">Enum 型。</typeparam>
        /// <param name="text">テキスト。</param>
        /// <param name="defaultValue">デフォルト値。</param>
        /// <returns>対魚する Enum 値が見つかった場合は、その値、見つからなかった場合は、指定デフォルト値。</returns>
        public static Enum ParseText<EnumType>(string text, Enum defaultValue)
        {
            try
            {
                return ParseText<EnumType>(text);
            }
            catch (ArgumentException)
            {
                return defaultValue;
            }
        }

        /// <summary>
        /// 指定テキストに対応する Enum 値を取得します。
        /// </summary>
        /// <typeparam name="EnumType">Enum 型。</typeparam>
        /// <param name="text">テキスト。</param>
        /// <returns>対魚する Enum 値が見つかった場合は、その値。</returns>
        public static Enum ParseText<EnumType>(string text)
        {
            if (null == text) { throw new ArgumentNullException("text"); }

            foreach (Enum value in Enum.GetValues(typeof(EnumType)))
            {
                if (GetText(value) == text) { return value; }
            }

            throw new ArgumentException("enum field not found.");
        }

        /// <summary>
        /// 指定フィールドに関連付けられたテキストを取得します。
        /// </summary>
        /// <param name="field">フィールド。</param>
        /// <returns>フィールドに関連付けられたテキスト。</returns>
        private static string GetText(FieldInfo field)
        {
            object[] attributes = field.GetCustomAttributes(typeof(TextAttribute), false);
            if (0 == attributes.Length) { return string.Empty; }

            return (attributes[0] as TextAttribute).Text;
        }
    }
}
