﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;

namespace NintendoWare.SoundFoundation.Core.Parameters
{
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class XmlParameterSerializerBase
    {
        private ParameterValueFactoryDictionary _Factorys = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public XmlParameterSerializerBase()
        {
            ElementNodeName = "Parameters";
            ElementLeafName = "Parameter";

            _Factorys = new ParameterValueFactoryDictionary();

            Factorys.Add("Text", new ParameterValueFactory<TextParameterValue>());
            Factorys.Add("FilePath", new ParameterValueFactory<FilePathParameterValue>());
            Factorys.Add("Integer", new ParameterValueFactory<IntParameterValue>());
            Factorys.Add("UInteger", new ParameterValueFactory<UIntParameterValue>());
            Factorys.Add("ULong", new ParameterValueFactory<ULongParameterValue>());
            Factorys.Add("Float", new ParameterValueFactory<FloatParameterValue>());
            Factorys.Add("Double", new ParameterValueFactory<DoubleParameterValue>());
            Factorys.Add("Boolean", new ParameterValueFactory<BoolParameterValue>());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ParameterValueFactoryDictionary Factorys
        {
            get { return _Factorys; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string ElementNodeName { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string ElementLeafName { get; set; }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class XmlParameterSerializer : XmlParameterSerializerBase
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public XmlElement Execute(XmlDocument doc, IParameterDictionary dictionary, Func<string, bool> filter)
        {
            XmlElement xmlCardinalElement = null;
            XmlElement xmlElement = null;
            IParameterValue value = null;

            xmlCardinalElement = doc.CreateElement(ElementNodeName);

            foreach (string key in dictionary.Keys)
            {
                value = dictionary[key];

                if (filter != null)
                {
                    if (filter(key) == false)
                    {
                        continue;
                    }
                }

                xmlElement = CreateParameterElement(doc, key, value);
                xmlCardinalElement.AppendChild(xmlElement);
            }

            return xmlCardinalElement;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public XmlElement Execute(XmlDocument doc, IParameterDictionary dictionary, string[] keys)
        {
            XmlElement xmlCardinalElement = null;
            XmlElement xmlElement = null;
            IParameterValue value = null;

            xmlCardinalElement = doc.CreateElement(ElementNodeName);

            foreach (string key in keys)
            {
                value = dictionary[key];
                xmlElement = CreateParameterElement(doc, key, value);
                xmlCardinalElement.AppendChild(xmlElement);
            }

            return xmlCardinalElement;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public XmlElement CreateParameterElement(XmlDocument doc, string key, IParameterValue value)
        {
            XmlElement xmlElement = null;
            string type = GetParameterType(value);

            if (type == String.Empty)
            {
                type = "Text";
            }

            string valueText = value != null ? value.ToString() : String.Empty;

            xmlElement = doc.CreateElement(ElementLeafName);
            xmlElement.SetAttribute("Name", key);
            xmlElement.SetAttribute("Type", type);
            xmlElement.SetAttribute("Value", valueText);
            return xmlElement;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string GetParameterType(IParameterValue value)
        {
            if (value != null)
            {
                Type targetType = value.GetType();

                foreach (KeyValuePair<string, IParameterValueFactory> pd in Factorys)
                {
                    if (targetType == pd.Value.Type)
                    {
                        return pd.Key;
                    }
                }
            }

            return String.Empty;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ParameterValueFactoryDictionary : Dictionary<string, IParameterValueFactory>
    {
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IParameterValueFactory
    {
        Type Type { get; }
        IParameterValue Create();
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ParameterValueFactory<T> : IParameterValueFactory where T : IParameterValue, new()
    {
        public Type Type { get { return typeof(T); } }
        public IParameterValue Create() { return new T(); }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IParameterFilter
    {
        bool Filter(string type);
    }
}
