﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Core.Parameters
{
    using System.Collections.Generic;
    using System.Text;
    using ToolDevelopmentKit;

    /// <summary>
    /// string 型のパラメータ値を格納します。
    /// </summary>
    public class TextParameterValue : ParameterValue<string>
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public TextParameterValue()
            : base(string.Empty)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータの値を指定します。</param>
        public TextParameterValue(string value)
            : base(value)
        {
        }

        /// <summary>
        /// パラメータ値を取得または設定します。
        /// </summary>
        public override string Value
        {
            get { return base.Value; }
            set
            {
                Ensure.Argument.NotNull(value);
                base.Value = value;
            }
        }

        /// <summary>
        /// パラメータの値を取得するキャストオペレータです。
        /// </summary>
        /// <param name="value">対象となるパラメータ値を指定します。</param>
        /// <returns>パラメータの値を返します。</returns>
        public static implicit operator string(TextParameterValue value)
        {
            Ensure.Argument.NotNull(value);
            return value.Value;
        }

        /// <summary>
        /// テキストからパラメータ値を取得します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        protected override string ParseInternal(string text)
        {
            Ensure.Argument.NotNull(text);
            return text;
        }

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        protected override ValidationResult ValidateInternal(string value)
        {
            return ValidationResult.NoError;
        }

        /// <summary>
        /// パラメータ値のバイト列を取得します。
        /// </summary>
        /// <returns>バイト列の列挙子を返します。</returns>
        protected override IEnumerable<byte> GetBytes()
        {
            return Encoding.Unicode.GetBytes(this.Value);
        }
    }
}
