﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Core.Parameters
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using ToolDevelopmentKit;
    using NintendoWare.SoundFoundation.Projects;

    /// <summary>
    /// キーとパラメータ値のコレクションを管理します。
    /// </summary>
    public class ParameterDictionary : IParameterDictionary
    {
        private static readonly string ParameterKeySeparator = ProjectParameterNames.ParameterKeySeparator;

        private static readonly IParameterValue NullParameterValue = new TextParameterValue(string.Empty);

        private Dictionary<string, IParameterValue> dictionary = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ParameterDictionary()
        {
            this.dictionary = new Dictionary<string, IParameterValue>();
        }

        /// <summary>
        /// パラメータ値が変更されると発生します。
        /// </summary>
        public event ParameterEventHandler ParameterValueChanged;

        /// <summary>
        /// 指定キーを持つパラメータ値を取得または設定します。
        /// </summary>
        /// <param name="key">キー。</param>
        /// <returns>パラメータ値。</returns>
        public IParameterValue this[string key]
        {
            get { return GetValue(key); }
        }

        /// <summary>
        /// パラメータディクショナリに含まれる全てのキーを取得します。
        /// </summary>
        public string[] Keys
        {
            get
            {
                ICollection<string> keys = this.dictionary.Keys;
                return keys.ToArray();
            }
        }

        /// <summary>
        /// 指定キーを持つパラメータが存在するか確認します。
        /// </summary>
        /// <remarks>
        /// キーにセパレータが含まれない場合は、自身が保持するパラメータを確認します。
        /// キーにセパレータが含まれる場合は、階層を辿ってパラメータを確認します。
        /// </remarks>
        /// <param name="key">キー。</param>
        /// <returns>存在する場合は true、存在しない場合は false。</returns>
        public bool ContainsKey(string key)
        {
            Ensure.Argument.NotNull(key);

            var value = this.GetValue(key);
            return value != null && value != NullParameterValue;
        }

        /// <summary>
        /// 指定キーを持つパラメータ値を取得します。
        /// </summary>
        /// <remarks>
        /// キーにセパレータが含まれない場合は、自身が保持するパラメータになります。
        /// キーにセパレータが含まれる場合は、階層を辿ってパラメータを取得します。
        /// </remarks>
        /// <param name="key">キー。</param>
        /// <returns>パラメータ値。</returns>
        public IParameterValue GetValue(string key)
        {
            Ensure.Argument.NotNull(key);

            var keys = key.Split(new[] { ParameterKeySeparator }, StringSplitOptions.None);

            if (keys.Length == 0 ||
                !this.dictionary.ContainsKey(keys[0]))
            {
                // HACK : null を返すように修正したい
                return NullParameterValue;
            }

            var value = this.dictionary[keys[0]];

            // 階層を含むキーでない場合は、自分が管理するパラメータから検索します。
            // 階層を含むキーの場合は、下層のパラメータプロバイダに処理を委譲します。
            if (keys.Length == 1)
            {
                Assertion.Operation.ObjectNotNull(value);
                return value;
            }

            var parameterProvider = value as IParameterProvider;

            if (parameterProvider == null)
            {
                // HACK : null を返すように修正したい
                return NullParameterValue;
            }

            // 下層のキーを取り出して処理を委譲します。
            return parameterProvider.Parameters.GetValue(
                key.Substring(keys[0].Length + 1));
        }

        /// <summary>
        /// 指定キーを持つパラメータ値を追加します。
        /// </summary>
        /// <param name="key">キー。</param>
        /// <param name="value">パラメータ値。</param>
        public void AddValue(string key, IParameterValue value)
        {
            Ensure.Argument.NotNull(key);
            Ensure.Argument.NotNull(value);

            this.dictionary.Add(key, value);
            value.ValueChanged += OnValueChanged;

            OnParameterValueChanged(this, new ParameterEventArgs(key, value));
        }

        /// <summary>
        /// 指定キーを持つパラメータ値を削除します。
        /// </summary>
        /// <param name="key">キー。</param>
        public void RemoveValue(string key)
        {
            Ensure.Argument.NotNull(key);

            IParameterValue value = null;
            this.dictionary.TryGetValue(key, out value);

            if (value == null) { return; }

            value.ValueChanged -= OnValueChanged;
            this.dictionary.Remove(key);
        }

        /// <summary>
        /// すべてのパラメータ値を削除します。
        /// </summary>
        public void Clear()
        {
            try
            {
                foreach (IParameterValue value in this.dictionary.Values)
                {
                    value.ValueChanged -= OnValueChanged;
                }
            }
            finally
            {
                this.dictionary.Clear();
            }
        }

        /// <summary>
        /// パラメータ辞書のハッシュコードを取得します。
        /// </summary>
        /// <param name="algorithm">ハッシュアルゴリズムを指定します。</param>
        /// <param name="key">パラメータキーを指定します。</param>
        /// <param name="filter">対象パラメータのフィルタを指定します。</param>
        /// <returns>ハッシュコードを返します。</returns>
        public HashCode GetParameterDictionaryHashCode(
            HashAlgorithm algorithm,
            string key,
            Func<IParameterValue, bool> filter)
        {
            Ensure.Argument.NotNull(algorithm);
            Ensure.Argument.NotNull(filter);

            HashCode valueHash = HashCode.Empty;

            foreach (var childKey in this.Keys)
            {
                IParameterValue value = this[childKey];

                if (filter != null && !filter(value))
                {
                    continue;
                }

                var childHash = value.GetParameterHashCode(algorithm, childKey, filter);

                if (valueHash == HashCode.Empty)
                {
                    valueHash = childHash;
                }
                else
                {
                    if (childHash != HashCode.Empty)
                    {
                        valueHash ^= childHash;
                    }
                }
            }

            if (valueHash == HashCode.Empty)
            {
                return valueHash;
            }

            // キーのバイト列と値のバイト列を連結してハッシュ計算します。
            var bytes = Encoding.Unicode.GetBytes(key).Concat(valueHash.Value);
            return new HashCode(algorithm.ComputeHash(bytes.ToArray()));
        }

        ///--------------------------------
        /// <summary>
        /// </summary>
        private string GetKeyByValue(IParameterValue value)
        {
            var keys = this.dictionary
                .Where(kv => kv.Value == value)
                .Select(kv => kv.Key);
            return keys.FirstOrDefault();
        }

        ///--------------------------------
        /// <summary>
        /// </summary>
        private void OnValueChanged(object sender, ParameterEventArgs e)
        {
            IParameterValue value = sender as IParameterValue;
            string key = this.GetKeyByValue(value); // 値からキーを取得します。例："Volume" や "Sends" 等

            // "MainSend" 等のサブキーがある場合
            if (e.Key != string.Empty)
            {
                // キーとサブキーを結合します。
                // 例：キー "Sends" サブキー "MainSend" → "Sends/MainSend"
                key += ParameterKeySeparator + e.Key;
            }

            OnParameterValueChanged(this, new ParameterEventArgs(key, value));
        }

        ///--------------------------------
        /// <summary>
        /// </summary>
        private void OnParameterValueChanged(object sender, ParameterEventArgs e)
        {
            if (ParameterValueChanged != null)
            {
                ParameterValueChanged(sender, e);
            }
        }
    }
}
