﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Xml;
    using System.Xml.Schema;
    using System.Xml.Serialization;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using ToolDevelopmentKit;

    /// <summary>
    /// 依存関係管理クラス
    /// </summary>
    public partial class DependencyManager : IXmlSerializable
    {
        #region ** 固定値

        // 要素名
        private const string RootElementName = "NintendoWareSoundDependency";
        private const string ElementName = "DependencyManager";
        private const string OutputItemsElementName = "OutputItems";
        private const string UserKeyMapsElementName = "UserKeyMaps";
        private const string UserKeyMapElementName = "UserKeyMap";
        private const string UserKeyMapItemElementName = "UserKeyMapItem";
        private const string UserParametersElementName = "UserParameters";
        private const string UserParameterElementName = "UserParameter";

        // 属性名
        private const string VersionAttributeName = "Version";
        private const string BaseDirectoryPathAttributeName = "BaseDirectoryPath";
        private const string IDAttributeName = "ID";
        private const string KeyAttributeName = "Key";

        // バージョン
        private static readonly Version Version = new Version(2, 0);

        #endregion

        #region ** フィールド

        // コレクション
        private DependentOutputCollection _outputs = new DependentOutputCollection();       // 出力コレクション
        private UserParameterDictionary _userParameters = new UserParameterDictionary();    // ユーザパラメータコレクション

        private string _managementFilePath = string.Empty;  // 管理ファイルのパス
        private string _baseDirectoryPath = string.Empty;   // ベースディレクトリパス

        #endregion

        #region ** プロパティ

        /// <summary>
        /// 管理ファイルのパスを取得または設定します。
        /// </summary>
        public string ManagementFilePath
        {
            get { return _managementFilePath; }
            set { _managementFilePath = value; }
        }

        /// <summary>
        /// 出力ファイルのベースディレクトリパスを取得または設定します。
        /// </summary>
        public string BaseDirectoryPath
        {
            get { return _baseDirectoryPath; }
            set { _baseDirectoryPath = value; }
        }

        /// <summary>
        /// 出力ファイルのベースディレクトリパスを取得または設定します。
        /// </summary>
        public string BaseDirectoryAbsolutePath
        {
            get
            {
                if (null == _baseDirectoryPath) { return null; }

                if (Path.IsPathRooted(_baseDirectoryPath)) { return _baseDirectoryPath; }

                if (null == _managementFilePath) { return null; }
                if (0 == _managementFilePath.Length) { return string.Empty; }

                return Path.Combine(Path.GetDirectoryName(_managementFilePath), _baseDirectoryPath);
            }
        }

        /// <summary>
        /// 出力コレクションを取得します。
        /// </summary>
        public IDependentOutputCollection Outputs
        {
            get { return _outputs; }
        }

        public UserParameterDictionary UserParameters
        {
            get { return _userParameters; }
        }

        #endregion

        #region ** メソッド

        //-----------------------------------------------------------------
        // ファイル入力
        //-----------------------------------------------------------------

        public static DependencyManager FromFile(string filePath)
        {
            DependencyManager dependencyManager = new DependencyManager();
            dependencyManager.Load(filePath);

            return dependencyManager;
        }

        public void Load(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            try
            {
                _managementFilePath = filePath;

                using (XmlReader reader = XmlTextReader.Create(filePath))
                {
                    ReadXml(reader);
                }
            }
            catch (FileNotFoundException exception)
            {
                _managementFilePath = string.Empty;
                _outputs.Clear();
                throw exception;
            }
            catch (Exception exception)
            {
                _managementFilePath = string.Empty;
                _outputs.Clear();
                Debug.WriteLine(exception.ToString());
                throw exception;
            }
        }

        public void Save()
        {
            if (null == _managementFilePath || 0 == _managementFilePath.Length)
            {
                throw new Exception("invalid management file path.");
            }

            Save(_managementFilePath);
        }

        public void Save(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            this.Validate();

            // ディレクトリを作成する
            string directoryPath = Path.GetDirectoryName(filePath);
            if (null != directoryPath && 0 < directoryPath.Length)
            {
                Directory.CreateDirectory(directoryPath);
            }

            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Encoding = Encoding.UTF8;
            settings.Indent = true;

            using (XmlWriter writer = XmlTextWriter.Create(filePath, settings))
            {
                WriteXml(writer);
            }
        }

        //-----------------------------------------------------------------
        // XML入出力
        //-----------------------------------------------------------------

        /// <summary>
        /// このプロパティは予約されています。代わりに System.Xml.Serialization.XmlSchemaProviderAttribute をクラスに適用します。
        /// </summary>
        /// <returns>XML 表現を記述する System.Xml.Schema.XmlSchema。</returns>
        public XmlSchema GetSchema() { return null; }

        /// <summary>
        /// オブジェクトの XML 表現からオブジェクトを生成します。
        /// </summary>
        /// <param name="reader">オブジェクトの逆シリアル化元である System.Xml.XmlReader ストリーム。</param>
        public void ReadXml(XmlReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            _outputs.Clear();


            XmlDocument document = new XmlDocument();
            document.Load(reader);

            if (RootElementName != document.DocumentElement.Name)
            {
                throw new DependencyManagerInvalidFileFormatException();
            }

            if (!document.DocumentElement.HasAttribute(VersionAttributeName))
            {
                throw new DependencyManagerInvalidFileVersionException();
            }

            // バージョンチェック
            string[] versions = document.DocumentElement.GetAttribute(VersionAttributeName).Split(new char[] { '.' });
            if (2 != versions.Length ||
                Version.Major != int.Parse(versions[0]) || Version.Minor != int.Parse(versions[1]))
            {
                throw new DependencyManagerInvalidFileVersionException();
            }

            // DependencyManager
            XmlElement dependencyManagerElement = document.DocumentElement[ElementName];
            if (null == dependencyManagerElement)
            {
                throw new DependencyManagerInvalidFileFormatException();
            }

            ReadDependencyManagerElement(dependencyManagerElement);
        }

        /// <summary>
        /// オブジェクトを XML 表現に変換します。
        /// </summary>
        /// <param name="writer">オブジェクトのシリアル化先の System.Xml.XmlWriter ストリーム。</param>
        public void WriteXml(XmlWriter writer)
        {
            Ensure.Argument.NotNull(writer);

            XmlDocument document = new XmlDocument();
            document.AppendChild(ToXml(document));

            document.Save(writer);
        }

        //-----------------------------------------------------------------
        // 出力の追加と削除
        //-----------------------------------------------------------------

        /// <summary>
        /// 新しい出力を追加します。
        /// </summary>
        /// <param name="item">追加する出力。</param>
        public void Add(IDependentOutput output)
        {
            Ensure.Argument.NotNull(output);
            AddInternal(output);
        }

        /// <summary>
        /// 出力を削除します。
        /// </summary>
        /// <param name="output">削除する出力。</param>
        public void Remove(IDependentOutput output)
        {
            Ensure.Argument.NotNull(output);

            if (!_outputs.Contains(output)) { return; }

            _outputs.Remove(output);
        }

        /// <summary>
        /// 既存の出力にユーザキーを設定します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">関連付けるユーザキー。</param>
        /// <param name="output">既存の出力。</param>
        public void SetUserKey(string userKeyMapID, string userKey, IDependentOutput output)
        {
            if (null == userKeyMapID) { throw new ArgumentNullException("userKeyMapID"); }
            if (0 == userKeyMapID.Length) { throw new ArgumentException("userKeyMapID"); }
            if (null == userKey) { throw new ArgumentNullException("userKey"); }
            if (0 == userKey.Length) { throw new ArgumentException("userKey"); }
            if (null == output) { throw new ArgumentNullException("output"); }

            if (!_outputs.Contains(output.Key))
            {
                throw new DependencyOutputNotFound(output.Key);
            }

            SetUserKeyInternal(userKeyMapID, userKey, output);
        }

        /// <summary>
        /// ユーザキーを削除し出力の参照カウントをデクリメントします。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">ユーザキー。</param>
        public void RemoveByUserKey(string userKeyMapID, string userKey)
        {
            if (null == userKeyMapID) { throw new ArgumentNullException("userKeyMapID"); }
            if (0 == userKeyMapID.Length) { throw new ArgumentException("userKeyMapID"); }
            if (null == userKey) { throw new ArgumentNullException("userKey"); }
            if (0 == userKey.Length) { throw new ArgumentException("userKey"); }

            RemoveByUserKeyInternal(userKeyMapID, userKey);
        }

        /// <summary>
        /// 出力ファイルを全て削除します。
        /// </summary>
        public void Clean()
        {
            // 出力ファイルを削除する
            foreach (DependentOutput output in _outputs.Values)
            {
                output.Clean();
            }

            _outputs.Clear();

            // DependencyManager管理ファイルを削除する
            try
            {
                File.Delete(_managementFilePath);
            }
            catch (Exception)
            {
            }
        }

        /// <summary>
        /// 出力のインスタンスを作成します。
        /// </summary>
        /// <returns>新しい出力のインスタンス</returns>
        public IDependentOutput CreateOutput()
        {
            return new DependentOutput(this, Guid.NewGuid().ToString());
        }

        /// <summary>
        /// アイテムのインスタンスを作成します。
        /// </summary>
        /// <remarks>
        /// 名前の自動補正は、同じ出力が複数のOutputItemに分かれる等
        /// 致し方ない場合にのみ無効化することを推奨します。
        /// </remarks>
        /// <param name="name">アイテムの名前</param>
        /// <param name="isAutoNameCorrection">名前の自動補正の有無を指定します。</param>
        /// <returns>新しいアイテムのインスタンス</returns>
        public IDependentOutputItem CreateOutputItem(string key, string name, bool isAutoNameCorrection)
        {
            return new DependentOutputItem(
                key,
                isAutoNameCorrection ? CreateOutputItemName(name) : name);
        }

        /// <summary>
        /// アイテム名を作成します。
        /// </summary>
        /// <param name="baseName">アイテムのベースネーム</param>
        /// <returns>新しいアイテム名</returns>
        public string CreateOutputItemName(string baseName)
        {
            if (null == baseName) { throw new ArgumentNullException("baseName"); }
            if (0 == baseName.Length) { throw new ArgumentException("baseName"); }

            string directory = Path.GetDirectoryName(baseName);
            string workName = baseName;
            int count = 0;

            while (_outputs.ContainsOutputName(workName))
            {
                count++;

                workName = Path.Combine(
                    directory,
                    string.Format("{0}_{1}{2}",
                        Path.GetFileNameWithoutExtension(baseName),
                        count.ToString(),
                        Path.GetExtension(baseName)
                        ));
            }

            return workName;
        }

        public IDependedFileInfo CreateDependedFileInfo(string filePath)
        {
            return new DependedFileInfo(filePath);
        }

        public IDependedFileInfo CreateDependedFileInfo(string filePath, HashCode hashCode)
        {
            return new DependedFileInfo(filePath, hashCode);
        }

        public IDependedFileInfo CreateDependedOutputInfo(IDependentOutput outputItem)
        {
            return CreateDependedOutputItemInfo(outputItem.Key);
        }

        public IDependedFileInfo CreateDependedOutputItemInfo(string outputItemKey)
        {
            return new DependedOutputItemInfo(outputItemKey);
        }

        //-----------------------------------------------------------------
        // 出力の追加と削除
        //-----------------------------------------------------------------

        /// <summary>
        /// 新しい出力を追加します。
        /// </summary>
        /// <param name="output">追加する出力。</param>
        private void AddInternal(IDependentOutput output)
        {
            Assertion.Argument.NotNull(output);
            _outputs.Add(output);
        }

        /// <summary>
        /// 既存の出力にユーザキーを設定します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">関連付けるユーザキー。</param>
        /// <param name="outputItem">既存の出力。</param>
        private void SetUserKeyInternal(string userKeyMapID, string userKey, IDependentOutput output)
        {
            Assertion.Argument.NotNull(userKeyMapID);
            Assertion.Argument.NotNull(userKey);
            Assertion.Argument.NotNull(output);
            _outputs.SetUserKey(userKeyMapID, userKey, output);
        }

        /// <summary>
        /// ユーザキーを削除し出力の参照カウントをデクリメントします。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">ユーザキー。</param>
        private void RemoveByUserKeyInternal(string userKeyMapID, string userKey)
        {
            Assertion.Argument.NotNull(userKeyMapID);
            Assertion.Argument.NotNull(userKey);
            _outputs.RemoveByUserKey(userKeyMapID, userKey);
        }

        //-----------------------------------------------------------------
        // 出力の正規化
        //-----------------------------------------------------------------

        private void Validate()
        {
            // 依存出力アイテムが出力一覧に存在しない場合は、依存を削除します。
            foreach (IDependentOutput output in this.Outputs.Values)
            {
                foreach (DependedOutputItemInfo dependedOutputItemInfo
                    in output.Dependencies.OfType<DependedOutputItemInfo>().ToArray())
                {
                    if (!this.Outputs.ContainsKey(dependedOutputItemInfo.Key))
                    {
                        output.Dependencies.Remove(dependedOutputItemInfo);
                    }
                }
            }
        }

        //-----------------------------------------------------------------
        // XML 入出力
        //-----------------------------------------------------------------

        private void ReadDependencyManagerElement(XmlElement dependencyManagerElement)
        {
            Assertion.Argument.NotNull(dependencyManagerElement);

            // BaseDirectoryPath
            if (dependencyManagerElement.HasAttribute(BaseDirectoryPathAttributeName))
            {
                _baseDirectoryPath = dependencyManagerElement.GetAttribute(BaseDirectoryPathAttributeName);
            }

            // OutputItems
            XmlElement outputItemsElement = dependencyManagerElement[OutputItemsElementName];
            if (null == outputItemsElement)
            {
                throw new DependencyManagerInvalidFileFormatException();
            }

            ReadOutputItemsElement(outputItemsElement);


            // UserKeyMaps
            XmlElement userKeyMapsElement = dependencyManagerElement[UserKeyMapsElementName];
            if (null != userKeyMapsElement)
            {
                ReadUserKeyMapsElement(userKeyMapsElement);
            }


            // UserParameters
            XmlElement userParametersElement = dependencyManagerElement[UserParametersElementName];
            if (null != userParametersElement)
            {
                ReadUserParametersElement(userParametersElement);
            }
        }

        private void ReadOutputItemsElement(XmlElement outputItemsElement)
        {
            Assertion.Argument.NotNull(outputItemsElement);

            foreach (XmlElement outputItemElement in outputItemsElement.GetElementsByTagName(DependentOutput.ElementName))
            {
                DependentOutput newOutputItem = DependentOutput.FromXml(this, outputItemElement);
                if (null == newOutputItem) { continue; }
                AddInternal(newOutputItem);
            }

        }

        private void ReadUserKeyMapsElement(XmlElement userKeyMapsElement)
        {
            Assertion.Argument.NotNull(userKeyMapsElement);

            // UserKeyMap
            foreach (XmlElement userKeyMapElement in userKeyMapsElement.GetElementsByTagName(UserKeyMapElementName))
            {
                ReadUserKeyMapElement(userKeyMapElement);
            }
        }

        private void ReadUserKeyMapElement(XmlElement userKeyMapElement)
        {
            Assertion.Argument.NotNull(userKeyMapElement);
            if (!userKeyMapElement.HasAttribute(IDAttributeName)) { return; }

            string userKeyMapID = userKeyMapElement.GetAttribute(IDAttributeName);

            // UserKeyMapItem
            foreach (XmlElement element in userKeyMapElement.GetElementsByTagName(UserKeyMapItemElementName))
            {
                ReadUserKeyMapItemElement(element, userKeyMapID);
            }
        }

        private void ReadUserKeyMapItemElement(XmlElement userKeyMapItemElement, string userKeyMapID)
        {
            Assertion.Argument.NotNull(userKeyMapItemElement);
            if (!userKeyMapItemElement.HasAttribute(KeyAttributeName)) { return; }

            try
            {

                string userKey = userKeyMapItemElement.GetAttribute(KeyAttributeName);
                DependentOutput outputItem = _outputs[userKeyMapItemElement.InnerText] as DependentOutput;

                SetUserKeyInternal(userKeyMapID, userKey, outputItem);

            }
            catch { }
        }

        private void ReadUserParametersElement(XmlElement userParametersElement)
        {
            Assertion.Argument.NotNull(userParametersElement);

            // UserParameter
            foreach (XmlElement element in userParametersElement.GetElementsByTagName(UserParameterElementName))
            {
                ReadUserParameterElement(element);
            }
        }

        private void ReadUserParameterElement(XmlElement userParametersElement)
        {
            Assertion.Argument.NotNull(userParametersElement);

            if (!userParametersElement.HasAttribute(KeyAttributeName)) { return; }

            string key = userParametersElement.GetAttribute(KeyAttributeName);
            if (_userParameters.ContainsKey(key)) { return; }

            _userParameters.Add(key, userParametersElement.InnerText);
        }

        /// <summary>
        /// DependencyManagerをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement ToXml(XmlDocument document)
        {
            // ルート要素
            XmlElement rootElement = document.CreateElement(RootElementName);
            rootElement.SetAttribute(VersionAttributeName, string.Format("{0}.{1}", Version.Major, Version.Minor));

            // DependencyManager 要素
            XmlElement dependencyManagerElement = document.CreateElement(ElementName);
            rootElement.AppendChild(dependencyManagerElement);

            // BaseDirectoryPath 属性
            if (0 < _baseDirectoryPath.Length)
            {
                dependencyManagerElement.SetAttribute(BaseDirectoryPathAttributeName, _baseDirectoryPath);
            }

            // OutputItems 要素
            XmlElement outputItemsElement = OutputItemsToXml(document);
            if (null != outputItemsElement)
            {
                dependencyManagerElement.AppendChild(outputItemsElement);
            }

            // UserKeyMaps 要素
            XmlElement userKeyMapsElement = UserKeyMapsToXml(document);
            if (null != userKeyMapsElement)
            {
                dependencyManagerElement.AppendChild(userKeyMapsElement);
            }

            // UserParameters 要素
            XmlElement userParametersElement = UserParametersToXml(document);
            if (null != userParametersElement)
            {
                dependencyManagerElement.AppendChild(userParametersElement);
            }

            return rootElement;
        }

        /// <summary>
        /// 出力コレクションをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement OutputItemsToXml(XmlDocument document)
        {
            Assertion.Argument.NotNull(document);

            if (0 == _outputs.Values.Count) { return null; }

            XmlElement outputItemsElement = document.CreateElement(OutputItemsElementName);

            foreach (DependentOutput outputItem in _outputs.Values)
            {

                XmlElement outputItemElement = outputItem.ToXml(document);
                if (null == outputItemElement) { continue; }

                outputItemsElement.AppendChild(outputItemElement);

            }

            return outputItemsElement;
        }

        /// <summary>
        /// 全てのユーザキーマップをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement UserKeyMapsToXml(XmlDocument document)
        {
            Assertion.Argument.NotNull(document);

            _outputs.CollectGarbage();

            if (0 == _outputs.UserKeyMapIDs.Count) { return null; }

            // UserKeyMaps
            XmlElement userKeyMapsElement = document.CreateElement(UserKeyMapsElementName);

            foreach (string userKeyMapID in _outputs.UserKeyMapIDs)
            {

                // UserKeyMap
                XmlElement userKeyMapElement = UserKeyMapToXml(document, userKeyMapID);
                if (null == userKeyMapElement) { return null; }

                userKeyMapsElement.AppendChild(userKeyMapElement);

            }

            return (0 < userKeyMapsElement.ChildNodes.Count) ? userKeyMapsElement : null;
        }

        /// <summary>
        /// ユーザキーマップをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement UserKeyMapToXml(XmlDocument document, string userKeyMapID)
        {
            Assertion.Argument.NotNull(document);
            Assertion.Argument.NotNull(userKeyMapID);

            if (0 == _outputs.GetUserKeys(userKeyMapID).Count) { return null; }

            XmlElement userKeyMapElement = document.CreateElement(UserKeyMapElementName);

            userKeyMapElement.SetAttribute(IDAttributeName, userKeyMapID);

            foreach (string userKey in _outputs.GetUserKeys(userKeyMapID))
            {

                XmlElement userKeyMapItemElement = UserKeyMapItemToXml(document, userKeyMapID, userKey);
                if (null == userKeyMapItemElement) { continue; }

                userKeyMapElement.AppendChild(userKeyMapItemElement);

            }

            return userKeyMapElement;
        }

        /// <summary>
        /// ユーザキーをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <param name="key">キー</param>
        /// <returns>XML要素</returns>
        private XmlElement UserKeyMapItemToXml(XmlDocument document, string userKeyMapID, string userKey)
        {
            Assertion.Argument.NotNull(document);
            Assertion.Argument.NotNull(userKey);
            Assertion.Argument.True(0 < userKey.Length);

            if (!_outputs.ContainsUserKey(userKeyMapID, userKey)) { return null; }

            XmlElement userKeyMapItemElement = document.CreateElement(UserKeyMapItemElementName);

            // UserKey
            userKeyMapItemElement.SetAttribute(KeyAttributeName, userKey);

            // Key
            userKeyMapItemElement.InnerText = _outputs[userKeyMapID, userKey].Key;

            return userKeyMapItemElement;
        }

        /// <summary>
        /// ユーザパラメータをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement UserParametersToXml(XmlDocument document)
        {
            Assertion.Argument.NotNull(document);

            if (0 == _userParameters.Count) { return null; }

            XmlElement userParametersElement = document.CreateElement(UserParametersElementName);

            // UserParameter
            foreach (string key in _userParameters.Keys)
            {
                XmlElement userParameterElement = document.CreateElement(UserParameterElementName);

                userParameterElement.SetAttribute(KeyAttributeName, key);   // Key
                userParameterElement.InnerText = _userParameters[key];      // Value

                userParametersElement.AppendChild(userParameterElement);
            }

            return userParametersElement;
        }

        #endregion

        #region ** クラス

        //-----------------------------------------------------------------
        // 依存情報クラス
        //-----------------------------------------------------------------

        /// <summary>
        /// 依存ファイル情報 基本クラス
        /// </summary>
        public abstract class DependedFileInfoBase : IDependedFileInfo
        {
            #region ** 固定値

            // 属性名
            private const string FileSizeAttributeName = "FileSize";
            private const string LastUpdatedTimeAttributeName = "LastUpdatedTime";

            #endregion

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public DependedFileInfoBase()
            {
                this.CurrentHashCode = HashCode.Empty;
                this.LastUpdatedHashCode = HashCode.Empty;
            }

            /// <summary>
            /// DependencyManagerを取得します。
            /// </summary>
            public DependencyManager Owner { get; set; }

            /// <summary>
            /// キーを取得します。
            /// </summary>
            public abstract string Key { get; }

            /// <summary>
            /// 依存ファイルのファイルパスを取得します。
            /// </summary>
            public abstract string FilePath { get; }

            /// <summary>
            /// 依存ファイルの絶対ファイルパスを取得します。
            /// </summary>
            public abstract string AbsoluteFilePath { get; }

            /// <summary>
            /// 最終更新時間を取得します。
            /// </summary>
            public DateTime LastUpdatedTime
            {
                get
                {
                    if (!File.Exists(AbsoluteFilePath))
                    {
                        return DateTime.MinValue;
                    }

                    return File.GetLastWriteTime(AbsoluteFilePath);
                }
            }

            /// <summary>
            /// 最終更新時のハッシュコードを取得または設定します。
            /// </summary>
            public HashCode LastUpdatedHashCode { get; set; }

            /// <summary>
            /// 現在のハッシュコードを取得または設定します。
            /// </summary>
            public HashCode CurrentHashCode { get; set; }

            #region ** XML入出力

            /// <summary>
            /// 依存ファイル情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            XmlElement IDependedFileInfo.ToXml(XmlDocument document)
            {
                return ToXml(document);
            }

            /// <summary>
            /// 依存ファイル情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            protected abstract XmlElement ToXml(XmlDocument document);

            #endregion
        }

        /// <summary>
        /// 依存ファイル情報クラス
        /// </summary>
        public class DependedFileInfo : DependedFileInfoBase
        {
            #region ** 固定値

            // 要素名
            public const string ElementName = "DependencyFileInfo";

            // 属性名
            private const string FilePathAttributeName = "FilePath";
            private const string HashCodeAttributeName = "HashCode";

            #endregion

            #region ** フィールド

            private string _filePath = string.Empty;
            private Lazy<string> fullPath;

            #endregion

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="filePath">依存ファイルのパス</param>
            public DependedFileInfo(string filePath) :
                this(filePath, HashCode.Empty)
            {
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="filePath">依存ファイルのパス</param>
            /// <param name="currentHashCode">現在のハッシュコードを指定します。</param>
            public DependedFileInfo(string filePath, HashCode currentHashCode)
            {
                if (null == filePath) { throw new ArgumentNullException("filePath"); }
                if (0 == filePath.Length) { throw new ArgumentException("filePath"); }

                _filePath = filePath;
                this.CurrentHashCode = currentHashCode;
                this.LastUpdatedHashCode = HashCode.Empty;

                // Path.GetFullPath() を毎回評価するとパフォーマンスに影響が出るので、キャッシュしています。
                this.fullPath = new Lazy<string>(() =>
                {
                    // 絶対パスの場合はそのまま返す
                    if (Path.IsPathRooted(FilePath)) { return FilePath; }
                    if (null == this.Owner) { return string.Empty; }
                    return Path.Combine(Owner.BaseDirectoryAbsolutePath, FilePath).GetFullPath();
                });
            }

            #region ** プロパティ

            /// <summary>
            /// キーを取得します。
            /// </summary>
            public override string Key
            {
                get { return this.FilePath; }
            }

            /// <summary>
            /// 依存ファイルのパスを取得します。
            /// </summary>
            public override string FilePath
            {
                get { return _filePath; }
            }

            /// <summary>
            /// 依存ファイルの絶対パスを取得します。
            /// </summary>
            public override string AbsoluteFilePath => this.fullPath.Value;

            #endregion

            #region ** XML入出力

            /// <summary>
            /// XML要素からDependencyFileを生成します。
            /// </summary>
            /// <param name="xmlElement">入力XML要素</param>
            /// <returns>生成したDependencyFile。</returns>
            public static DependedFileInfo FromXml(XmlElement xmlElement)
            {
                if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

                if (ElementName != xmlElement.Name) { throw new DependencyManagerInvalidFileFormatException(); }
                if (!xmlElement.HasAttribute(FilePathAttributeName)) { throw new DependencyManagerInvalidFileFormatException(); }

                DependedFileInfo newInstance = new DependedFileInfo(xmlElement.GetAttribute(FilePathAttributeName));

                // FilePath
                newInstance._filePath = xmlElement.GetAttribute(FilePathAttributeName);

                // HashCode
                newInstance.LastUpdatedHashCode = HashCode.Parse(xmlElement.GetAttribute(HashCodeAttributeName));

                return newInstance;
            }

            /// <summary>
            /// 依存ファイル情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            protected override XmlElement ToXml(XmlDocument document)
            {
                Assertion.Argument.NotNull(document);

                // DependencyFileInfo
                XmlElement dependencyFileElement = document.CreateElement(ElementName);
                dependencyFileElement.SetAttribute(FilePathAttributeName, FilePath);
                dependencyFileElement.SetAttribute(HashCodeAttributeName, CurrentHashCode.ToString());

                return dependencyFileElement;
            }

            #endregion
        }

        /// <summary>
        /// 依存出力情報クラス
        /// </summary>
        public class DependedOutputItemInfo : DependedFileInfoBase
        {
            #region ** 固定値

            // 要素名
            public const string ElementName = "DependencyOutputItemInfo";

            // 属性名
            private const string OutputItemKeyAttributeName = "OutputItemKey";

            #endregion

            #region ** フィールド

            private string _outputItemKey = string.Empty;

            #endregion

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="fileName">依存する出力情報の名前</param>
            public DependedOutputItemInfo(string outputItemKey)
            {
                if (null == outputItemKey) { throw new ArgumentNullException("outputKey"); }
                if (0 == outputItemKey.Length) { throw new ArgumentException("outputKey"); }

                _outputItemKey = outputItemKey;
            }

            #region ** プロパティ

            /// <summary>
            /// キーを取得します。
            /// </summary>
            public override string Key
            {
                get { return this._outputItemKey; }
            }

            /// <summary>
            /// 依存する出力ファイルのパスを取得します。
            /// </summary>
            public override string FilePath
            {
                get
                {
                    if (null == Output) { return string.Empty; }
                    if (0 == Output.OutputItems.Count) { return string.Empty; }
                    return Output.OutputItems[0].FilePath;
                }
            }

            /// <summary>
            /// 依存する出力ファイルの絶対パスを取得します。
            /// </summary>
            public override string AbsoluteFilePath
            {
                get
                {
                    if (null == Output) { return string.Empty; }
                    if (0 == Output.OutputItems.Count) { return string.Empty; }
                    return Output.OutputItems[0].AbsoluteFilePath;
                }
            }

            /// <summary>
            /// 依存する出力情報を取得します。
            /// </summary>
            public IDependentOutput Output
            {
                get
                {
                    if (this.Owner == null) { return null; }
                    if (!this.Owner.Outputs.ContainsKey(_outputItemKey)) { return null; }
                    return this.Owner.Outputs[_outputItemKey];
                }
            }

            #endregion

            #region ** XML入出力

            /// <summary>
            /// XML要素からDependencyOutputItemInfoを生成します。
            /// </summary>
            /// <param name="xmlElement">入力XML要素</param>
            /// <returns>生成したDependencyOutputItemInfo。</returns>
            public static DependedOutputItemInfo FromXml(XmlElement xmlElement)
            {
                if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

                if (ElementName != xmlElement.Name) { throw new DependencyManagerInvalidFileFormatException(); }
                if (!xmlElement.HasAttribute(OutputItemKeyAttributeName)) { throw new DependencyManagerInvalidFileFormatException(); }

                DependedOutputItemInfo newInstance = new DependedOutputItemInfo(xmlElement.GetAttribute(OutputItemKeyAttributeName));

                // OutputItemKey
                newInstance._outputItemKey = xmlElement.GetAttribute(OutputItemKeyAttributeName);

                return newInstance;
            }

            /// <summary>
            /// 依存出力ファイル情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            protected override XmlElement ToXml(XmlDocument document)
            {
                Assertion.Argument.NotNull(document);

                // DependencyFileInfo
                XmlElement dependencyOutputItemInfoElement = document.CreateElement(ElementName);
                dependencyOutputItemInfoElement.SetAttribute(OutputItemKeyAttributeName, _outputItemKey.ToString());

                return dependencyOutputItemInfoElement;
            }

            #endregion
        }

        #endregion

        #region ** イベントデリゲートとイベントパラメータ

        public delegate void OutputItemEventHandler(object sender, OutputItemEventArgs e);

        public class OutputItemEventArgs : EventArgs
        {
            private IDependentOutputItem outputItem = null;

            public OutputItemEventArgs(IDependentOutputItem outputItem)
            {
                this.outputItem = outputItem;
            }

            public IDependentOutputItem OutputItem
            {
                get { return outputItem; }
            }
        }

        #endregion

        #region ** コレクション

        public class UserParameterDictionary : Dictionary<string, string> { }

        #endregion

        #region ** NoCaseStringCompare

        private class NoCaseStringCompare : IEqualityComparer<string>
        {
            private static NoCaseStringCompare InstanceInternal = new NoCaseStringCompare();

            private NoCaseStringCompare() { }

            public static NoCaseStringCompare Instance
            {
                get { return NoCaseStringCompare.InstanceInternal; }
            }

            /// <summary>
            /// 指定した string が等しいかどうかを判断します。
            /// </summary>
            /// <param name="x">string1。</param>
            /// <param name="y">string2。</param>
            /// <returns>指定した string が等しい場合は true。それ以外の場合は false。</returns>
            public bool Equals(string x, string y)
            {
                if (null == x) { throw new ArgumentNullException("x"); }
                if (null == y) { throw new ArgumentNullException("y"); }

                return string.Compare(x, y, true) == 0;
            }

            /// <summary>
            /// 指定した string のハッシュ コードを返します。
            /// </summary>
            /// <param name="text">string。</param>
            /// <returns>ハッシュコード</returns>
            public int GetHashCode(string text)
            {
                if (null == text) { throw new ArgumentNullException("text"); }
                return text.ToLower().GetHashCode();
            }
        }

        #endregion
    }

    #region ** インターフェイス

    /// <summary>
    /// 出力コレクション インターフェイス
    /// </summary>
    public interface IDependentOutputCollection
    {
        /// <summary>
        /// キーコレクションを取得します。
        /// </summary>
        Dictionary<string, Dictionary<string, IDependentOutput>>.KeyCollection UserKeyMapIDs { get; }

        /// <summary>
        /// 出力コレクションを取得します。
        /// </summary>
        ReadOnlyCollection<IDependentOutput> Values { get; }

        #region ** インデクサ

        /// <summary>
        /// 指定したキーを持つ出力を取得します。
        /// </summary>
        /// <param name="key">取得する出力のキー。</param>
        /// <returns>指定したキーを持つ出力。</returns>
        IDependentOutput this[string key] { get; }

        /// <summary>
        /// 指定したユーザキーを持つ出力を取得します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">取得する出力のユーザキー。</param>
        /// <returns>指定したキーを持つ出力。</returns>
        IDependentOutput this[string userKeyMapID, string userKey] { get; }

        #endregion

        #region ** メソッド

        /// <summary>
        /// ユーザキーコレクションを取得します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        Dictionary<string, IDependentOutput>.KeyCollection GetUserKeys(string userKeyMapID);

        /// <summary>
        /// 指定したキーを持つ出力が含まれているかどうかを確認します。
        /// </summary>
        /// <param name="key">取得する出力のキー。</param>
        /// <returns>指定した名前を持つ出力を含む場合は true。それ以外の場合は false。</returns>
        bool ContainsKey(string key);

        /// <summary>
        /// 指定したユーザキーマップの出力が含まれているかどうかを確認します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <returns>指定したキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
        bool ContainsUserKeyMap(string userKeyMapID);

        /// <summary>
        /// 指定したユーザキーの出力が含まれているかどうかを確認します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">検索されるユーザキー。</param>
        /// <returns>指定したキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
        bool ContainsUserKey(string userKeyMapID, string userKey);

        #endregion
    }

    /// <summary>
    /// 出力情報コレクション インターフェイス
    /// </summary>
    public interface IDependentOutputItemCollection : ICollection<IDependentOutputItem>, IEnumerable<IDependentOutputItem>, IEnumerable
    {
        bool Contains(string key);

        /// <summary>
        /// 指定したインデックスを持つ出力情報を取得します。
        /// </summary>
        /// <param name="index">取得する出力情報のインデックス。</param>
        /// <returns>指定したインデックスを持つ出力情報。</returns>
        IDependentOutputItem this[int index] { get; }

        /// <summary>
        /// 指定したキーを持つ出力情報を取得します。
        /// </summary>
        /// <param name="key">取得する出力情報のキー。</param>
        /// <returns>指定したキーを持つ出力情報。</returns>
        IDependentOutputItem this[string key] { get; }
    }

    /// <summary>
    /// 依存情報コレクション インターフェイス
    /// </summary>
    public interface IDependedFileInfoCollection : ICollection<IDependedFileInfo>, IEnumerable<IDependedFileInfo>, IEnumerable
    {
        /// <summary>
        /// 指定したインデックスを持つ出力情報を取得します。
        /// </summary>
        /// <param name="index">取得する出力情報のインデックス。</param>
        /// <returns>指定したインデックスを持つ出力情報。</returns>
        IDependedFileInfo this[int index] { get; }

        /// <summary>
        /// 指定したキーを持つ出力情報を取得します。
        /// </summary>
        /// <param name="key">取得する出力情報に関連付けられたキー。</param>
        /// <returns>指定したキーを持つ出力情報。</returns>
        IDependedFileInfo this[string key] { get; }

        /// <summary>
        /// 指定したキーに関連付けられたアイテムがコレクションに含まれているかどうかを調べます。
        /// </summary>
        /// <param name="key">キーを指定します。</param>
        /// <returns>
        /// 指定したキーに関連付けられたアイテムがコレクションに含まれている場合は true、
        /// 含まれていない場合は false を返します。
        /// </returns>
        bool Contains(string key);

        /// <summary>
        /// 指定したキーに関連付けられたアイテムを削除します。
        /// </summary>
        /// <param name="key">キーを指定します。</param>
        /// <returns>削除に成功した場合は true、失敗した場合、指定キーが見つからない場合には false を返します。</returns>
        bool Remove(string key);
    }

    #endregion

    #region ** 例外クラス

    public class DependencyManagerInvalidFileFormatException : Exception
    {
        public override string Message
        {
            get { return "the dependency manager file is invalid format."; }
        }
    }

    public class DependencyManagerInvalidFileVersionException : Exception
    {
        public override string Message
        {
            get { return "the dependency manager file version is not supported."; }
        }
    }

    public class DependencyFileNotFoundException : FileNotFoundException
    {
        public DependencyFileNotFoundException(string fileName) : base("dependency file not found.", fileName) { }
    }

    public class DependencyOutputNotFound : Exception
    {
        private string _outputKey = string.Empty;

        public DependencyOutputNotFound(string outputKey)
        {
            _outputKey = (null == outputKey) ? string.Empty : outputKey;
        }

        public override string Message
        {
            get { return string.Format("output \"{0}\" not found.", _outputKey); }
        }
    }

    public class DependencyManagerInvalidOutputItem : Exception
    {
        private string _name = string.Empty;

        public DependencyManagerInvalidOutputItem(string name)
        {
            _name = (null == name) ? string.Empty : name;
        }

        public override string Message
        {
            get { return string.Format("invalid output info \"{0}\".", _name); }
        }
    }

    public class DependencyManagerInvalidDependencyInfo : Exception
    {
        private string _filePath = string.Empty;

        public DependencyManagerInvalidDependencyInfo(string filePath)
        {
            _filePath = (null == filePath) ? string.Empty : filePath;
        }

        public override string Message
        {
            get { return string.Format("invalid dependency info \"{0}\".", _filePath); }
        }
    }

    #endregion
}
